/*
 * -------------------------------------------------------------------------
 *
 *  pn4webScroll.js Copyright (C) 2011, 
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *   Author: Patrick Rammelt 
 *   E-Mail: patrick.rammelt@go4more.de
 *   Site:   http://www.patricks-seite.de
 * -------------------------------------------------------------------------
 *
 * Custom scroll-bars.
 */

/**
 * Scroll object (container for all global constants and parameters) 
 */
var SCROLL = {};


// INITIALIZATION ===================================================

/**
 * Initialization.
 * @param subLevels  number of sub-directories to the root directory
 */
function initScroll (subLevel)
{
  // attributes needed for slider dragging
  SCROLL.DRAG = {};
  SCROLL.DRAG.drag    = dummyEventHandler; // drag function
  SCROLL.DRAG.release = dummyEventHandler; // release function
  SCROLL.DRAG.x0      = 0;                 // start x-coordinate of dragging
  SCROLL.DRAG.y0      = 0;                 // start y-coordinate of dragging
  SCROLL.DRAG.xOffset = -1;                // offset grab position to left border
  SCROLL.DRAG.yOffset = -1;                // offset grab position to top border

  // get base path
  SCROLL.BASE_PATH = "";
  for (var i = 0; i < subLevel; i++) SCROLL.BASE_PATH += "../";

  // scrollbar for popup windows of bayesian networks / junction trees
  SCROLL.PN4W = {};
  SCROLL.PN4W.HSCROLL                = {};
  SCROLL.PN4W.HSCROLL.CLAZZ          = 'pn4w_hscroll';
  SCROLL.PN4W.HSCROLL.YOFFSET        = 16;
  SCROLL.PN4W.HSCROLL.SLIDER         = {};
  SCROLL.PN4W.HSCROLL.SLIDER.MINSIZE = 4;
  SCROLL.PN4W.HSCROLL.SLIDER.CLAZZ   = 'pn4w_hscroll_slider';
  SCROLL.PN4W.HSCROLL.SLIDER.STRETCH = true;
  SCROLL.PN4W.HSCROLL.SLIDER.BORDER  = { E:2, W:2, N:0, S:0 };  // CSS3
  SCROLL.PN4W.HSCROLL.SLIDER.SPLIT   = { E:8, W:8, N:0, S:0 };  // CSS3
  SCROLL.PN4W.HSCROLL.SLIDER.BG      = "js/scroll/hscroll.png"; // CSS3
  SCROLL.PN4W.HSCROLL.SLIDER.BG_W    = "js/scroll/hscroll_w.png"; // Non-CSS3 (deprecated)
  SCROLL.PN4W.HSCROLL.SLIDER.BG_M    = "js/scroll/hscroll_m.png"; // Non-CSS3 (deprecated)
  SCROLL.PN4W.HSCROLL.SLIDER.BG_E    = "js/scroll/hscroll_e.png"; // Non-CSS3 (deprecated)
  SCROLL.PN4W.VSCROLL                = {};
  SCROLL.PN4W.VSCROLL.CLAZZ          = 'pn4w_vscroll';
  SCROLL.PN4W.VSCROLL.XOFFSET        = 15;
  SCROLL.PN4W.VSCROLL.SLIDER         = {};
  SCROLL.PN4W.VSCROLL.SLIDER.MINSIZE = 4;
  SCROLL.PN4W.VSCROLL.SLIDER.CLAZZ   = 'pn4w_vscroll_slider';
  SCROLL.PN4W.VSCROLL.SLIDER.STRETCH = true;
  SCROLL.PN4W.VSCROLL.SLIDER.BORDER  = { E:0, W:0, N:2, S:2 };  // CSS3
  SCROLL.PN4W.VSCROLL.SLIDER.SPLIT   = { E:0, W:0, N:8, S:8 };  // CSS3
  SCROLL.PN4W.VSCROLL.SLIDER.BG      = "js/scroll/vscroll.png"; // CSS3
  SCROLL.PN4W.VSCROLL.SLIDER.BG_N    = "js/scroll/vscroll_n.png"; // Non-CSS3 (deprecated)
  SCROLL.PN4W.VSCROLL.SLIDER.BG_M    = "js/scroll/vscroll_m.png"; // Non-CSS3 (deprecated)
  SCROLL.PN4W.VSCROLL.SLIDER.BG_S    = "js/scroll/vscroll_s.png"; // Non-CSS3 (deprecated)

  // scrollbar for blue frames
  SCROLL.BLUE = {};
  SCROLL.BLUE.HSCROLL                 = {};
  SCROLL.BLUE.HSCROLL.CLAZZ           = 'blue_hscroll';
  SCROLL.BLUE.HSCROLL.YOFFSET         = 0;
  SCROLL.BLUE.HSCROLL.SLIDER          = {};
  SCROLL.BLUE.HSCROLL.SLIDER.MINSIZE  = 17;
  SCROLL.BLUE.HSCROLL.SLIDER.CLAZZ    = 'blue_hscroll_slider';
  SCROLL.BLUE.HSCROLL.SLIDER.STRETCH  = true;
  SCROLL.BLUE.HSCROLL.SLIDER.BORDER   = { E:8, W:8, N:0, S:0 };         // CSS3
  SCROLL.BLUE.HSCROLL.SLIDER.SPLIT    = { E:32, W:32, N:0, S:0 };       // CSS3
  SCROLL.BLUE.HSCROLL.SLIDER.BG       = "js/scroll/blue_hscroll.png";   // CSS3
  SCROLL.BLUE.HSCROLL.SLIDER.SHADOW   = "js/scroll/shadow_hscroll.png"; // CSS3
  SCROLL.BLUE.HSCROLL.SLIDER.BG_W     = "js/scroll/blue_hscroll_w.png";   // Non-CSS3 (deprecated)
  SCROLL.BLUE.HSCROLL.SLIDER.BG_M     = "js/scroll/blue_hscroll_m.png";   // Non-CSS3 (deprecated)
  SCROLL.BLUE.HSCROLL.SLIDER.BG_E     = "js/scroll/blue_hscroll_e.png";   // Non-CSS3 (deprecated)
  SCROLL.BLUE.HSCROLL.SLIDER.SHADOW_W = "js/scroll/shadow_hscroll_w.png"; // Non-CSS3 (deprecated)
  SCROLL.BLUE.HSCROLL.SLIDER.SHADOW_M = "js/scroll/shadow_hscroll_m.png"; // Non-CSS3 (deprecated)
  SCROLL.BLUE.HSCROLL.SLIDER.SHADOW_E = "js/scroll/shadow_hscroll_e.png"; // Non-CSS3 (deprecated)
  SCROLL.BLUE.VSCROLL                 = {};
  SCROLL.BLUE.VSCROLL.CLAZZ           = 'blue_vscroll';
  SCROLL.BLUE.VSCROLL.XOFFSET         = 0;
  SCROLL.BLUE.VSCROLL.SLIDER          = {};
  SCROLL.BLUE.VSCROLL.SLIDER.MINSIZE  = 17;
  SCROLL.BLUE.VSCROLL.SLIDER.CLAZZ    = 'blue_vscroll_slider';
  SCROLL.BLUE.VSCROLL.SLIDER.STRETCH  = true;
  SCROLL.BLUE.VSCROLL.SLIDER.BORDER   = { E:0, W:0, N:8, S:8 };         // CSS3
  SCROLL.BLUE.VSCROLL.SLIDER.SPLIT    = { E:0, W:0, N:32, S:32 };       // CSS3
  SCROLL.BLUE.VSCROLL.SLIDER.BG       = "js/scroll/blue_vscroll.png";   // CSS3
  SCROLL.BLUE.VSCROLL.SLIDER.SHADOW   = "js/scroll/shadow_vscroll.png"; // CSS3
  SCROLL.BLUE.VSCROLL.SLIDER.BG_N     = "js/scroll/blue_vscroll_n.png";   // Non-CSS3 (deprecated)
  SCROLL.BLUE.VSCROLL.SLIDER.BG_M     = "js/scroll/blue_vscroll_m.png";   // Non-CSS3 (deprecated)
  SCROLL.BLUE.VSCROLL.SLIDER.BG_S     = "js/scroll/blue_vscroll_s.png";   // Non-CSS3 (deprecated)
  SCROLL.BLUE.VSCROLL.SLIDER.SHADOW_N = "js/scroll/shadow_vscroll_n.png"; // Non-CSS3 (deprecated)
  SCROLL.BLUE.VSCROLL.SLIDER.SHADOW_M = "js/scroll/shadow_vscroll_m.png"; // Non-CSS3 (deprecated)
  SCROLL.BLUE.VSCROLL.SLIDER.SHADOW_S = "js/scroll/shadow_vscroll_s.png"; // Non-CSS3 (deprecated)

  // add dragging functionality 
  var drag    = function (evnt) { SCROLL.DRAG.drag(evnt);    };
  var release = function (evnt) { SCROLL.DRAG.release(evnt); };
  addEventListener(document, "mousemove", drag);
  addEventListener(document, "mouseup",   release);
}

/**
 * Create a vertical scroll bar.
 * @param outerId    id of the element containing the scrollable 
 *                   element and the newly created scrollbar
 * @param innerId    id if the element defining the visible frame 
 *                   of the scrollable area
 * @param contentId  id of the scrollable element (partly visible)
 * @param type       type of scrollbar (e.g. SCROLL.PN4W)
 */
function vscroll (outerId, innerId, contentId, type)
{
  var outer   = document.getElementById(outerId);
  var inner   = document.getElementById(innerId);
  var content = document.getElementById(contentId);
  removeElement(innerId + '.vscroll.frame');

  // frame and background
  var scroll = appendNewElement(outer, 'div', innerId + '.vscroll.frame', type.VSCROLL.CLAZZ);
  var h = inner.offsetHeight;
  scroll.style.height = h;
  scroll.style.left   = inner.offsetLeft + inner.offsetWidth + type.VSCROLL.XOFFSET;
  scroll.style.top    = inner.offsetTop;
  if (type.VSCROLL.BG) {
    decorate(inner.id + '.vscroll', false, false, false, true, false, type.VSCROLL.BG);
  } 
  
  // slider
  var slider = appendNewElement(scroll, 'div', scroll.id + '.slider.frame', type.VSCROLL.SLIDER.CLAZZ);
  var o = type.HSCROLL.SLIDER.MINSIZE;
  var s = toInt((h-o) * Math.min(1, h/getHeight(content)) + o);
  var m = (content.style.marginTop ? parseInt(content.style.marginTop) : 0);
  var r = (content.offsetHeight <= inner.offsetHeight ? 0 : -m / (content.offsetHeight-inner.offsetHeight));
  slider.style.height = s;
  slider.style.left   = 0;
  slider.style.top    = toInt(Math.max(0.0, Math.min(1.0, r)) * (h - s));
  if (type.VSCROLL.SLIDER) {
    var shadow = (type.VSCROLL.SLIDER.SHADOW ? true : false);
    decorate(scroll.id + '.slider', false, false, false, true, shadow, type.VSCROLL.SLIDER);
  }
  
  // grab, drag and drop
  var grab = function(evnt) { 
    if (!evnt) evnt = window.event;
    SCROLL.DRAG.drag    = function(e) { SCROLL.drag(slider, inner, content, false, e); };
    SCROLL.DRAG.release = function(e) { SCROLL.release(e); };
    SCROLL.grab(slider, evnt);
  };
  addEventListener(slider, 'mousedown', grab);
  
  // hide unnecessary scrollbars
  if (content.offsetHeight <= inner.offsetHeight) {
    scroll.style.visibility = 'hidden';
    content.style.marginTop = 0;
  }
}


/**
 * Create a horizontal scroll bar.
 * @param outerId    id of the element containing the scrollable 
 *                   element and the newly created scrollbar
 * @param innerId    id if the element defining the visible frame 
 *                   of the scrollable area
 * @param contentId  id of the scrollable element (partly visible)
 * @param type       type of scrollbar (e.g. SCROLL.PN4W)
 */
function hscroll (outerId, innerId, contentId, type)
{
  var outer   = document.getElementById(outerId);
  var inner   = document.getElementById(innerId);
  var content = document.getElementById(contentId);
  removeElement(innerId + '.hscroll.frame');

  // frame and background
  var scroll = appendNewElement(outer, 'div', innerId + '.hscroll.frame', type.HSCROLL.CLAZZ);
  var w = inner.offsetWidth;
  scroll.style.width = w;
  scroll.style.top   = inner.offsetTop + inner.offsetHeight + type.HSCROLL.YOFFSET;
  scroll.style.left  = inner.offsetLeft;
  if (type.HSCROLL.BG) {
    decorate(inner.id + '.hscroll', false, false, false, true, false, type.HSCROLL.BG);
  } 
  
  // slider
  var slider = appendNewElement(scroll, 'div', scroll.id + '.slider.frame', type.HSCROLL.SLIDER.CLAZZ);
  var o = type.HSCROLL.SLIDER.MINSIZE;
  var s = toInt((w-o) * Math.min(1, w/getWidth(content)) + o);
  var m = (content.style.marginLeft ? parseInt(content.style.marginLeft) : 0);
  var r = (content.offsetWidth <= inner.offsetWidth ? 0 : -m / (content.offsetWidth-inner.offsetWidth));
  slider.style.width = s;
  slider.style.top   = 0;
  slider.style.left  = toInt(Math.max(0.0, Math.min(1.0, r)) * (w - s));
  if (type.HSCROLL.SLIDER) {
    var shadow = (type.HSCROLL.SLIDER.SHADOW ? true : false);
    decorate(scroll.id + '.slider', false, false, false, true, shadow, type.HSCROLL.SLIDER);
  }
  
  // grab, drag and drop
  var grab = function(evnt) { 
    if (!evnt) evnt = window.event;
    SCROLL.DRAG.drag    = function(e) { SCROLL.drag(slider, inner, content, true, e); };
    SCROLL.DRAG.release = function(e) { SCROLL.release(e); };
    SCROLL.grab(slider, evnt);
  };
  addEventListener(slider, 'mousedown', grab);
  
  // hide unnecessary scrollbars
  if (content.offsetWidth <= inner.offsetWidth) {
    scroll.style.visibility = 'hidden';
    content.style.marginLeft = 0;
  }
}


// DRAGGING =========================================================

/**
 * Grab slider for dragging (mouse down). 
 * @param slider  slider html element
 * @param evnt    event
 */
SCROLL.grab = function (slider, evnt)
{
  if (!evnt) evnt = window.event;
  preventDefault(evnt);
  SCROLL.DRAG.x0      = pageX(evnt);
  SCROLL.DRAG.y0      = pageY(evnt);
  SCROLL.DRAG.xOffset = SCROLL.DRAG.x0 - getLeft(slider);
  SCROLL.DRAG.yOffset = SCROLL.DRAG.y0 -  getTop(slider);
};

/**
 * Drag slider (mouse move while button pressed). 
 * @param slider   slider html element
 * @param frame    visible area element
 * @param content  scrollable element
 * @param evnt     event
 */
SCROLL.drag = function (slider, frame, content, horizontal, evnt)
{
  if (!evnt) evnt = window.event;
  preventDefault(evnt);

  // notice: if outer is already removed then it has no parent anymore
  var scroll = slider.parentNode;
  if (scroll == null) { SCROLL.release(evnt); return; }

  // drag (set position)
  var xMax  = scroll.offsetWidth  - slider.offsetWidth;
  var yMax  = scroll.offsetHeight - slider.offsetHeight;
  var x     = Math.max(0, Math.min(xMax, pageX(evnt) - SCROLL.DRAG.xOffset - getLeft(scroll)));
  var y     = Math.max(0, Math.min(yMax, pageY(evnt) - SCROLL.DRAG.yOffset -  getTop(scroll)));
  if (horizontal) {
    slider.style.left = x;
    var w1 = scroll.offsetWidth - slider.offsetWidth;
    var w2 = content.offsetWidth - frame.offsetWidth;
    var xOffset = -toInt(slider.offsetLeft / w1 * w2);
    content.style.marginLeft = xOffset;
  } else {
    slider.style.top = y; 
    var h1 = scroll.offsetHeight - slider.offsetHeight;
    var h2 = content.offsetHeight - frame.offsetHeight;
    var yOffset = -toInt(slider.offsetTop / h1 * h2);
    content.style.marginTop = yOffset;
  }
};

/**
 * Release slider (mouse up). 
 */
SCROLL.release = function (evnt)
{  
  if (!evnt) evnt = window.event;
  preventDefault(evnt);
  SCROLL.DRAG.drag    = dummyEventHandler;
  SCROLL.DRAG.release = dummyEventHandler;
};
