/*
 * -------------------------------------------------------------------------
 *
 *  pn4webYJTree.js Copyright (C) 2011, 
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *   Author: Patrick Rammelt 
 *   E-Mail: patrick.rammelt@go4more.de
 *   Site:   http://www.patricks-seite.de
 * -------------------------------------------------------------------------
 *
 * Junction tree for y-propagation for bayesian networks. This class forms
 * a special junction tree which is usable to find the joint probability
 * table for a pre-defined fixed set of nodes.
 * (These types are fully independent of display functionalities)
 */


// JUNCTION TREE FOR Y-PROPAGATION ==================================
// CONSTRUCT / COMPILE ----------------------------------------------

/**
 * Constructs a junction tree, which is usable for far
 * more complex networks than the full joint propagator.
 * It's also usable for disconnected networks.
 * @param net     bayesian network to work on
 * @param restr   minimum allowed order for all or some nodes 
 *                (restr[nodeId])
 * @param fillin  put fill-in links for moralization and triangulation 
 *                into the original bayesian network? [optional]
 * @param max     use max-propagation?
 * @param nodes   mapping node Id to node, for all nodes to find the 
 *                joint probability table for
 */
function YJTree (net, restr, fillin, max, nodes)
{
  if (net != undefined) {
    this.initYJTree(net, restr, fillin, max, nodes);
  }
}

/* Inheritance */
YJTree.prototype             = new JTree();
YJTree.prototype.parent      = JTree.prototype;
YJTree.prototype.constructor = YJTree;

/**
 * Compile / Recompile the given bayesian network 
 * (see constructor {@link YJTree}).
 */
YJTree.prototype.initYJTree = function (net, restr, fillin, max, nodes) 
{
  this.initJTree(net, restr, fillin, max);

  this.id     = net.id + '.jtree';
  this.name   = 'junction tree for y-propagation for net "' + net.name + '"';
  this.joints = nodes;
  
  // create a "super-root" clique (initially empty, will be filled by 'expand()')
  var root = new Clique(-1, 'Super-Root', {}, -1);
  this.addNode(root);

  // connect super-root to the best connector clique in each cluster
  // (links don't need separators, they will be created later by 'expand()')
  for (var rId in this.roots) {
    var connector = this.findConnector(this.roots[rId], nodes);
    this.addLink(root, connector.clique);
  }

  // now establish the the super-root clique as the new single root
  // (including changed of link-directions as needed)
  this.roots = {};
  this.roots[root.id] = root;                      
  this.align(root, null);
  
  // expand cliques and separators such that all given 'nodes' will
  // be rooted through all the way towards the super-root-clique
  // notice: it's rather inefficient to recreate all cliques and 
  //         separators, but it's easy to implement...
  this.expand(root, nodes);

  // notice: all elements in all clique JPT's have been initialized with 1
  this.absorp();
};

/**
 * Find the clique in a cluster starting with the given clique
 * which has the biggest intersection with the nodes to find the 
 * joint probability table for
 * 
 * @param clique  clique to start at (initially one of the 'roots')
 * @param nodes   mapping node Id to node, for all nodes to find 
 *                the joint probability table for
 * @returns       best match, where 'match.clique' is the clique with 
 *                the highest intersection with 'nodes', and 
 *                'match.match' is a intersection measurement
 */
YJTree.prototype.findConnector = function (clique, nodes)
{
  var best = {clique : clique,
	      match  : this.getMatch(clique, nodes)};
  for (var chId in this.children[clique.id]) {
    var child = this.nodes[chId];
    var sub = this.findConnector(child, nodes);
    if (sub.match > best.match) {
      best = sub;
    }
  }
  return (best);
};

/**
 * Calculates a number which is higher the more qualified the 
 * given clique is to become the connector-clique to the new 
 * super-root-clique. In this implementation we're just counting 
 * the number of nodes which exist in both: 'nodes' and also in 
 * 'clique.nodes'.
 * @param clique  clique to calculate the match-value
 * @param nodes   mapping node Id to node, for all nodes to find 
 *                the joint probability table for
 */
YJTree.prototype.getMatch = function (clique, nodes)
{
  var n = 0;
  for (var nId in nodes) {
    if (clique.nodes[nId]) n++;
  }
  return (n);
};

/**
 * Flip directions of all separator links such that clique 
 * becomes the root of its sub-tree.
 * @param clique      clique to become the root of its sub-tree
 * @param realParent  the only clique parent clique (initially null 
 *                    or undefined)
 */
YJTree.prototype.align = function (clique, realParent)
{
  for (var chId in this.children[clique.id]) {
    var child = this.nodes[chId];
    if (child != realParent) {
      this.align(child, clique);
    }
  }
  for (var paId in this.parents[clique.id]) {
    var parent = this.nodes[paId];
    if (parent != realParent) {
      this.align(parent, clique);
      var sep = this.parents[clique.id][paId];
      this.rmLink(parent, clique);
      this.addLink(clique, parent, sep);
    }
  }
};

/**
 * Expand cliques and separators, such that the nodes to
 * create the joint probability table for never get marginalized
 * out on their way towards the "super-root" clique
 * @param clique  root of current sub-tree (start with one and only 
 *                clique in this.roots, i.e. the super-root-clique)
 * @param nodes   mapping node Id to node, for all nodes to find 
 *                the joint probability table for
 */
YJTree.prototype.expand = function (clique, nodes)
{
  var cId = clique.id, chId1 = undefined;
  for (chId1 in this.children[cId]) {
    var child = this.nodes[chId1];
    //var sep = this.children[cId][chId1];
    this.expand(child, nodes);

    // collect joint nodes to be added to this clique
    for (var chId2 in this.children[cId]) {
      var child = this.nodes[chId2];
      for (var nId in child.nodes) {
        if (nodes[nId] && !clique.nodes[nId]) {
          clique.nodes[nId]  = nodes[nId];
          clique.addins[nId] = nodes[nId];
        }
      }
    }

    // recreate JPT of this clique and all separators towards child-cliques
    clique.createJPT();
    for (var chId2 in this.children[cId]) {
      var child = this.nodes[chId2];
      var sep = new Separator(clique, child);
      this.addLink(clique, child, sep);       // replace old link (separator)
    }
  }

  // if this clique has no children, just initialize the JPT
  if (!chId1) clique.jpt.init(1);
};

/**
 * Fully recreate this propagator (includes 
 * layouting (if 'layout()' has been called).
 */
YJTree.prototype.recreate = function () 
{
  this.initYJTree(this.net, this.restr, this.fillin, this.max, this.joints);
  if (this.w || this.h) {
    this.layout(this.w, this.h, this.rot90);
  }
};

// GETTER / SETTER --------------------------------------------------

/**
 * Return a joint probability table containing
 * at least the given node and all its parents.
 * @param node  node in the bayesian network ('this.net')
 *              (if no node is given the JPT for the defined 
 *              set of nodes is returned)
 * @returns     joint probability table P(node,pa(node),...)
 */
YJTree.prototype.getJPT = function (node) 
{
  if (!node) {
    return (this.roots[0].jpt);
  } else {
    return (this.home[node.id].jpt);
  }
};

// PROPAGATE --------------------------------------------------------

/**
 * Y-Propagate. This method does not update the posteriors of
 * any node. If you want to also update posteriors just use
 * 'propagate(observe,true)' and then get the joint probability
 * table by 'getJPT()' (the main difference is that this method
 * does not perform the distribute phase and therefor couldn't 
 * update the posteriors).
 * @param observe  use current evidence?
 * @param return   joint probability table for the nodes in question
 */ 
JTree.prototype.yPropagate = function (observe) 
{
  // re-initialize and include evidence
  this.init();
  if (observe) this.observe();

  // propagate (HUGIN scheme)
  var pe = 1.0;
  for (var rId in this.roots) {                // over all disconnected sub-trees
    this.collect(this.roots[rId]);             // collect phase
    pe *= (this.max                            // normalize root which gives...
	   ? this.roots[rId].jpt.mnorm()       // ...P(C|e) or
	   : this.roots[rId].jpt.norm());      // ...P(e|M)
  }

  this.pe = pe;
  return (this.roots[0].jpt);
};
