/*
 * -------------------------------------------------------------------------
 *
 *  pn4webSampling.js Copyright (C) 2011, 
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *   Author: Patrick Rammelt 
 *   E-Mail: patrick.rammelt@go4more.de
 *   Site:   http://www.patricks-seite.de
 * -------------------------------------------------------------------------
 *
 * Propagators using sampling methods. These propagators could be used 
 * even for very complex networks, because their representation is not 
 * much bigger than the original bayesian network. Results are 
 * approximative only, although for many networks the results are quite 
 * accuarate. Unfortunately there are also situations where sampling 
 * methods fail (even for very large numbers of samples).
 * The main focus of this implementation was to keep it as simple as 
 * possible rather than to get the highest possible performance (see 
 * some comments). Furthermore these types are fully independent of 
 * display functionalities.
 */


// LIKELIHOOD WEIGHTING SAMPLER =====================================
// CONSTRUCT --------------------------------------------------------

/**
 * Constructs a likelihood weighting sampler (propagator).
 * @param net           bayesian network to work on
 * @param numOfSamples  number of samples create for one propagation
 * @param keepSamples   store sampled data in this.samples for each
 *                      propagation (e.g. to be shown), where
 *                      stateIndex = this.samples.data[r][c]
 *                      and weight = this.samples.data[r].weight?
 */
function LWSampler (net, numOfSamples, keepSamples)
{
  if (net != undefined) {
    this.initLWSampler(net, numOfSamples, keepSamples);
  }
}

/* Inheritance */
// LWSampler.prototype          = ;
// LWSampler.prototype.parent   = ;
LWSampler.prototype.constructor = LWSampler;

/**
 * Compile / Recompile the given bayesian network 
 * (see constructor {@link LWSampler}).
 */
LWSampler.prototype.initLWSampler = function (net, numOfSamples, keepSamples) 
{
  // set/init attributes
  this.id       = net.id + '.lws';
  this.name     = 'LW-Sampler for "' + net.name + '"';
  this.net      = net;              // bayesian network
  this.n        = numOfSamples;     // number of samples for one propagation
  this.keep     = keepSamples;         
  this.pe       = 1.0;              // probability of evidence
  this.order    = new NaturalOrdering(net).getOrderedNodes();

  // create contingency tables for each node (becoming 
  // joint probability tables after normalization)
  this.reload();

  // create dataset for sampled data
  if (keepSamples) {
    this.samples = new Samples(this.order, this.n);
  }
};

/**
 * Re-load probabilities from the bayesian network
 * (probabilities of some nodes might have changed
 * or the number of states of one dimensions might 
 * have changed).
 */
LWSampler.prototype.reload = function () 
{
  // create contingency tables for each node (becoming 
  // joint probability tables after normalization)
  // it is also possible to use posterior probability
  // tables of the nodes in the bayesian network directly
  // as contingency tables, but this has two drawbacks:
  // 1. it wouldn't be possible to propagate without 
  //    changing/updating the bayesian network (its nodes)
  // 2. we couldn't implement method 'getJPT(node)', which
  //    is used e.g. by some learning algorithms
  this.cts = {};
  for (var nId in this.net.nodes) {
    var node = this.net.nodes[nId];
    this.cts[nId] = new Float32Table(node.cpt.dims);
  }
};

/**
 * Fully recreate this propagator.
 */
LWSampler.prototype.recreate = function () 
{
  this.initLWSampler(this.net, this.n, this.keep);
};

// GETTER / SETTER --------------------------------------------------

/**
 * Return a joint probability table containing
 * at least the given node and all its parents.
 * @param node  node in the bayesian network ('this.net')
 * @returns     joint probability table P(node,pa(node),...)
 */
LWSampler.prototype.getJPT = function (node) 
{
  return (this.cts[node.id]);
};

/**
 * Get the samples (only if 'keepSamples' was enabled).
 */
LWSampler.prototype.getSamples = function () 
{
  return (this.samples);
};

// PROPAGATE --------------------------------------------------------

/**
 * Create samples (could then be retrieved by 'getSamples()').<br>
 * Notice: This is just a synonym for:
 *         'propagate(false, false);' and 'getSamples();'
 * @returns  samples (only if 'keepSamples' was enabled)
 */ 
LWSampler.prototype.sample = function ()
{
  this.propagate(false, false);
  return (this.samples);
};

/**
 * Propagate (and update a-posteriori probability of all nodes).
 * @param observe  use current evidence?
 * @param update   update posterior probability tables of all nodes?
 * @returns        probability of evidence given the model: P(e|M)
 */ 
LWSampler.prototype.propagate = function (observe, update) 
{
  // re-initialize contingency tables
  this.initCTs();

  // creating n samples
  var pe     = 0;
  var sample = {};   // current sample (sample[nodeId] = stateIndex)
  for (var m = 0; m < this.n; m++) {

    // sample a state for all nodes given the states of their 
    // parents (parents have to be sampled before children)
    var weight = 1;
    for (var o = 0; o < this.order.length; o++) { 
      var node = this.order[o];

      // find starting index i into node's CPT
      var i = node.cpt.getIndex(sample, 1, node.cpt.dims.length-1);

      // calculate weight and update overall weight
      var w = this.getWeight(observe, node.f, node.cpt, i);
      weight *= w;

      // sample a new value (node.id == node.cpt.dims[0].id)
      sample[node.id] = this.sampleState(observe, node.f, node.cpt, i, w);
      if (this.keep) this.samples.data[m][o] = sample[node.id];
    }

    // adding (weighted) samples to all contingency tables
    this.cumulateCTs(sample, weight);

    if (this.keep) this.samples.data[m].weight = weight;
    pe += weight;
  }

  // normalize contingency tables and update node posteriors
  this.normCTs();
  if (update) this.update();

  pe /= this.n;
  this.pe = pe;
  return (pe);
};

/**
 * Initialize all contingency tables (all values = 0).
 */
LWSampler.prototype.initCTs = function ()
{
  for (var nId in this.cts) {
    this.cts[nId].init(0);
  }
};

/**
 * Adding a (weighted) sample to all contingency tables
 * @param sample  sample[dimId] = sampledStateNumber
 * @param weight  overall weight for that sample
 */
LWSampler.prototype.cumulateCTs = function (sample, weight)
{
  for (var nId in this.cts) {
    var i = this.cts[nId].getIndex(sample, 0, this.cts[nId].dims.length-1);
    this.cts[nId].values[i] += weight;
  }
};

/**
 * Normalize contingency tables (they are no longer
 * contingency tables but (joint) probability tables).
 */
LWSampler.prototype.normCTs = function ()
{
  for (var nId in this.cts) {
    this.cts[nId].norm();
  }
};

/**
 * Update posteriors of all nodes using (normalized
 * contingency tables, i.e. joint probability tables)
 */
LWSampler.prototype.update = function () 
{
  for (var nId in this.cts) {
    var node = this.net.nodes[nId];
    node.p.marg(this.cts[nId]);
  }
};

/**
 * Get the weight for a single node's evidence
 * observe: really use evidence?
 * @param finding  finding vector representing the current evidence 
 *                 for that node (even if in = false)
 * @param cpt      conditional probability table of that node
 * @param i        first index into the cpt for the current sample
 * @returns        weight for the given evidence of that node [0..1]
 *                 (i.e. sum(f .* p(n|pa(n))))
 */
LWSampler.prototype.getWeight = function (observe, finding, cpt, i)
{
  var weight = 0;
  var id = cpt.dims[0].id;    // id of first dimension (=finding.dims[0].id)
  for (var s = 0, j = 0; s < cpt.sizes[id]; s++) {
    weight += cpt.values[i] * (observe ? finding.values[j] : 1); 
    i += cpt.steps[id];       // (cpt.steps[id] and finding.steps[id]
    j += finding.steps[id];   //  should be 1 because it's the first dim)
  }
  return (weight);
};

/**
 * Sample the state for one node
 * @param observe  really use evidence?
 * @param finding  finding vector representing the current evidence 
 *                 for that node (even if in = false)
 * @param cpt      conditional probability table of that node
 * @param i        first index into the cpt for the current sample
 * @param weight   weight for of the evidence for that node 
 *                 (i.e. sum(f .* p(n|pa(n))))
 * @returns        sampled state-index
 */ 
LWSampler.prototype.sampleState = function (observe, finding, cpt, i, weight)
{
  var u = 0;
  var v = Math.random() * weight;
  var id = cpt.dims[0].id;    // id of first dimension (=finding.dims[0].id)
  for (var s = 0, j = 0; s < cpt.sizes[id]; s++) {
    u += cpt.values[i] * (observe ? finding.values[j] : 1);
    if (u >= v) break;
    i += cpt.steps[id];       // (cpt.steps[id] and finding.steps[id]
    j += finding.steps[id];   //  should be 1 because it's the first dim)
  }
  return (s);
};


// GIBBS SAMPLER ====================================================
// CONSTRUCT / COMPILE ----------------------------------------------

/**
 * Constructs a gibbs sampler (propagator).
 * @param net           bayesian network to work on
 * @param numOfSamples  number of samples create for one propagation
 * @param reSampleIn    number of samples before a new starting 
 *                      configuration is created
 * @param burnIn        additional number of samples created first and
 *                      after every re-sample-in step (these samples are
 *                      not used for posterior estimation)
 * @param minWeight     minimumWeight for a initial sample to be accepted
 *                      (in fact the weight has to be bigger than this)
 * @param maxTrials     maximum number of trials to find an initial sample
 * @param keepSamples   store sampled data in this.samples for each
 *                      propagation (e.g. to be shown), where
 *                      stateIndex = this.samples.data[r][c]
 *                      and weight = this.samples.data[r].weight
 */
function GibbsSampler (net, numOfSamples, reSampleIn, burnIn, 
                       minWeight, maxTrials, keepSamples)
{
  if (net != undefined) {
    this.initGibbsSampler(net, numOfSamples, reSampleIn, burnIn, 
			  minWeight, maxTrials, keepSamples);
  }
}

/* Inheritance */
GibbsSampler.prototype             = new LWSampler();
GibbsSampler.prototype.parent      = LWSampler.prototype;
GibbsSampler.prototype.constructor = GibbsSampler;

/**
 * Compile / Recompile the given bayesian network 
 * (see constructor {@link GibbsSampler}).
 */
GibbsSampler.prototype.initGibbsSampler = function (net, numOfSamples, reSampleIn, burnIn, 
                                                    minWeight, maxTrials, keepSamples) 
{
  // set/init attributes
  this.id       = net.id + '.gibbs';
  this.name     = 'Gibbs-Sampler for "' + net.name + '"';
  this.net      = net;              // bayesian network
  this.n        = numOfSamples;     // number of samples for one propagation
  this.interval = reSampleIn;
  this.skip     = burnIn;
  this.weight   = minWeight;    
  this.trials   = maxTrials;
  this.keep     = keepSamples; 
  this.pe       = 1.0;              // probability of evidence
  this.order    = new NaturalOrdering(net).getOrderedNodes();
  this.mbNet    = null;

  // create conditional probability table for each node 
  // given all other nodes in its markov blanket set
  this.reload();
  
  // create dataset for sampled data
  if (keepSamples) {
    this.samples = new Samples(this.order, this.n);
  }
};

/**
 * Re-load probabilities from the bayesian network
 * (probabilities of some nodes might have changed
 * or the number of states of one dimensions might 
 * have changed).
 */
GibbsSampler.prototype.reload = function () 
{
  // call method in parent class (create contingency tables)
  this.parent.reload.call(this);

  // to illustrate the structure we create a network of markov
  // blankets, where each markov blanket is represented by a
  // clique (normally used by junction-trees); for sampling / 
  // propagation we only need the markov blanket tables for 
  // each node (see 'getMBT()')
  // important: re-use existing network and nodes otherwise 
  //            display methods are confused!
  if (!this.mbNet) this.mbNet = new Net(this.id + '.mbnet', 'Markov-Blanket-Network');

  // over all nodes
  for (var i = 0; i < this.order.length; i++) { // parents berfore children
    var node = this.order[i];

    // create a markov-blanket "clique" for that node
    var mbSet  = this.net.getMarkovBlanket(node);
    var reuse  = this.mbNet.nodes[node.id] !== undefined;
    var mbNode = (reuse ? this.mbNet.nodes[node.id]
                        : new Clique(node.id, 'MB(' + node.name + ')', mbSet, 0));
    mbNode.x = node.x; 
    mbNode.y = node.y;
    if (!reuse) this.mbNet.addNode(mbNode);
    else        mbNode.createJPT();         // (jpt could be and has to be recreated)

    // multiply in CPTs of that node and all children and renormalize
    mbNode.jpt.mult(node.cpt);
    for (var chId in this.net.children[node.id]) {
      var child = this.net.nodes[chId];
      mbNode.jpt.mult(child.cpt);
    }
    mbNode.jpt.cnorm(); // make conditional (P(node|markovBlanket))

    // also add links (not used, just for illustration)
    for (var paId in this.net.parents[node.id]) {
      var mbParent = this.mbNet.nodes[paId];
      this.mbNet.addLink(mbParent, mbNode);
    }
  }
};

/**
 * Get the markov blanket probability table (kind of a CPT). 
 * @param node  node in the bayesian network (this.net)
 * @returns     markov blanket probability table for that node
 */
GibbsSampler.prototype.getMBT = function (node) 
{
  return (this.mbNet.nodes[node.id].jpt);
};

/**
 * Fully recreate this propagator.
 */
GibbsSampler.prototype.recreate = function () 
{
  this.initGibbsSampler(this.net, this.n, this.interval, this.skip, 
			this.weight, this.trials, this.keep);
};

// PROPAGATE --------------------------------------------------------

/**
 * Propagate (and update a-posteriori probability of all nodes).
 * @param observe  use current evidence?
 * @param update   update posterior probability tables of all nodes?
 * @returns        probability of evidence given the model: P(e|M)
 */ 
GibbsSampler.prototype.propagate = function (observe, update) 
{
  // re-initialize contingency tables
  this.initCTs();

  // creating n samples
  var pe     = 0;
  var sample = {};   // current sample (sample[nodeId] = stateIndex)
  var m      = 0;    // number of samples created
  do {

    // find starting config
    this.start(observe, sample);

    // burn in phase
    this.burnIn(observe, sample);

    // create real samples (used to estimate posteriors)
    for (var k = 0; k < this.interval && m < this.n; k++, m++) {
      for (var o = 0; o < this.order.length; o++) {  // ordering doesn't matter 
        var node = this.order[o];
        var mbt  = this.getMBT(node);   // (markov-blanket CPT)
        var i    = mbt.getIndex(sample, 1, mbt.dims.length-1);
        var w    = this.getWeight(observe, node.f, mbt, i);
        sample[node.id] = this.sampleState(observe, node.f, mbt, i, w);
        if (this.keep) this.samples.data[m][o] = sample[node.id];
      }
      this.cumulateCTs(sample, 1);      // (every sample has the same weight!)

      // calculate probability of evidence P(e|M) using the CPT's of
      // the bayesian network instead of the markov blanket tables,
      // which are kind of circularly defined 
      // TODO: is there a more efficient way to get this weight?
      var weight = 1;
      for (var o = 0; o < this.order.length; o++) {  // parents before children 
        var node = this.order[o];
        var i    = node.cpt.getIndex(sample, 1, node.cpt.dims.length-1);
        var w    = this.getWeight(observe, node.f, node.cpt, i);
        weight  *= w;
      }
      pe += weight;
      if (this.keep) this.samples.data[m].weight = weight;
    }

  } while (m < this.n);

  // normalize contingency tables and update node posteriors
  this.normCTs();
  if (update) this.update();

  pe /= this.n;
  this.pe = pe;
  return (pe);
};

/**
 * Find a starting configuration (sample) 
 * @param observe  use evidence?
 * @param sample   configuration (sample[nodeId] = stateIndex)
 */
GibbsSampler.prototype.start = function (observe, sample)
{
  var weight, trials = 0;
  do {
    if (trials >= this.trials) {
      throw (new Error('No initial configuration found - max. number of ' + this.trials + ' trials reached.'));
    }
    weight = 1.0;
    for (var o = 0; o < this.order.length; o++) {   // parents before children
      var node = this.order[o];
      var i    = node.cpt.getIndex(sample, 1, node.cpt.dims.length-1);
      var w    = this.getWeight(observe, node.f, node.cpt, i);
      sample[node.id] = this.sampleState(observe, node.f, node.cpt, i, w);
      weight *= w;
    }
    trials++;
  } while (weight <= this.weight);
};

/**
 * Burn in phase, create samples which aren't used
 * to estimate posteriors probabilities of nodes
 * @param observe  use evidence?
 * @param sample   configuration (sample[nodeId] = stateIndex)
 */
GibbsSampler.prototype.burnIn = function (observe, sample)
{
  for (var m = 0; m < this.skip; m++) {
    for (var o = 0; o < this.order.length; o++) { // ordering doesn't matter
      var node = this.order[o];
      var mbt  = this.getMBT(node);               // (markov-blanket CPT)
      var i    = mbt.getIndex(sample, 1, mbt.dims.length-1);
      var w    = this.getWeight(observe, node.f, mbt, i);
      sample[node.id] = this.sampleState(observe, node.f, mbt, i, w);
    }
  }
};
