/*
 * -------------------------------------------------------------------------
 *
 *  pn4webLoopyBelief.js Copyright (C) 2011, 
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *   Author: Patrick Rammelt 
 *   E-Mail: patrick.rammelt@go4more.de
 *   Site:   http://www.patricks-seite.de
 * -------------------------------------------------------------------------
 *
 * Loopy belief propagator for bayesian networks using cliques and
 * separators (taken from pn4webJTree.js). This propagator could be 
 * used even for very complex networks, because its representation is 
 * not much bigger than the original bayesian network. For loopy belief
 * propagation there is a network created containing one clique for each
 * node of the bayesian network (plus the parents of this node). Each 
 * such clique is holding a joint probability table of the same size as 
 * the node's CPT. Additionally all cliques sharing common nodes are 
 * connected by undirected links, holding a separator. During a loopy 
 * belief propagation every clique receives messages from all neighboring 
 * cliques. This process has to be repeated for several iterations (loops) 
 * to converge. For tree-like bayesian networks it is guaranteed that the 
 * results converge towards the correct posterior distributions. For non 
 * tree-like structures the results are often very close to the true ones. 
 * Unfortunately there is no indicator for the quality of loopy belief 
 * propagation for general bayesian networks.
 * The main focus of this implementation was to keep it as simple as 
 * possible rather than to get the highest possible performance (see 
 * some comments). Furthermore these types are fully independent of 
 * display functionalities.
 */


// LOOPY BELIEF PROPAGATOR ==========================================
// CONSTRUCT / COMPILE ----------------------------------------------

/**
 * Constructs a loopy belief propagator, which is usable for
 * very complex networks. On the other hand results will 
 * converge towards the correct values only for bayesian 
 * networks with a tree-like structure.
 * @param net    bayesian network to work on
 * @param loops  number of loops for one propagation
 * @param max    max-propagation?
 */
function LoopyBelief (net, loops, max)
{
  if (net != undefined) {
    this.initLoopyBelief(net, loops, max);
  }
}

/* Inheritance */
LoopyBelief.prototype             = new Net();
LoopyBelief.prototype.parent      = Net.prototype;
LoopyBelief.prototype.constructor = LoopyBelief; 

/**
 * Compile / Recompile the given bayesian network 
 * (see constructor {@link LoopyBelief}).
 */
LoopyBelief.prototype.initLoopyBelief = function (net, loops, max) 
{
  this.initNet(net.id + '.lbp', 'Loopy belief propagator for "' + net.name + '"');

  // set/init attributes
  this.net      = net;    // bayesian network
  this.max      = max;    // max-propagation?
  this.loops    = loops;  // number of loops for one propagation
  this.pe       = 1.0;    // probability of evidence / of the most probable config. (max-prop)
  this.home     = {};     // map: node-id to home clique

  // create loopy belief network
  this.build();           // create and connect cliques
};

/**
 * Re-load probabilities from the bayesian network
 * (probabilities of some nodes might have changed
 * or the number of states of one dimensions might 
 * have changed).
 */
LoopyBelief.prototype.reload = function () 
{
  for (var cId in this.nodes) {
    var clique = this.nodes[cId];
    clique.jpt.initPTable(clique.jpt.dims); // number of states might have changed
    for (var chId in this.children[cId]) {
      var sep = this.children[cId][chId];
      sep.jpt.initPTable(sep.jpt.dims);     // number of states might have changed
    }
  }
};

/**
 * Fully recreate this propagator.
 */
LoopyBelief.prototype.recreate = function () 
{
  this.initLoopyBelief(this.net, this.loops);
};

/**
 * Creates a "loopy belief network" consisting of
 * cliques (nodes) and separators (links) like a
 * junction tree. Unlike a junction tree these 
 * cliques are not guaranteed to be connected in
 * a tree-like structure.
 */
LoopyBelief.prototype.build = function ()
{
  // create cliques (one for each node and its parents)
  var o = 0;    // "order" (like elim.-order in jtrees)
  for (var nId in this.net.nodes) {
    var node  = this.net.nodes[nId];
    var nodes = {};
    for (var i = 0; i < node.cpt.dims.length; i++) {
      nodes[node.cpt.dims[i].id] = node.cpt.dims[i];
    }
    var clique = new Clique(node.id, 'Clique ' + o, nodes, o);
    clique.x = node.x;
    clique.y = node.y;
    this.addNode(clique);
    this.home[nId] = clique;
    o++;
  }

  // connect cliques of parents and children
  for (var paId in this.net.children) {
    var parent = this.home[paId];
    for (var chId in this.net.children[paId]) {
      var child = this.home[chId];
      var sep = new Separator(parent, child);
      this.addLink(parent, child, sep);
    }
  }
};

// GETTER / SETTER --------------------------------------------------

/**
 * Return a joint probability table containing
 * at least the given node and all its parents.
 * @param node  node in the bayesian network ('this.net')
 * @returns     joint probability table P(node,pa(node),...)
 */
LoopyBelief.prototype.getJPT = function (node) 
{
  return (this.home[node.id].jpt);
};

// PROPAGATE --------------------------------------------------------

/**
 * Propagate (and update a-posteriori probability of all nodes).
 * @param observe  use current evidence?
 * @param update   update posterior probability tables of all nodes?
 * @returna        probability of evidence given the model: P(e|M)
 *                 or of the most probable config. (max-prop): P(C|e)
 */ 
LoopyBelief.prototype.propagate = function (observe, update) 
{
  // re-initialize and include evidence
  this.init();
  if (observe) this.observe();

  // loopy belief propagation: for the given number of 
  // iterations (loops) and for all cliques: collect 
  // evidence from all neighboring cliques
  var pe = 1;
  for (var loop = 0; loop < this.loops; loop++) {
    for (var id in this.nodes) {
      var clique = this.nodes[id];

      // collect evidence from neighboring cliques
      for (var paId in this.parents[id]) {
        var parent = this.nodes[paId];
        var sep    = this.parents[id][paId];
        this.send(parent, clique, sep);
      }
      for (var chId in this.children[id]) {
        var child = this.nodes[chId];
        var sep   = this.children[id][chId];
        this.send(child, clique, sep);
      }

      // normalize
      pe *= (this.max 
             ? clique.jpt.mnorm()
             : clique.jpt.norm());
    }
  }

  // update a-posteriori probabilities of all nodes
  if (update) this.update();

  this.pe = pe;
  return (pe);
};

/**
 * Re-initialize probability tables of all cliques and separators.
 */
LoopyBelief.prototype.init = function ()
{
  for (var nId in this.net.nodes) {
    var node   = this.net.nodes[nId];
    var clique = this.home[nId];
    clique.jpt.values = node.cpt.values.slice();
    for (var chId in this.children[clique.id]) {
      var sep = this.children[clique.id][chId];
      sep.jpt.init(1);
    }
  }
};

/**
 * Multiply in all finding vectors (evidence) of all nodes
 * into the joint probability table of their home clique.
 */
LoopyBelief.prototype.observe = function ()
{
  for (var nId in this.home) {
    var node   = this.net.nodes[nId];
    var clique = this.home[nId];
    clique.jpt.mult(node.f);
  }
};

/**
 * Send message from one clique to another clique.
 * via the given separator.<br>
 * Notice: simple but rather inefficient implementation
 *         because c * (s'/s) would need less operations
 *         than (c/s) * s', because c is bigger than s
 *         (where c = clique table, s = separator table)
 * @param from  sending clique
 * @param to    reveiving clique
 * @param sep   separator
 */
LoopyBelief.prototype.send = function (from, to, sep)
{
  to.jpt.div(sep.jpt);         // divide by old separator (s)
  (this.max 
   ? sep.jpt.mmarg(from.jpt)
   : sep.jpt.marg(from.jpt));  // create new separator (s')
  to.jpt.mult(sep.jpt);        // multiplicate by new separator
};

/**
 * Update a-posteriori probabilities of all nodes.
 */ 
LoopyBelief.prototype.update = function () 
{
  for (var nId in this.net.nodes) {
    var node = this.net.nodes[nId];
    (this.max 
     ? node.p.mmarg(this.home[node.id].jpt)
     : node.p.marg(this.home[node.id].jpt)); 
  }
};
