/*
 * -------------------------------------------------------------------------
 *
 *  pn4webDynamic.js Copyright (C) 2011, 
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *   Author: Patrick Rammelt 
 *   E-Mail: patrick.rammelt@go4more.de
 *   Site:   http://www.patricks-seite.de
 * -------------------------------------------------------------------------
 *
 * Algorithms to retrieve conditional probabilities from (incomplete) data
 * (Expectation-Maximization-Algorithm) and to learn the structure (causal
 * dependencies) from (complete) data (these types are fully independent of 
 * display functionalities).
 */


// SIMPLE PARAMETRIZER FOR COMPLETE DATA ============================

/**
 * Constructor for a parameztrizer which creates the CPTs 
 * of some/all nodes from complete(!) data. This means just 
 * "counting", i.e. no real "learning" is involved.
 */
function Parametrizer ()
{
  this.initParametrizer();
}

/* Inheritance */
// Parametrizer.prototype          = ;
// Parametrizer.prototype.parent   = ;
Parametrizer.prototype.constructor = Parametrizer;

/**
 * Initialize this Parametrizer-object 
 * (see constructor {@link Parametrizer}).
 */
Parametrizer.prototype.initParametrizer = function () 
{
  this.net       = null;  // to be set by 'learn()'
  this.nodes     = null;  // to be set by 'learn()'
  this.samples   = null;  // to be set by 'learn()'
};

/**
 * Get the contingency table for the given node
 * containing the node and its parents (in the
 * same order as in the CPT of that node).
 * @param node  node to get the contingency table for
 */
Parametrizer.prototype.getCET = function (node)
{
  return (this.cets[node.id]);
};

/**
 * Start algorithm: set new probabilities for the 
 * CPTs of the given nodes of the bayesian network.
 * @param net      bayesian network 
 * @param nodes    array of nodes which should be learned
 * @param samples  data (see 'Samples'), complete data(!)
 *                 (i.e. 'samples.data[r][c]' >= 0 for all r, c)
 */
Parametrizer.prototype.learn = function (net, nodes, samples)
{
  // set remaining attributes
  this.net     = net;
  this.nodes   = nodes;
  this.samples = samples;

  // prepare (create cumulation tables)
  this.prepare();

  // set new probabilities (this just means counting)
  this.count();
};

/**
 * Craete tables for cumulated expectations.<br>
 * Notice: It is crucial that the order of dims in each
 *         CET is the same as in the corresponding CPT
 *         because the values are copied using method
 *         'PTable.copy()'!
 */
Parametrizer.prototype.prepare = function ()
{
  this.cets = {};
  for (var i = 0; i < this.nodes.length; i++) {
    var node = this.nodes[i];
    this.cets[node.id] = new Float32Table(node.cpt.dims);
  }
};

/**
 * Set new probabilities (this just means counting, how
 * many pieces of data are falling into each element of
 * a CPT/CET to be learned).
 */
Parametrizer.prototype.count = function ()
{
  // over all nodes to be learned
  for (var nId in this.cets) {
    var node = this.net.nodes[nId];
    var cet  = this.cets[nId];

    // fill CET from data 
    cet.init(0);
    for (var r = 0; r < this.samples.data.length; r++) {
      var row = this.samples.data[r];

      // get index into the CET for current 'row'
      var i = 0;
      for (var d = 0; d < cet.dims.length; d++) {
        var dim = cet.dims[d];
        var c   = this.samples.cols[dim.id]; // column-index
        var s   = row[c];                    // state-index
        if (s < 0) {
          throw ("This parametrizer cannot handle incomplete data!");
        }
        i += cet.steps[dim.id] * s;
      }

      // add this weighted piece of data ('row')
      cet.values[i] += row.weight;
    }

    // copy CET to CPT and re-normalize CPT
    node.cpt.copy(cet);
    node.cpt.cnorm();
  }
};


// EXPECTATION-MAXIMIZATION PARAMETRIZER ============================

/**
 * Constructor for a expectation maximization algorithm object
 * which estimates the conditional probabilities for all nodes
 * from (incomplete) data. For complete data one iteration is
 * enough (maxIter = 1). 
 * @param maxIter        max. number of iterations
 * @param maxDelta       max. delta of conditional probabilities allowed 
 * @param randomizeCPTs  should 'PTable.randomize()' be used for CPTs of 
 *                       nodes to be learned before learning starts? 
 *                       notice: these CPTs should have full degree of 
 *                       freedom
 * @param recreateProp   should 'net.prop.recreate()' be used to recreate 
 *                       the propagator before learning starts? (this is 
 *                       useful if this algorithm is used by a structurizer, 
 *                       which changes the structure)
 * @param delay          delay in [ms] between two iterations;
 *                       if delay < 0 'learn()' works synchronously
 */
function EM (maxIter, maxDelta, randomizeCPTs, recreateProp, delay)
{
  if (maxIter != undefined) {
    this.initEM(maxIter, maxDelta, randomizeCPTs, recreateProp, delay);
  }
}

/* Inheritance */
// EM.prototype          = ;
// EM.prototype.parent   = ;
EM.prototype.constructor = EM;

/**
 * Initialize this EM-object 
 * (see constructor {@link EM}).
 */
EM.prototype.initEM = function (maxIter, maxDelta, randomizeCPTs, recreateProp, delay) 
{
  this.net       = null;  // to be set by 'learn()'
  this.nodes     = null;  // to be set by 'learn()'
  this.samples   = null;  // to be set by 'learn()'
  this.randomize = randomizeCPTs;
  this.recreate  = recreateProp;
  this.maxIter   = maxIter;
  this.maxDelta  = maxDelta;
  this.delay     = delay;
  this.listeners = {};
};

/**T
 * Add a new listener.
 * @param listener  listener with methods:<br> 
 *                  'start()'                     called before the algorithm starts<br>
 *                  'improve(maxDelta, avgDelta)' called after each maximization step<br>
 *                  'stop()'                      called just before finishing
 */
EM.prototype.addListener = function (listener)
{
  this.listeners[listener] = listener;
};

/**
 * Remove a listener.
 */
EM.prototype.rmListener = function (listener)
{
  delete (this.listeners[listener]);
};

/**
 * Get the contingency table for the given node
 * containing the node and its parents (in the
 * same order as in the CPT of that node).
 * @param node  node to get the contingency table for
 */
EM.prototype.getCET = function (node)
{
  return (this.cets[node.id]);
};

/**
 * Start algorithm: set new probabilities for the 
 * CPTs of the given nodes of the bayesian network.
 * @param net      bayesian network including a propagator 
 *                 ('net.prop') which supports 'getJPT(node)'
 * @param nodes    array of nodes which should be learned
 * @param samples  data (see {@link Samples}), some values might 
 *                 be missing (i.e. 'samples.data[r][c]' < 0 for 
 *                 some r, c)
 */
EM.prototype.learn = function (net, nodes, samples)
{
  // set remaining attributes
  this.net       = net;
  this.nodes     = nodes;
  this.samples   = samples;
  this.maxDeltas = new Array(); // max. abs. deltas for each iteration 
  this.avgDeltas = new Array(); // avg. abs. deltas for each iteration

  // prepare (create cumulation tables)
  this.prepare();

  // expectation-maximization
  this.iter = 0;
  if (this.maxIter > 0) {
    for (var l in this.listeners) this.listeners[l].start();
    if (this.delay >= 0) {
      var me = this;
      setTimeout(function(){ me.improve(); }, this.delay); // (wrapper needed)
    } else {
      while (this.improve());
    }
  }
};

/**
 * Craete tables for cumulated expectations and randomize
 * CPTs of the nodes which should be learned and finally
 * fully recreate the propagator.<br>
 * Notice: It is crucial that the order of dims in each
 *         CET is the same as in the corresponding CPT
 *         because the values are copied using method
 *         'PTable.copy()'!
 */
EM.prototype.prepare = function ()
{
  // create cumulation tables for nodes to be learned
  this.cets = {};
  for (var i = 0; i < this.nodes.length; i++) {
    var node = this.nodes[i];
    this.cets[node.id] = new Float32Table(node.cpt.dims);
  }

  // randomize CPTs of nodes to be learned
  if (this.randomize) {
    for (var i = 0; i < this.nodes.length; i++) {
      var node = this.nodes[i];
      node.cpt.randomize();
      node.cpt.cnorm();
    }
  }

  // fully recreate the propagator or reload it
  if (this.recreate) {
    this.net.prop.recreate();
  } else if (this.randomize) {
    this.net.prop.reload();
  } // (otherwise the propagator is assumed to be up-to-date)
};

/**
 * Iteratively learn the CPTs of the nodes given by 
 * 'this.nodes' as long as the minimum change of any
 * single probability is > 'maxDelta' or the maximum
 * number of iterations ('maxIter') is reached.
 * @returns  true if improvement is not yet finished, false 
 *           otherwise (important for async. mode only, i.e.
 *           'delay' >= 0)
 */
EM.prototype.improve = function ()
{
  // expectation maximization
  this.initialization();
  this.expectation();
  var maxDelta = this.maximization();

  // go on or stop  
  this.iter++;
  if (maxDelta > this.maxDelta && this.iter < this.maxIter) {
    if (this.delay >= 0) {
      var me = this;
      setTimeout(function(){ me.improve(); }, this.delay); // (wrapper needed)
    }
    return (true);
  } else {
    this.net.rmEvidence();
    for (var l in this.listeners) this.listeners[l].stop();
    return (false);
  }
};

/**
 * Initialization for next expectation step: set all 
 * values in all cumulated expectation tables to 0.
 */
EM.prototype.initialization = function ()
{
  for (var nId in this.cets) {
    this.cets[nId].init(0);
  }
};

/**
 * Expectation: propagate all samples and cumulate results
 */
EM.prototype.expectation = function ()
{
  // over all samples
  for (var r = 0; r < this.samples.data.length; r++) {
    var row = this.samples.data[r];

    // set evidence for all(!) nodes
    for (var nId in this.net.nodes) {
      var node = this.net.nodes[nId];
      var c    = this.samples.cols[nId];
      if (c != undefined && row[c] >= 0) {
        node.f.init(0);
        node.f.values[row[c]] = 1; // row[c] = state
      } else {
        node.f.init(1);
      }
    }

    // propagate and cumulate results
    this.net.prop.propagate(true, false);  // (no need to update posteriors)
    for (var nId in this.cets) {
      var node = this.net.nodes[nId];
      var cet  = this.cets[nId];
      var jpt  = this.net.prop.getJPT(node);
      var w    = row.weight;
      cet.cumulate(jpt, w);
    }
  }
};

/**
 * Maximization: normalize cumulated results, set 
 * them as the new conditional probability tables
 * and update propagator ('prop.reload()').
 * @returns  max. abs. delta of CPT-values
 */
EM.prototype.maximization = function ()
{
  var maxDelta = 0;
  var avgDelta = 0, n = 0;
  for (var nId in this.cets) {
    var node = this.net.nodes[nId];
    var cet  = this.cets[nId];
    var cpt = node.cpt;

    // set new values (but keep old ones to calculate delta)
    var values = cpt.values;
    cpt.values = cet.values.slice();
    cpt.cnorm();

    // calculate max absolute delta
    for (var i = 0; i < values.length; i++) {
      var delta = Math.abs(values[i] - cpt.values[i]);
      maxDelta  = Math.max(maxDelta, delta);
      avgDelta += delta;
      n++;
    }
  }
  avgDelta /= n;

  // update the propagator (some CPTs have changed)
  this.net.prop.reload(); 

  // call listeners
  for (var l in this.listeners) this.listeners[l].improve(maxDelta, avgDelta);

  // store and return max. abs. delta for this maximization step
  this.maxDeltas.push(maxDelta);
  this.avgDeltas.push(avgDelta);
  return (maxDelta);
};


// CRITERIA FOR STRUCTURE LEARNING ==================================
// P(Model|Data) (+Reference-Network) -------------------------------

/**
 * log(P(Data|Structure)) using a reference net and a so-called
 * "user-sample-size" to weight the user-knowledge (alpha-values).
 * In this form it's usable for small networks only, since a 
 * 'FullJoint' propagator is build and used for every call to 
 * 'rate()' (notice: the 'FullJoint' propagator is the only one 
 * which is able to give us a probability table which contains 
 * each sub-set of nodes).
 * @param net  reference network describing the user knowledge
 * @param uss  user sample size (to weight the user knowledge)
 */
function USS (net, uss)
{
  this.name = 'P(M|D)';
  this.net  = net;
  this.uss  = uss;
}

/* Inheritance */
// USS.prototype          = ;
// USS.prototype.parent   = ;
USS.prototype.constructor = USS;

/**
 * Rate of one node defined by its contingency table N.
 * @param N  contingency table of one node and its parents
 */
USS.prototype.rate = function (N) 
{
  // get the user knowledge (really inefficient)
  var prop = new FullJoint(this.net);
  prop.propagate(false, false);
  var A = new Float32Table(N.dims);
  A.marg(prop.fjpt);

  // calculate rate (notice: tables N and A
  // have the same size and dimension-order)
  var dim  = N.dims[0];
  var s    = N.sizes[dim.id];
  var rate = 0;
  for (var j = 0; j < N.n; j += s) {
    var Aij = 0; // cumulated value: sum_k A_ijk
    var Nij = 0; // cumulated value: sum_k N_ijk
    for (var k = 0; k < s; k++) {
      var Nijk = N.values[j+k];
      var Aijk = A.values[j+k] * this.uss;
      Nij  += Nijk;
      Aij  += Aijk;
      rate += lgamma(Aijk + Nijk) - lgamma(Aijk);
    }
    rate += lgamma(Aij) - lgamma(Aij + Nij);
  }
  return (rate);
};

// P(Model|Data) Alpha=1 --------------------------------------------

/**
 * log(P(Data|Structure)) with: Alpha_ijk = 1
 */
function Alpha1 () 
{
  this.name    = 'P(M|D)';
  this.lgamma1 = lgamma(1);
}

/* Inheritance */
// Alpha1.prototype          = ;
// Alpha1.prototype.parent   = ;
Alpha1.prototype.constructor = Alpha1;

/**
 * Rate of one node defined by its contingency table N
 * @param N  contingency table of one node and its parents
 */
Alpha1.prototype.rate = function (N) 
{
  var dim  = N.dims[0];
  var s    = N.sizes[dim.id];
  var rate = 0;
  for (var j = 0; j < N.n; j += s) {
    var Nij = 0;
    for (var k = 0; k < s; k++) {
      var Nijk = N.values[j+k];
      Nij  += Nijk;
      rate += lgamma(1 + Nijk) - this.lgamma1;
    }
    rate += lgamma(s) - lgamma(s + Nij);
  }
  return (rate);
};

// P(Model|Data) Alpha_ijk = N_ijk ----------------------------------

/**
 * log(P(Data|Structure)) with:<br>
 * alpha_ijk =  beta    * N_ijk + 'minAlpha' and<br> 
 * N'_ijk    = (beta-1) * N_ijk
 * @param beta      weighting factor
 * @param minAlpha  minimum alpha (see above)
 * @param reRate    true if the rating should be recalculated after 
 *                  each change with the same contingency tables for 
 *                  "prior" and "current" structure as it was done 
 *                  for the initial rating; setting this to true is 
 *                  strongly recommended
 */
function AlphaBeta (beta, minAlpha, reRate)
{
  this.name     = 'P(M|D)';
  this.beta     = beta;
  this.minAlpha = minAlpha;
  this.reRate   = reRate; // re-rate after change
}

/* Inheritance */
// AlphaBeta.prototype          = ;
// AlphaBeta.prototype.parent   = ;
AlphaBeta.prototype.constructor = AlphaBeta;

/**
 * Rate of one node defined by its contingency table N
 * @param N    contingency table of one node and its parents
 * @param A    contingency table of the same node and its 
 *             former parents
 * @param add  added dimension
 * @param del  removed dimension
 */
AlphaBeta.prototype.rate = function (N, A, add, del) 
{
  return (this.merge(N, A, add, del, N.dims.length-1, 0, 0));
};

/**
 * Rate of one node defined by its contingency table N. This is
 * more complicated than for the other criteria because we use
 * the corresponding contingency table of the unchanged network 
 * as well, which has one dimension more or less (i.e. the array
 * of values in table A is in a different order than in table N).
 * @param N    contingency table of one node and its parents
 * @param A    contingency table of one node and its former parents
 * @param add  added dimension (i.e. a dimension in N which is not
 *             yet present in A); null if there was no dim. added
 * @param del  removed dimension (i.e. a dimension in A which isn't 
 *             present in N anymore); null if no dim. was removed
 * @param d    current dimension           (initially: N.dims.length-1)
 * @param jN   index into array 'N.values' (initially: 0)
 * @param jA   index into array 'A.values' (initially: 0)
 */
AlphaBeta.prototype.merge = function (N, A, add, del, d, jN, jA) 
{
  var rate = 0;
  var dim  = N.dims[d];
  var size = dim.states.length; 
  if (d < 1) {

    // first dimension (the dimension of the node to be rated)
    var Aij = 0; // cumulated value: sum_k A_ijk
    var Nij = 0; // cumulated value: sum_k N_ijk
    for (var k = 0; k < size; k++) {

      // get A_ijk and N_ijk
      var Aijk = this.minAlpha;  // avoid Nijk = Aijk = 0 ( => NaN)
      if (del) {                 // cumulate values over removed dim
        for (var s = 0, j = jA; s < del.states.length; s++) {
          Aijk += A.values[j+k]; 
          j    += A.step(del);
        }
      } else if (add) {          // spread values over added dim.
        Aijk += A.values[jA+k] / add.states.length;
      } else {                   // no dim. added/removed (initial rating)
        Aijk += A.values[jA+k];
      }
      Aijk *= this.beta;
      var Nijk = N.values[jN+k] * (1 - this.beta);

      // calculate rate (partly)
      rate += lgamma(Aijk + Nijk) - lgamma(Aijk);

      // cumulate
      Aij  += Aijk;
      Nij  += Nijk;
    }

    // calculate rate (remaining part)
    rate += lgamma(Aij) - lgamma(Aij + Nij);
    return (rate);

  } else {

    // parent dimension
    for (var k = 0; k < size; k++) { // over all states
      rate += this.merge(N, A, add, del, d-1, jN, jA);
      jA   += A.step(dim);
      jN   += N.step(dim);
    }

  }
  return (rate);
};

// BAYESIAN QUALITY -------------------------------------------------

/**
 * Bayesian quality
 */
function BQ () 
{
  this.name    = 'BQ';
}

/* Inheritance */
// BQ.prototype          = ;
// BQ.prototype.parent   = ;
BQ.prototype.constructor = BQ;

/**
 * Rate of one node defined by its contingency table N.
 * @param N  contingency table of one node and its parents
 */
BQ.prototype.rate = function (N) 
{
  var dim  = N.dims[0];
  var s    = N.sizes[dim.id];
  var rate = 0;
  for (var j = 0; j < N.n; j += s) {
    var Nij = 0;
    for (var k = 0; k < s; k++) {
      var Nijk = N.values[j+k];
      Nij  += Nijk;
      rate += lgamma(Nijk + 1);
    }
    rate += lgamma(s) - lgamma(s + Nij);
  }
  return (rate);
};

// BAYESIAN QUALITY -------------------------------------------------

/**
 * Bayesian quality
 */
function MDL () 
{
  this.name = 'MDL';
}

/* Inheritance */
// BQ.prototype          = ;
// BQ.prototype.parent   = ;
MDL.prototype.constructor = MDL;

/**
 * Rate of one node defined by its contingency table N.
 * @param N  contingency table of one node and its parents
 */
MDL.prototype.rate = function (N) 
{
  var n    = N.sum();         // (doing this every time, with the same result)
  var dim  = N.dims[0];
  var s    = N.sizes[dim.id]; // number states of the current node
  var q    = N.n / s;         // number of parent configurations
  var rate = 0;
  for (var j = 0; j < N.n; j += s) {
    var Nij = 0;
    for (var k = 0; k < s; k++) {
      var Nijk = N.values[j+k];
      Nij  += Nijk;
    }
    for (var k = 0; k < s; k++) {
      var Nijk = N.values[j+k];
      rate += Nijk * Math.log(Nijk/Nij) - q * (s-1) * 0.5 * Math.log(n);
    }
  }
  return (rate);
};


// GREEDY STRUCTURIZER ==============================================

/**
 * Constructs a structure learning algorithm, which always adds that 
 * link which improves the structure of the given bayesian network 
 * the most. This is done als long as the structure could be improved 
 * this way (or a maximum number of changes has been done).<br> 
 * <dl><dt>Notice:</dl>
 * <dd>In order to keep it simple we do not take other actions 
 *     (such as removing, reversing links) into consideration,
 *     because if we would do so, it would be more compliacted
 *     to distinguish allowed changes from those which are 
 *     permitted in a DAG (directed acyclic graph). In addition
 *     we would have to create ratings for combined actions for
 *     reversing links (remove one link + add the reverse link)</dd></dl>
 * @param criterion:    criterion (e.g. Alpha1, AlphaBeta, BQ, MDL, BIC)
 *                      the criterion has to be separabel and cumulative 
 *                      for all nodes and the better a structure is the 
 *                      higher has to be the rating (altough it could be 
 *                      negative)
 * @param parametrizer  creates/learnes the CPTs from data and gives
 *                      corresponding contingency tables (CETs); this
 *                      parametrizer has to work synchronously
 * @param maxIter:      maximum number of changes (if maxIter <= 0
 *                      only the initial rating is done, and no 
 *                      listeners are called)
 * @param fullReset:    forget ratings of all changes from previous 
 *                      itertion for each new iteration? this could be
 *                      useful if a 'EM' parameterizer is used which
 *                      uses the whole network (propagator) to estimate
 *                      the contingency tables (i.e. every change matters)
 * @param randomFactor  add an randomized offset to the delta-rating:<br>
 *                      delta' = delta + delta * f,<br>
 *                      where f is a randomized value in [0..randomFactor];
 * @param delay:        (min) delay in [ms] between two iterations;
 *                      if delay < 0 'learn()' works synchronously
 */
function Structurizer (criterion, parametrizer, maxIter, 
		       fullReset, randomFactor, delay)
{
  if (criterion != undefined) {
    this.initStructurizer(criterion, parametrizer, maxIter, 
			  fullReset, randomFactor, delay);
  }
}

/* Inheritance */
// Structurizer.prototype          = ;
// Structurizer.prototype.parent   = ;
Structurizer.prototype.constructor = Structurizer;

/**
 * Re-initializes this Structurizer 
 * (see constructor {@link Structurizer}).
 */
Structurizer.prototype.initStructurizer = function (criterion, parametrizer, maxIter, 
						    fullReset, randomFactor, delay)
{
  // attributes to be set later by 'learn()'
  this.net          = null; 
  this.nodes        = null; 
  this.samples      = null; 
  this.cets         = null;
  this.rates        = null;
  this.trials       = null;
  this.ordering     = null;
  this.clustering   = null;

  // attributes
  this.maxIter      = maxIter;
  this.fullReset    = fullReset;
  this.randomFactor = randomFactor;
  this.delay        = delay;
  this.criterion    = criterion;
  this.parametrizer = parametrizer;

  // listeners
  this.listeners    = {};
};

/**
 * Add a new listener.
 * @param listener  listener with methods:<br> 
 *                  'start()'          called before the algorithm starts<br>
 *                  'trial(trial)'     called after a new trial was rated<br>
 *                  'favor(trial)'     called if a trial is the best so far<br>
 *                  'skip(trial'       called if a trial is not the best<br>
 *                  'finalize(trial)'  called if a trial is finally done<br>
 *                  'stop()'           called just before finishing<br>
 */
Structurizer.prototype.addListener = function (listener)
{
  this.listeners[listener] = listener;
};

/**
 * Remove a listener.
 */
Structurizer.prototype.rmListener = function (listener)
{
  delete (this.listeners[listener]);
};

/**
 * Learning the structure of a bayesian network
 * @param net      bayesian network; depending on the
 *                 parametrizer this network needs or 
 *                 needs not to have a propagator attached
 * @param nodes    array of nodes which should be learned
 * @param samples  data (see 'Samples'); depending on the
 *                 paramtrizer values might or might not 
 *                 be missing (complete/incomplete data)
 */
Structurizer.prototype.learn = function (net, nodes, samples)
{
  // set remaining attributes
  this.net        = net;
  this.nodes      = nodes;
  this.samples    = samples;
  this.cets       = {};      // cets[nodeId]         = contingency table
  this.rates      = {};      // rates[nodeId]        = rating per node
  this.trials     = {};      // trials[nodeId][paId] = change-object
  this.ordering   = new NaturalOrdering(net);
  this.clustering = new Clustering(net);

  // improve
  this.initialize();
  this.iter = 0;
  if (this.maxIter > 0) {
    for (var l in this.listeners) this.listeners[l].start();
    if (this.delay >= 0) {
      var me = this;
      setTimeout(function(){ me.improve(); }, this.delay); // (wrapper needed)
    } else {
      while (this.improve());
    }
  }
};

/**
 * Iteratively improve the structure of the bayesian network
 * as long as the structure could be improved or the maximum
 * numbers of changes is reached.
 * @returns  true if improvement is not yet finished, false 
 *           otherwise (important for async. mode only, i.e.
 *           'delay' >= 0)
 */
Structurizer.prototype.improve = function ()
{
  var best = null;   // best link-change found
  
  // re-initialize
  if (this.iter > 0 && this.fullReset) {
    this.initialize();
  }
  
  // over all nodes to be learned
  for (var i = 0; i < this.nodes.length; i++) {
    var node = this.nodes[i];
    var nId = node.id;
    
    // try to add all possible parents
    for (var paId in this.net.nodes) {
      var parent = this.net.nodes[paId];
      if (this.isLinkable(parent, node)) {  // (no cycles)

        // if that change wasn't rated before do that now
        if (!this.trials[nId][paId]) { 
          this.trials[nId][paId] = this.trial(parent, node);
          for (var l in this.listeners) this.listeners[l].trial(this.trials[nId][paId]);
        }

        // store best trial
        var trial = this.trials[nId][paId];
        if (!best || trial.delta > best.delta) {
          best = trial;
          for (var l in this.listeners) this.listeners[l].favor(best);
        } else {
          for (var l in this.listeners) this.listeners[l].skip(trial);
        }
      }
    } 
  }  
  
  // perform best change
  if (best && best.delta > 0) {
    this.finalize(best);
    for (var l in this.listeners) this.listeners[l].finalize(best);
  } 

  // go on or stop  
  this.iter++;
  if (best && best.delta > 0 && this.iter < this.maxIter) {
    if (this.delay >= 0) {
      var me = this;
      setTimeout(function(){ me.improve(); }, this.delay); // (wrapper needed)
    }
    return (true);
  } else {
    if (this.net.prop) this.net.prop.recreate();
    for (var l in this.listeners) this.listeners[l].stop();
    return (false);
  }
};

/**
 * Create initial ratings for the current 
 * structure of the bayesian network.
 */
Structurizer.prototype.initialize = function ()
{
  // learn CPTs/CETs, store CETs, calculate and store 
  // initial ratings and initialize the "matrix" of trials
  this.parametrizer.learn(this.net, this.nodes, this.samples);
  for (var i in this.nodes) {
    var node = this.nodes[i];
    var cet0 = this.parametrizer.getCET(node);
    var rate = this.criterion.rate(cet0, cet0, null, null);
    this.cets[node.id]   = cet0;
    this.rates[node.id]  = rate;
    this.trials[node.id] = {};
  }
};

/**
 * Is it allowed to create a new link parent->node.
 * @returns  true if no such link exists and if adding such
 *           a link wouldn't violate the acyclic structure
 */
Structurizer.prototype.isLinkable = function (parent, node) 
{
  if (parent.id == node.id) {
    return (false);  // self-links are not allowed
  } else if (this.net.children[parent.id][node.id]) {
    return (false);  // link exists
  } else if (!this.clustering.isConnected(node, parent)) {
    return (true);   // different clusters are always linkable
  } else if (!this.ordering.isAncestor(node, parent)) {
    return (true);   // net will stay acyclic
  } else {
    return (false);  // link would create a cycle
  }
};

/**
 * Calculate the new rating for the given node 
 * as if a new link has been added between the 
 * two given nodes.
 * @param parent  parent node
 * @param node    child node
 * @returns       trial object describing that change
 */
Structurizer.prototype.trial = function (parent, node)
{
  // add link (this doesn't affect the CPT)
  this.net.addLink(parent, node); 

  // get CPT/CET of node 'node' for the unchanged structure
  var cpt0 = node.cpt;
  var cet0 = this.cets[node.id];
  
  // create CPT/CET of node 'node' for the changed structure
  var cpt1 = new Float32Table(this.net.getParents(node, true));
  node.cpt = cpt1;
  this.parametrizer.learn(this.net, new Array(node), this.samples);  
  var cet1  = this.parametrizer.getCET(node);
  
  // calculate rating of node 'node' in the changed structure
  var rate  = this.criterion.rate(cet1, cet0, parent, null);
  var delta = rate - this.rates[node.id];
  if (this.randomFactor) {
    delta += Math.random() * this.randomFactor * delta;
  }

  // craete a "trial-object"
  var trial = { parent : parent,  // a "trial-object" isn't
		node   : node,    // very meaningful outside
		rate   : rate,    // this algorithm, so we
		delta  : delta,   // create it "on the fly" 
		cpt    : cpt1,    // (without a constructor)
		cet    : cet1 };
  
  // restore CPT(!)
  node.cpt = cpt0;

  // undo structural changes
  this.net.rmLink(parent, node);

  return (trial);
};

/**
 * Finally add the link as described by the given 
 * trial-object ('trial.parent' -> 'trial.node').
 * @param trial  trial-object
 */
Structurizer.prototype.finalize = function (trial)
{
  this.net.addLink(trial.parent, trial.node); 
  trial.node.cpt             = trial.cpt;
  this.cets[trial.node.id]   = trial.cet;
  this.rates[trial.node.id]  = (this.criterion.reRate 
				? this.criterion.rate(trial.cet, trial.cet, null, null)
				: trial.rate);
  this.trials[trial.node.id] = {};  
  this.ordering.connect(trial.parent, trial.node);
  this.clustering.connect(trial.parent, trial.node);
};
