/*
 * -------------------------------------------------------------------------
 *
 *  pn4webJTree.js Copyright (C) 2011, 
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *   Author: Patrick Rammelt 
 *   E-Mail: patrick.rammelt@go4more.de
 *   Site:   http://www.patricks-seite.de
 * -------------------------------------------------------------------------
 *
 * Junction tree propagator for bayesian networks. This propagator could 
 * be used for more complex networks as the "full joint propagator" in 
 * "pn4webBasic.js". Nevertheless the main focus was to keep it as simple 
 * as possible rather than to get the highest possible performance (see 
 * some comments).
 * (These types are fully independent of display functionalities)
 */


// JUNCTION TREE PROPAGATOR =========================================
// CONSTRUCT / COMPILE ----------------------------------------------

/**
 * Constructs a junction tree, which is usable for far
 * more complex networks than the full joint propagator.
 * It's also usable for disconnected networks.
 * @param net     bayesian network to work on
 * @param restr   minimum allowed order for all or some nodes (restr[nodeId])
 * @param fillin  put fill-in links for moralization and triangulation 
 *                into the original bayesian network? [optional]
 * @param max     use max-propagation?
 * @param rnd     max. random offset added to the number of fill-ins 
 *                (introducing some indeterminedess to the tree-decomposition) 
 * @param preNet  network containing pre-defined fill-ins
 */
function JTree (net, restr, fillin, max, rnd, preNet)
{
  if (net != undefined) {
    this.initJTree(net, restr, fillin, max, rnd, preNet);
  }
}

/* Inheritance */
JTree.prototype             = new Net();
JTree.prototype.parent      = Net.prototype;
JTree.prototype.constructor = JTree;

/**
 * Compile / Recompile the given bayesian network 
 * (see constructor {@link JTree}).
 */
JTree.prototype.initJTree = function (net, restr, fillin, max, rnd, preNet) 
{
  this.initNet(net.id + '.jtree', 'Junction-tree for "' + net.name + '"');

  // set/init attributes
  this.net       = net;         // bayesian network
  this.max       = max;         // max-propagation?
  this.rnd       = rnd;         // max. random offset to the number of fill-ins
  this.fillin    = fillin;      // put fill-in-links (including moral links) into 'net'?
  this.fillins   = 0;           // number of fill-in links added (without moralization links)
  this.morals    = 0;           // number of links added for moralization
  this.pe        = 1.0;         // probability of evidence / of most probable config (max-prop)
  this.roots     = {};          // map: clique-id to root cliques (to be set later)
  this.cliques   = new Array(); // array of cliques (in the order of elimination!)
  this.order     = {};          // map: node-id to elimination order (index into cliques)
  this.home      = {};          // map: node-id to home clique
  this.restr     = restr;       // map: node-id to min allowed elim. order
  this.initData  = new Data();  // filled by absorp()
  this.preNet    = preNet;      // contains pre-defined fill-ins (undirected links)
  this.filledNet = null;        // finally this network contains all undir.+moral+fill-in links
  
  // create junction tree
  var moralNet = this.moralize(net, preNet); // create "moralized" network 
  this.filledNet = this.copy(moralNet);      // copy moralized network
  this.eliminate(moralNet);                  // create cliques by node elimination (moralNet is cleared)
  this.build();                              // connect cliques (build tree structure)
  this.aggregate();                          // aggregate cliques that are subsets of other cliques
  this.settle();                             // set home cliques for all nodes
  this.absorp();                             // absorp probabilities from bayesian network (init)

  // notice: the 'JTree' itself does not need 'this.cliques' and 'this.order' anymore; 
  //         'this.fillins' and 'this.filled' are just created/filled although useless
  //         for 'JTree' itself; nevertheless these attributes might be useful for other
  //         classes...
};

/**
 * Re-load probabilities from the bayesian network
 * (probabilities of some nodes might have changed
 * or the number of states of one dimensions might 
 * have changed).
 */
JTree.prototype.reload = function () 
{
  for (var cId in this.nodes) {
    var clique = this.nodes[cId];
    clique.jpt.initPTable(clique.jpt.dims); // number of states might have changed
    for (var chId in this.children[cId]) {
      var sep = this.children[cId][chId];
      sep.jpt.initPTable(sep.jpt.dims);     // number of states might have changed
    }
  }
  this.absorp();
};

/**
 * Fully recreate this propagator (includes 
 * layouting (if 'layout()' has been called).
 */
JTree.prototype.recreate = function () 
{
  this.initJTree(this.net, this.restr, this.fillin);
  if (this.w || this.h) {
    this.layout(this.w, this.h, this.rot90);
  }
};

/**
 * Create "moralized" network (replace links by undirected 
 * links and connect parents of common children by additional 
 * udirected links).
 * @param net     bayesian network ({@link BayesNet})
 * @param preNet  a network with pre-defined fill-ins
 * @param return  moralized graph
 */
JTree.prototype.moralize = function (net, preNet) 
{
  var moralNet = new Net('moralNet', 'moralized network');

  // moralized network has the very same nodes as the original net
  for (var nId in net.nodes) {
    moralNet.addNode(net.nodes[nId]); 
  }

  // copy all undirected links (i.e. pre-defined fill-ins)
  if (preNet) {
    for (var nId1 in preNet.siblings) {
      for (var nId2 in preNet.siblings[nId1]) {
        if (!moralNet.siblings[nId1][nId2]) {
          moralNet.addRelation(preNet.nodes[nId1], preNet.nodes[nId2], 
                               preNet.siblings[nId1][nId2]);
          this.fillins++;
        }
      }
    }
  }

  // copy all directed links as undirected links
  for (var paId in net.children) {
    for (var chId in net.children[paId]) {
      moralNet.addRelation(net.nodes[paId], net.nodes[chId], "undir");
    }
  }
  
  // moralization: for each parent node, search 
  // for other parent nodes of its child nodes and
  // connect that parent with each other parent
  for (var pa1Id in net.children) {
    for (var chId in net.children[pa1Id]) {
      for (var pa2Id in net.parents[chId]) {
        if (pa2Id != pa1Id                   &&
            !moralNet.siblings[pa1Id][pa2Id] &&
            !net.parents[pa1Id][pa2Id]       &&
            !net.children[pa1Id][pa2Id]) 
        {
          var pa1 = net.nodes[pa1Id];
          var pa2 = net.nodes[pa2Id];
          this.morals++;
          moralNet.addRelation(pa1, pa2, "moral");
          if (this.fillin) {
            this.net.addRelation(pa1, pa2); 
          }
        }
      }

    }
  }
  return (moralNet);
};

/**
 * Create a copy of the given undirected (moralized) network.
 * @param net     moralized network
 * @param return  copyied network
 */
JTree.prototype.copy = function (net) 
{
  var copy = new Net('undirNet', 'undirected network');

  // copied network has the very same nodes as the original net
  for (var nId in net.nodes) {
    copy.addNode(net.nodes[nId]); 
  }

  // copy all undirected links (relations)
  for (var nId1 in net.siblings) {
    for (var nId2 in net.siblings[nId1]) {
      if (!copy.siblings[nId1][nId2]) {    // don't do it twice for each relation
        copy.addRelation(net.nodes[nId1], net.nodes[nId2], net.siblings[nId1][nId2]);
      }
    }
  }
  return (copy);
};

/**
 * Eliminate all nodes from the network and create cliques
 * attributes 'this.order' and 'this.cliques' are filled.
 * cliques are also added to this junction tree as nodes.
 * @param moralNet  moralized network (to be "eliminated")
 */
JTree.prototype.eliminate = function (moralNet)
{
  // eliminate until all nodes are eliminated
  var eId;
  while (eId = this.elim(moralNet)) {
    var node = moralNet.nodes[eId]; 

    // build the cluster of this node and all siblings
    // (connect all yet unconnected pairs of siblings)
    var nodes = this.clusterize(node, moralNet);

    // create new clique, add it to this junction tree (also add 
    // it to this.cliques and store order for node eId in this.order)
    var clique = this.createClique(node, nodes);
    this.addNode(clique); 
    this.order[eId] = this.cliques.length;
    this.cliques[this.cliques.length] = clique;

    // finally eliminate node
    moralNet.rmNode(node);
  }
};

/**
 * Create a new clique containing the given nodes.
 * @param node   elimination node of the new clique
 * @param nodes  mapping node Id to node, for all nodes to 
 *               be contained by the new clique
 * @returns      a new clique containing all given nodes
 *               having the same id as the given elim. node
 */
JTree.prototype.createClique = function (node, nodes)
{
  return (new Clique(node.id, 'Clique ' + this.cliques.length, 
                     nodes, this.cliques.length));
};

/**
 * Create a new separator connecting the given cliques.
 * @param clique1  parent clique
 * @param clique2  child clique
 * @returns        a new separator to connect the given cliques
 */
JTree.prototype.createSeparator = function (clique1, clique2)
{
  return (new Separator(clique1, clique2));
};

/**
 * Find next node to be eliminated from a moralized network,
 * which is the node for which the least fill-ins are 
 * needed (this is just one of several possible criteria).
 * Restritions on elimination order (this.restr) will be
 * respected.
 * @param moralNet  moralized network
 * @returns         id of the node to be eliminated next 
 *                  (null if there are no more nodes left)
 */
JTree.prototype.elim = function (moralNet) 
{
  var nextId     = null;
  var minFillIns = -1; 
  for (var nId in moralNet.nodes) {
    if (this.restr[nId] == undefined ||           // allowed to...
        this.restr[nId] <= this.cliques.length) { // ...eliminate?

      // over all pairs of neighbor nodes: count missing links
      var fillIns = (this.rnd ? Math.random() * this.rnd : 0); // random offset
      for (var n1Id in moralNet.siblings[nId]) {
        for (var n2Id in moralNet.siblings[nId]) {
          if (n1Id != n2Id && !moralNet.siblings[n1Id][n2Id]) {
            fillIns++;
          }
        }
      }

      // remember best node and min number of fill-ins
      if (!nextId || fillIns < minFillIns) {
        nextId     = nId;
        minFillIns = fillIns;
      }

    }
  }
  return (nextId);
};

/**
 * Get the cluster of this node and all its siblings,
 * and add fill-in links such that this cluster is fully
 * connected afterwards.
 * @param node      node (the node to be eliminated)
 * @param moralNet  moralized network
 * @param return    map: node-id to node for all nodes in 
 *                  that cluster
 */
JTree.prototype.clusterize = function (node, moralNet)
{
  var nodes = {};
  nodes[node.id] = node;
  for (var n1Id in moralNet.siblings[node.id]) {
    var neighbor1 = moralNet.nodes[n1Id];
    nodes[n1Id] = neighbor1;           
    for (var n2Id in moralNet.siblings[node.id]) {
      var neighbor2 = moralNet.nodes[n2Id];
      if (n1Id != n2Id && !moralNet.siblings[n1Id][n2Id]) {
        this.fillins++;
        moralNet.addRelation(neighbor1, neighbor2, "fillin");
        this.filledNet.addRelation(neighbor1, neighbor2, "fillin"); // also add link here!
        if (this.fillin) {
          this.net.addRelation(neighbor1, neighbor2);
        }
      }
    }
  }
  return (nodes);
};

/**
 * Connect cliques to a junction tree
 * (algorithm described by Cowell).
 */
JTree.prototype.build = function ()
{
  // over all cliques in reverse elim-order
  for (var order1 = this.cliques.length-1; order1 >= 0; order1--) { 
    var clique1 = this.cliques[order1];

    // find the clique which has a non-empty intersection 
    // with the given clique and has been created first
    var order2 = this.cliques.length;    // minimum order
    for (var nId in clique1.nodes) {      
      var o = this.order[nId];
      if (o < order2 && o != order1) { 
        order2 = this.order[nId];
      }
    }
    var clique2 = this.cliques[order2];

    // connect clique2 -> clique1 and set 
    // clique2 as the new root of this tree
    if (clique2) {
      var sep = this.createSeparator(clique2, clique1);
      this.addLink(clique2, clique1, sep);
    } else {
      this.roots[clique1.id] = clique1; // root: clique has no parent
    }
  }
};

/**
 * Remove cliques which are subsets of other cliques, i.e.
 * make it a real junction tree (algorithm described by Cowell)
 */
JTree.prototype.aggregate = function ()
{
  // over all cliques in reverse elim-order
  for (var order1 = this.cliques.length-1; order1 >= 0; order1--) { 
     var clique1  = this.cliques[order1];

     // find latest child clique which contains all nodes of clique1
     var order2 = -1;                               // maximum order
     for (var chId in this.children[clique1.id]) {
       var clique2 = this.nodes[chId];
       if (clique2.order > order2 && this.isSubstitutable(clique1, clique2)) {
         order2 = clique2.order;
       }
     }

     // replace clique1 by the super-child-clique (if there is one)
     var clique2 = this.cliques[order2];
     if (clique2) {
       this.substitute(clique1, clique2);
     }
  }
};

/**
 * Checks whether 'clique1' could be substituted by 'clique2'
 * (i.e. 'clique2' has to contain all nodes of 'clique1').
 * @param clique1  clique which should be substituted
 * @param clique2  clique to substitute clique1
 * @returns        true iff clique1 could be substituted by clique2
 */
JTree.prototype.isSubstitutable = function (clique1, clique2)
{
  return (clique2.containsAll(clique1));
};

/**
 * Replace clique1 by its child clique2 (aggregation).
 * @param clique1  clique which should be removed
 * @param clique2  clique replacing clique1 
 *                 (has to be a child of clique1)
 */
JTree.prototype.substitute = function (clique1, clique2)
{
  // make parents of clique1 parents of clique2
  for (var paId in this.parents[clique1.id]) {
    var sep = this.createSeparator(this.nodes[paId], clique2);
    this.addLink(this.nodes[paId], clique2, sep);
  }

  // make (other) children of clique1 children of clique2
  for (var chId in this.children[clique1.id]) {
    if (chId != clique2.id) {
      var sep = this.createSeparator(clique2, this.nodes[chId]);
      this.addLink(clique2, this.nodes[chId], sep);
    }
  }

  // replace roots
  if (this.roots[clique1.id]) {
    delete (this.roots[clique1.id]);
    this.roots[clique2.id] = clique2;
  }

  // remove clique1
  this.rmNode(clique1);
};

/**
 * Set home cliques for all nodes. A home is a clique containing
 * the node itself and all its parents. Since this is better for
 * some extended classes (hybrid junction trees) we're using the
 * clique that was created latest.
 */
JTree.prototype.settle = function ()
{
  for (var o = this.cliques.length - 1; o >= 0; o--) {
    var clique = this.cliques[o];
    if (this.nodes[clique.id]) {        // only non-aggregated cliques
      for (var nId in clique.nodes) {
        if (!this.home[nId]) {          // no home set for that node
          this.home[nId] = clique;
          for (var paId in this.net.parents[nId]) {
            if (!clique.nodes[paId]) {  // clique is not a home of that node
              delete (this.home[nId]);  
              break; 
            }
          }
        }
      }
    }
  }
};

/**
 * "Absorps" CPT's from all nodes of the bayesian network and
 * create data for re-initializing this junction-tree from a 
 * new set of observations. All elements of all clique JPT's
 * have to be initialized with 1 to make this work correctly!
 */
JTree.prototype.absorp = function ()
{
  // initialize joint probability tables of all cliques
  // (multiply cpt of each node into its home cliques jpt)
  for (var nId in this.home) {
    var node   = this.net.nodes[nId];
    var clique = this.home[nId];
    clique.jpt.mult(node.cpt);
  }

  // store initialized joint probability tables of all cliques
  // (separator tables are reinitialized by setting all values to 1)
  for (var cId in this.nodes) {
    var clique = this.nodes[cId];
    this.initData.store(clique.id, clique.jpt.values);
  }
};

// GETTER / SETTER --------------------------------------------------

/**
 * Return a joint probability table containing
 * at least the given node and all its parents.
 * @param node  node in the bayesian network ('this.net')
 * @returns     joint probability table P(node,pa(node),...)
 */
JTree.prototype.getJPT = function (node) 
{
  return (this.home[node.id].jpt);
};

// PROPAGATE --------------------------------------------------------

/**
 * Propagate (and update a-posteriori probability of all nodes).
 * The HUGIN propagation scheme is used, which first collects
 * information from the leafs to the root(s) and then distributes
 * information from the root(s) back to the leafs. After that the
 * junction tree is in equilibrium state (i.e. consistent)
 * @param observe  use current evidence?
 * @param update   update posterior probability tables of all nodes?
 * @returns        probability of evidence given the model: P(e|M)
 */ 
JTree.prototype.propagate = function (observe, update) 
{
  // re-initialize and include evidence
  this.init();
  if (observe) this.observe();

  // propagate (HUGIN scheme)
  var pe = 1.0;
  for (var rId in this.roots) {                // over all disconnected sub-trees
    this.collect(this.roots[rId]);             // collect phase
    pe *= (this.max                            // normalize root which gives...
	   ? this.roots[rId].jpt.mnorm()       // ...P(C|e) or
	   : this.roots[rId].jpt.norm());      // ...P(e|M)
    this.distribute(this.roots[rId]);          // distribute phase
  }

  // update a-posteriori probabilities of all nodes
  if (update) this.update();

  this.pe = pe;
  return (this.pe);
};

/**
 * Re-initialize probability tables of all cliques and separators.
 */
JTree.prototype.init = function ()
{
  for (var cId in this.nodes) {
    var clique = this.nodes[cId];
    clique.jpt.values = this.initData.stored(clique.id).slice();
    for (var chId in this.children[cId]) {
      var sep = this.children[cId][chId];
      sep.jpt.init(1);
    }
  }
};

/**
 * Multiply in all finding vectors (evidence) of all nodes
 * into the joint probability table of their home clique.
 */
JTree.prototype.observe = function ()
{
  for (var nId in this.home) {
    var node   = this.net.nodes[nId];
    var clique = this.home[nId];
    clique.jpt.mult(node.f);
  }
};

/**
 * Collect phase of HUGIN propagation scheme (phase 1)
 * clique: root of current sub-tree (start with one clique
 *         in this.roots)
 */
JTree.prototype.collect = function (clique)
{
  for (var chId in this.children[clique.id]) {
    var child = this.nodes[chId];
    var sep   = this.children[clique.id][chId];
    this.collect(child);
    this.send(child, clique, sep);
  }
};

/**
 * Distribute phase of HUGIN propagation scheme (phase 2)
 * clique: root of current sub-tree (start with one clique
 *         in this.roots)
 */
JTree.prototype.distribute = function (clique)
{
  for (var chId in this.children[clique.id]) {
    var child = this.nodes[chId];
    var sep   = this.children[clique.id][chId];
    this.send(clique, child, sep);
    this.distribute(child);
  }
};

/**
 * Send message from one clique to another clique.
 * via the given separator.<br>
 * Notice: simple but rather inefficient implementation
 *         because c * (s'/s) would need less operations
 *         than (c/s) * s', because c is bigger than s
 *         (where c = clique table, s = separator table)
 * @param from  sending clique
 * @param to    reveiving clique
 * @param sep   separator
 */
JTree.prototype.send = function (from, to, sep)
{
  to.jpt.div(sep.jpt);         // divide by old separator (s)
  (this.max 
   ? sep.jpt.mmarg(from.jpt)
   : sep.jpt.marg(from.jpt));  // create new separator (s')
  to.jpt.mult(sep.jpt);        // multiplicate by new separator
};

/**
 * Update a-posteriori probabilities of all nodes.
 */ 
JTree.prototype.update = function () 
{
  for (var nId in this.net.nodes) {
    var node = this.net.nodes[nId];
    (this.max 
     ? node.p.mmarg(this.home[node.id].jpt)
     : node.p.marg(this.home[node.id].jpt)); 
  }
}; 

// LAYOUT -----------------------------------------------------------

/**
 * Set clique coordinates such that the given area is fully used
 * @param width   available width (max x-coordinate)
 * @param height  available height (max y-coordinate)
 * @param rot90   rotate 90 degrees (reverse x and y)
 */
JTree.prototype.layout = function (width, height, rot90)
{
  this.w     = width;
  this.h     = height;
  this.rot90 = rot90;
  var raster = new Array();
  for (var rId in this.roots) {
    this.arrange(this.roots[rId], 0, raster);
    var y  = (raster.length <= 1 ? height / 2 : 0);
    var dy = height / (raster.length-1);
    for (var r = 0; r < raster.length; r++) {
      var x = (raster[r].length <= 1 ? width / 2 : 0);
      var dx = width / (raster[r].length-1);
      for (var c = 0; c < raster[r].length; c++) {
        raster[r][c].x = (rot90 ? y : x);
        raster[r][c].y = (rot90 ? x : y);
        x += dx;
      }
      y += dy;
    }
  }
};

/**
 * Arrange cliques in a raster.
 * @param clique  current clique (initially one of 'this.roots')
 * @param r       current row index (initially 0)
 * @param raster  array of rows, where each row is an array of
 *                columns, where each cell contains a clique
 */
JTree.prototype.arrange = function (clique, r, raster)
{
  if (raster[r]) raster[r].push(clique);
  else           raster[r] = new Array(clique);
  for (var chId in this.children[clique.id]) {
    var child = this.nodes[chId];
    this.arrange(child, r+1, raster);
  }
};
