/*
 * -------------------------------------------------------------------------
 *
 *  pn4webDynamic.js Copyright (C) 2011, 
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *   Author: Patrick Rammelt 
 *   E-Mail: patrick.rammelt@go4more.de
 *   Site:   http://www.patricks-seite.de
 * -------------------------------------------------------------------------
 *
 * Propagators for hybrid bayesian networks, where currently only one 
 * type of distribution is supported, which is the conditional gaussian 
 * distribution (CG). A CG(head|tail1,tail2,...) basically is a gaussian 
 * distribution where the mean is given by a linear function of the "tail" 
 * dimensions of that CG (these types are fully independent of display 
 * functionalities).
 */


// CG-CLUSTERS (HYBRID CLIQUES) =====================================

/**
 * Constructs a cluster for a hybrid (strong) elimination tree 
 * ({@link HybridJTree}). This is similar to a {@link Clique} 
 * except that for a 'CGCluster' a table of conditional gaussians 
 * is stored instead of the joint probability table of normal cliques.
 * @param id          id
 * @param name        name
 * @param nodes       mapping nodes[nId] = node 
 * @param continuous  array of continous nodes
 * @param discrete    array of discrete nodes
 * @param order       elimination order
 * @param elim        eliminated node
 */
function CGCluster (id, name, nodes, continuous, discrete, order)
{
  if (id != undefined) {
    this.initCGCluster(id, name, nodes, continuous, discrete, order);
  }
}

/* Inheritance */
// CGCluster.prototype          = ;
// CGCluster.prototype.parent   = ;
CGCluster.prototype.constructor = CGCluster;

/**
 * Initialize this 'CGCluster' 
 * (see constructor {@link CGCluster}).
 */
CGCluster.prototype.initCGCluster = function (id, name, nodes, continuous, discrete, order)
{
  this.id    = id;    // every node in a net needs an unique id
  this.name  = name;  // name (just for visualization)
  this.order = order;
  this.fixed = false;
  this.x     = 0;     // x-coordinate (for 'HybridJTrees' a layouter is needed)
  this.y     = 0;     // y-coordinate (for 'HybridJTrees' a layouter is needed)
  this.n     = 0;     // number of nodes contained

  this.isContinuous = true;       // marker
  this.continuous   = continuous; // array of continuous nodes
  this.discrete     = discrete;   // array of discrete nodes
  this.nodes        = nodes;      // mapping nodes[id] = node for all nodes

  this.lpt  = null;         // table of LP-potentials (CG's)
  this.post = new Array();  // postbag
};

/**
 * Get the number of nodes contained in this cluster.
 */
CGCluster.prototype.size = function () 
{
  return (this.continuous.length + this.discrete.length);
};

/**
 * Does this cluster contain all nodes 
 * of the other given cluster/clique? 
 */
CGCluster.prototype.containsAll = function (other) 
{
  for (var id in other.nodes) {
    if (!this.nodes[id]) return (false);
  }
  return (true);
};


// JUNCTION TREE PROPAGATOR FOR HYBRID BAYESIAN NETWORKS ============

/**
 * (Strong) Elimination tree class for hybrid bayesian networks, i.e. 
 * networks containing discrete 'Nodes' and 'CGNodes' (other types of 
 * continuous nodes are not yet supported). The algorithm described 
 * by Cowell is implemented.<br>
 * <br>
 * <b>Drawbacks of this algorithm:</b><br>
 * <ul>
 * <li>limited to 'CGNodes' ("exchange" operation needed)</li>
 * <li>joint probabilities P(N,pa(N)) are not available</li>
 * <li>no max-propagation</li>
 * <li>high complexity for some (not unusal) structures<br>
 *   <b>Lemma:</b>
 *     For two discrete nodes A and B with continuous children C and 
 *     D there will be at least one (boundary) clique containing both 
 *     A and B if there is a path from C to D and this path includes
 *     at least one continuous node.</li>
 * </ul>
 * For purely discrete networks everything which is supported by 
 * normal junction-trees is also supported by this algorithm (e.g. 
 * joint probabilities for families of discrete nodes).
 *
 * @param net     bayesian network to work on
 * @param restr   minimum allowed order for all or some nodes 
 *                (restr[nodeId]) (either do not set restrictions for 
 *                discrete nodes, or make sure that they are eliminated 
 *                after all continuous nodes)
 * @param window  time window size (max number of steps for 
 *                back-propagation)
 * @param fillin  put fill-in links for moralization and triangulation 
 *                into the original bayesian network? [optional]
 */
function HybridJTree (net, restr, window, fillin)
{
  if (net != undefined) {
    this.initHybridJTree(net, restr, window, fillin);
  }
}

/* Inheritance */
HybridJTree.constructor      = HybridJTree;
HybridJTree.prototype        = new JTree();
HybridJTree.prototype.parent = JTree.prototype;

/**
 * Compile / Recompile the given bayesian network 
 * (see constructor {@link HybridJTree}).
 */
HybridJTree.prototype.initHybridJTree = function (net, restr, window, fillin) 
{
  // manipulate restrictions: all discrete nodes 
  // should be eliminated after all continuous nodes 
  var o = 0; 
  for (var nId in net.nodes) {
    if (!net.nodes[nId].states) o++;
  }
  for (var nId in net.nodes) {
    if (net.nodes[nId].states && restr[nId] == undefined) restr[nId] = o;
  }

  // init normal junction tree
  this.initJTree(net, restr, fillin);

  // create topological odering
  this.ordering = new NaturalOrdering(net);

  // reshape function for 'PTable.merge()' which 
  // copies all CG's in a CGT (in principle we 
  // could keep the first and copy all others)
  this.reshape = function (t1, t2, i1, i2) { t1.values[i1] = t2.values[i2].clone(); };
};

/**
 * Re-load probabilities from the bayesian network
 * (probabilities of some nodes might have changed
 * or the number of states of one dimensions might 
 * have changed).
 */
HybridJTree.prototype.reload = function ()
{
  for (var cId in this.nodes) {
    var clique = this.nodes[cId];
    if (clique.jpt) {
      clique.jpt.initPTable(clique.jpt.dims); // number of states might have changed
      for (var chId in this.children[cId]) {
        var sep = this.children[cId][chId];
        if (sep.jpt) {
          sep.jpt.initPTable(sep.jpt.dims);   // number of states might have changed
        }
      }
    }
  }
  this.absorp();
};

/**
 * Fully recreate this propagator (includes 
 * layouting (if 'layout()' has been called).
 */
HybridJTree.prototype.recreate = function () 
{
  this.initHybridJTree(this.net, this.restr, this.fillin);
  if (this.w || this.h) {
    this.layout(this.w, this.h, this.rot90);
  }
};

/**
 * Create a new clique/cluster containing the given nodes.
 * @param node   elimination node of the new clique
 * @param nodes  nodes to be contained by the new clique
 * @returns      a new clique containing all given nodes
 *               having the same id as the given elim. node
 */
HybridJTree.prototype.createClique = function (node, nodes)
{
  var sep = this.separate(nodes);
  if (sep.continuous.length) {
    return (new CGCluster(node.id, 'CG-Cluster ' + this.cliques.length, 
			  nodes, sep.continuous, sep.discrete,
			  this.cliques.length));
  } else {
    return (this.parent.createClique.call(this, node, nodes));
  }
};

/**
 * Create a new separator connecting the given cliques.
 * @param clique1  parent clique
 * @param clique2  child clique
 * @returns        a new separator to connect the given cliques
 */
HybridJTree.prototype.createSeparator = function (clique1, clique2)
{
  if (clique1.cpt && clique2.cpt) {
    return (new Separator(clique1, clique2));
  } else {
    return (true); // (just a normal link)
  }
};

/**
 * Separate continuous from discrete nodes.
 * @param nodes  nodes
 * @returns      'return.continuous' = array of continuous nodes<br>
 *               'return.discrete'   = array of discrete nodes
 */
HybridJTree.prototype.separate = function (nodes)
{
  var sep = { continuous : new Array(),
              discrete   : new Array() };
  for (var id in nodes) {
    (nodes[id].isContinuous 
        ? sep.continuous.push(nodes[id]) 
        : sep.discrete.push(nodes[id]));
  }
  return (sep);
};

/**
 * Checks whether 'clique1' could be substituted by 'clique2'
 * (i.e. both cliques have to be pureley discrete 'Cliques'
 *  and 'clique2' has to contain all nodes of 'clique1').
 * @param clique1  clique which should be substituted
 * @param clique2  clique to substitute clique1
 * @returns        true iff 'clique1' could be substituted by 
 *                 'clique2'
 */
JTree.prototype.isSubstitutable = function (clique1, clique2)
{
  return (clique2.jpt && clique1.jpt && clique2.containsAll(clique1));
};


/**
 * "Absorps" CPT's from all nodes of the bayesian network and
 * create data for re-initializing this junction-tree from a 
 * new set of observations.
 */
HybridJTree.prototype.absorp = function ()
{
  // initialize joint probability / CG-tables 
  // of all cliques / CG-clusters
  for (var nId in this.home) {
    var node   = this.net.nodes[nId];
    var clique = this.home[nId];
    if (node.cpt) clique.jpt.mult(node.cpt);
  }

  // store initialized joint probability tables of all cliques
  // (separator tables are reinitialized by setting all values to 1)
  for (var cId in this.nodes) {
    var clique = this.nodes[cId];
    if (clique.jpt) this.initData.store(clique.id, clique.jpt.values);
  }
};


// PROPAGATE --------------------------------------------------------

/**
 * Propagate (and update a-posteriori probability of all nodes).
 * @param observe  use current evidence?
 * @param update   update posterior probability tables of all nodes?
 * @return         probability of evidence given the model: P(e|M)
 */ 
HybridJTree.prototype.propagate = function (observe, update) 
{
  // re-initialize and include evidence
  this.init();

  // collect postbags (remove postbags)
  this.collectPostbags();

  // include evidence
  if (observe) this.observe();

  // propagate (HUGIN scheme)
  var pe = 1.0;
  for (var rId in this.roots) {                // over all disconnected sub-trees
    if (this.roots[rId].jpt) {
      this.collect(this.roots[rId]);           // collect phase
      pe *= (this.max                          // normalize root which gives...
             ? this.roots[rId].jpt.mnorm()     // ...P(C|e) or
             : this.roots[rId].jpt.norm());    // ...P(e|M)
      this.distribute(this.roots[rId]);        // distribute phase
    }
  }

  // update a-posteriori probabilities of all nodes
  if (update) this.update();

  this.pe = pe;
  return (this.pe);
};

/**
 * Re-initialize probability tables of all cliques and separators.
 */
HybridJTree.prototype.init = function ()
{
  // init all cliques
  for (var cId in this.nodes) {
    var clique = this.nodes[cId];
    if (clique.jpt) {                          // discrete clique
      clique.jpt.values = this.initData.stored(clique.id).slice();
      for (var chId in this.children[cId]) {
        var sep = this.children[cId][chId];
        if (sep.jpt) sep.jpt.init(1);
      }
    } else {                                   // hybrid cluster
      clique.lpt   = null;
      clique.post  = new Array();
      clique.fixed = false;
    }
  }

  // create CGTs for all nodes and store them in their home-clusters
  for (var nId in this.home) {
    var node = this.net.nodes[nId];
    if (!node.cpt) {
      var cluster = this.home[nId];
      var cgt1 = node.getCGT();
      var cgt2 = new PTable(cluster.discrete, null);
      cgt2.merge(this.reshape, 0, 0, 0, cgt1); // reshape to the cluster size
      if (cluster.id == node.id) {             // node is elim.-node of cluster
        cluster.lpt = cgt2;  
      } else {
        cluster.post.push(cgt2);
      }
    }
  }
};

/**
 * Collect postbags, such that the list of 
 * postbags is empty for all clusters afterwards.
 */
HybridJTree.prototype.collectPostbags = function ()
{
  // over all clusters in order of elimination
  for (var i = 0; i < this.cliques.length; i++) {
    var cluster = this.cliques[i];
    if (cluster.jpt) break;                    // no more continuous clusters

    // sort postbag in reverse topological order 
    var comparator = function(a, b) { 
      return (ordering.level(a.values[0].head) - 
	      ordering.level(b.values[0].head)); 
    };
    cluster.post.sort(comparator);
    
    // remove all CGs in the postbag (starting at topmost nodes)
    while (cluster.post.length > 0) {
      var cgt = cluster.post.pop();
      var cg  = cgt.values[0];                 // just one of the CG's
      if (cg.fact[cluster.id] != undefined) {  // (cluster.id == elimNode.id)

        // reshape postbag cgt and exchange 
        // with current cluster's LP-potential
        var cgt2 = new PTable(cluster.lpt.dims, null);
        cgt2.merge(this.reshape, 0, 0, 0, cgt);
        for (var i = 0; i < cgt2.values.length; i++) {
          cgt2.values[i].exchange(cluster.lpt[i]);
        }

        // move the postbag to the parent cluster
        for (var paId in this.parents[cluster.id]) { 
          var parent = this.nodes[paId];
          if (parent.id == cg.head) {          // cg.head = elim.-node
            parent.lpt = cgt2;
          } else {
            parent.post.push(cgt2);
          }
          break; // (there is at most one parent)
        }

      }
    }

  }
};

/**
 * Multiply in all finding vectors (evidence) of all nodes
 * into the joint probability table of their home clique.
 */
HybridJTree.prototype.observe = function ()
{
  for (var nId in this.home) {
    var node = this.net.nodes[nId];
    if (node.cpt) {

      // standard discrete observation
      var clique = this.home[nId];
      clique.jpt.mult(node.f);

    } else if (node.v != null) { 

      // set observation into every node having this node
      // as a tail-factor (i.e. clusters created earlier)
      for (var o = this.order[nId] - 1; o >= 0; o--) {
        var cluster = this.cliques[o];
        for (var i = 0; i < cluster.lpt.values.length; i++) {
          cluster.lpt.values[i].setTail(node.id, node.v);
        }
      }
      
      // copy the LP-potential and "push" that copy 
      // towards a boundary cluster to send evidence
      // into the purely discrete part of this jtree
      var cluster = this.cliques[this.order[nId]];
      var cgt = new PTable(cluster.lpt.dims, null);
      for (var i = 0; i < cgt.values.length; i++) {
        cgt.values[i] = cluster.lpt.values[i].clone();
      }
      this.pushEvidence(cluster, cgt, node);

      // mark node to be observed
      cluster.fixed = true;

    }
  }
};

/**
 * Push evidence towards a boundary clique and multiply
 * in "dicretized" evidence into that boundary clique.
 * @param cluster  current cluster
 * @param cgt      CGT to be pushed (we do not use the postbag here)
 * @param node     observed node (node.v != null)
 */
HybridJTree.prototype.pushEvidence = function (cluster, cgt, node)
{
  for (var paId in this.parents[cluster.id]) {
    var parent = this.nodes[paId];
    if (parent.jpt) {

      // multiplicate values of the gaussian function into
      // the JPT of the boundary clique (no reshape needed)
      var setEvid = function (t1, t2, i1, i2) { 
        t1.values[i1] *= t2.values[i2].value(node.v); 
      };
      parent.jpt.merge(setEvid, 0, 0, 0, cgt);

    } else {

      // reshape postbag CGT, exchange with 
      // parent's LP-potential and set tail
      var cgt2 = new PTable(parent.lpt.dims, null);
      cgt2.merge(this.reshape, 0, 0, 0, cgt);
      if (!parent.fixed) {
        for (var i = 0; i < cgt2.values.length; i++) {
          cgt2.values[i].exchange(parent.lpt.values[i]);
          parent.lpt.values[i].setTail(node.id, node.v);
        }
      }

      // push further (until a boundary clique is reached)
      this.pushEvidence(parent, cgt2, node);

    }
    return; // (there is at most one parent)
  }
};

/**
 * Push posteriors towards a boundary clique and get
 * weights for unconditional CG's from that clique.
 * @param cluster  current cluster
 * @param cgt      CGT to be pushed (we do not use the postbag here)
 * @param node     node to create posteriors (node.p)
 */
HybridJTree.prototype.pushPosterior = function (cluster, cgt, node)
{
  for (var paId in this.parents[cluster.id]) { 
    var parent = this.nodes[paId];
    if (parent.jpt) {

      // cumulate/marginalize JPT-values as weights 
      // of the CG's (cumulation avoids reshaping)
      var setWeight = function (t1, t2, i1, i2) { 
        t2.values[i2].wght += t1.values[i1]; 
      };
      for (var i = 0; i < cgt.values.length; i++) {
        cgt.values[i].wght = 0; // init
      }
      parent.jpt.merge(setWeight, 0, 0, 0, cgt);        // cumulate
      node.p = cgt.values;                              // set posterior

    } else {

      // reshape CGT and exchange with the LP-potential of
      // the parent cluster (LP-potential stays unchanged)
      var cgt2 = new PTable(parent.lpt.dims, null);
      cgt2.merge(this.reshape,  0, 0, 0, cgt);
      if (!parent.fixed) {
        for (var i = 0; i < cgt2.values.length; i++) {
          cgt2.values[i].exchange(parent.lpt.values[i], false, true);
        }
      }

      // push further (until a boundary clique is reached)
      this.pushPosterior(parent, cgt2, node);

    }
    return; // (there is at most one parent)
  }
  
  // no parent - just use the CGT (all weights = 1)
  node.p = cgt.values;
};

/**
 * Send message from one clique to another clique.
 * via the given separator (only between discrete 
 * cliques, otherwise this method does nothing).
 * @param from  sending clique
 * @param to    reveiving clique
 * @param sep   separator
 */
JTree.prototype.send = function (from, to, sep)
{
  if (sep.jpt) {
    to.jpt.div(sep.jpt);         // divide by old separator (s)
    (this.max 
     ? sep.jpt.mmarg(from.jpt)
     : sep.jpt.marg(from.jpt));  // create new separator (s')
    to.jpt.mult(sep.jpt);        // multiplicate by new separator
  }
};

/**
 * Update a-posteriori probabilities of all nodes.
 */ 
HybridJTree.prototype.update = function () 
{
  for (var nId in this.net.nodes) {
    var node = this.net.nodes[nId];
    if (node.cpt) {

      // discrete node
      (this.max 
       ? node.p.mmarg(this.home[node.id].jpt)
       : node.p.marg(this.home[node.id].jpt)); 

    } else {

      // continuous CG-node: copy LP-potential and 
      // "push" that copy towards a boundary cluster
      // where CG's will recieve their weights
      var cluster = this.cliques[this.order[node.id]];
      var cgt = new PTable(cluster.lpt.dims, null);
      for (var i = 0; i < cgt.values.length; i++) {
        cgt.values[i] = cluster.lpt.values[i].clone();
      }
      this.pushPosterior(cluster, cgt, node);

    }
  }
};


// HYBRID LIKELIHOOD WEIGHTING SAMPLER ==============================
// CONSTRUCT --------------------------------------------------------

/**
 * Constructs a likelihood weighting sampler (propagator)
 * for hybrid bayesian networks.
 * @param net           bayesian network to work on
 * @param numOfSamples  number of samples create for one propagation
 * @param keepSamples   store sampled data in this.samples for each
 *                      propagation (e.g. to be shown), where
 *                      stateIndex = this.samples.data[r][c]
 *                      and weight = this.samples.data[r].weight?
 */
function HybridLWSampler (net, numOfSamples, keepSamples)
{
  if (net != undefined) {
    this.initHybridLWSampler(net, numOfSamples, keepSamples);
  }
}

/* Inheritance */
HybridLWSampler.prototype             = new LWSampler();
HybridLWSampler.prototype.parent      = LWSampler;
HybridLWSampler.prototype.constructor = HybridLWSampler;

/**
 * Compile / Recompile the given bayesian network 
 * (see constructor {@link HybridLWSampler}).
 */
HybridLWSampler.prototype.initHybridLWSampler = function (net, numOfSamples, keepSamples) 
{
  this.initLWSampler(net, numOfSamples, keepSamples);
  this.id       = net.id + '.lws';
  this.name     = 'Hybrid LW-Sampler for "' + net.name + '"';
};

/**
 * Re-load probabilities from the bayesian network
 * (probabilities of some nodes might have changed
 * or the number of states of one dimensions might 
 * have changed).
 */
HybridLWSampler.prototype.reload = function () 
{
  this.cts = {};  // contingency tables for all discrete nodes
  this.cdts = {}; // tables of conditional distributions for all continuous nodes
  for (var nId in this.net.nodes) {
    var node = this.net.nodes[nId];
    if (node.cpt) {            // discrete node
      this.cts[nId] = new Float32Table(node.cpt.dims);
    } else if (node.getCGT) {  // continuous node (CGNode)
      this.cdts[nId] = node.getCGT();
    }
  }
};

/**
 * Fully recreate this propagator.
 */
HybridLWSampler.prototype.recreate = function () 
{
  this.initHybridLWSampler(this.net, this.n, this.keep);
};

// PROPAGATE --------------------------------------------------------

/**
 * Propagate (and update a-posteriori probability of all nodes).
 * @param observe  use current evidence?
 * @param update   update posterior probability tables of all nodes?
 * @returns        probability of evidence given the model: P(e|M)
 */ 
HybridLWSampler.prototype.propagate = function (observe, update) 
{
  // re-initialize contingency tables and 
  // clear posteriors of continuous nodes
  this.initCTs();
  if (update) {
    for (var nId in this.cdts) {
      this.net.nodes[nId].p = new Array();
    }
  }

  // creating n samples
  var pe     = 0;
  var sample = {};   // current sample (sample[nodeId] = stateIndex)
  for (var m = 0; m < this.n; m++) {

    // sample a state for all nodes given the states of their 
    // parents (parents have to be sampled before children)
    var weight = 1;
    for (var o = 0; o < this.order.length; o++) { 
      var node = this.order[o];

      if (this.cts[node.id]) {           // discrete node

        // find index into CPT, calculate weight and sample new value
        var i = node.cpt.getIndex(sample, 1, node.cpt.dims.length-1);
        var w = this.getWeight(observe, node.f, node.cpt, i);
        sample[node.id] = this.sampleState(observe, node.f, node.cpt, i, w);
        weight *= w;

      } else if (this.cdts[node.id]) {   // continuous node

        // find index into cond. distribution table, 
        // calculate weight and sample a new value
        var cdt = this.cdts[node.id];    // table of distributions
        var i   = cdt.getIndex(sample, 0, cdt.dims.length-1); 
        var cd  = cdt.values[i];         // distribution for current state-config.
        var w   = (node.v == null ? 1 : cd.value(node.v, sample));
        sample[node.id] = (node.v == null ? cd.sample(sample) : node.v);
        weight *= w;

        // add (unconditional) distribution to the list of 
        // posteriors of this node (will be weighted later)
        if (update) {
          cd = cd.clone();
          for (var t in cd.fact) cd.setTail(t, sample[t]); // make unconditional
          node.p.push(cd);
        }
      }

      // store sampled value
      if (this.keep) this.samples.data[m][o] = sample[node.id];
    }

    // adding (weighted) samples to all contingency tables
    // and set weights for distributions of continuous nodes
    this.cumulateCTs(sample, weight);
    if (update) {
      for (var nId in this.cdts) {
        var node = this.net.nodes[nId];
        node.p[node.p.length - 1].wght = weight;
      }
    }

    if (this.keep) this.samples.data[m].weight = weight;
    pe += weight;
  }

  // normalize contingency tables and update node posteriors
  this.normCTs();
  if (update) this.update();

  pe /= this.n;
  this.pe = pe;
  return (pe);
};

