/*
 * -------------------------------------------------------------------------
 *
 *  pn4webGenie.js Copyright (C) 2011, 
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *   Author: Patrick Rammelt 
 *   E-Mail: patrick.rammelt@go4more.de
 *   Site:   http://www.patricks-seite.de
 * -------------------------------------------------------------------------
 *
 * Reading bayesian networks stored by "Genie" (*.xdsl).
 */


// GRAPHS ===========================================================
// BAYESIAN NETWORK -------------------------------------------------

/**
 * Reading a bayesian network stored by Smile/Genie (*.xdsl).
 * <dl><dt>Notice:</dt>
 * <dd>Due to the same-origin-policy this file has to be
 *     stored in the same domain as the calling site.
 *     This also means that this doesn't work on a local
 *     installation. Used a local web-server like xampp to
 *     test this locally.</dd>
 * <dt>Notice II:</dt>
 * <dd>Currently only purely discrete bayesian networks are
 *     supported (no continuous/decision/utility nodes).
 *     Deterministic nodes are supported by converting them
 *     to a normal discrete node with a CPT containing ones
 *     and zeros only.</dd>    
 * </dl>
 * @param url           url of the xdsl-file
 * @param callback      function called when finished reading 
 *                      (no parameters)
 * @param errorhandler  function called in case of errors 
 *                      (one paremeter: error-message)
 * @param xScale        x-scale factor
 * @param yScale        y-scale factor
 * @param xoffset       x-offset (before scaling)
 * @param yoffset       y-offset (before scaling)
 */
BayesNet.prototype.readXDSL = function (url, callback, errorhandler, xScale, yScale, xOffset, yOffset) 
{
  // create xmlHttpRequest
  var req = newXMLHttpRequest();
  req.open('GET', url, true); 
  req.setRequestHeader('Content-Type', 'text/xml');

  // parse network
  var me = this;
  req.onreadystatechange = function() 
  {
    if (req.readyState == 4) {
      if (req.status == 200) {

        // parse xml and create this network
        try {
          var xml = (req.responseXML ? req.responseXML : parseXML(req.responseText));
          me.parseXDSL(xml, xScale, yScale, xOffset, yOffset);
        } catch (exception) {
          errorhandler(exception);
        }
        
        // call callback
        callback();
        
      } else {
        errorhandler("Could not read xdsl file: \"" + url + "\"\nReason: " + req.statusText + '(Code: ' + req.status + ')');
      }
    }
  };

  // send request
  try {
    req.send();
  } catch (exception) {
    errorhandler("Could not read xdsl file: \"" + url + "\"\nReason: " + exception);
  }
};


/**
 * Reading a bayesian network stored by Smile/Genie (*.xdsl).
 * <dl><dt>Notice:</dt>
 * <dd>Currently only purely discrete bayesian networks are
 *     supported (no continuous/decision/utility nodes).
 *     Deterministic nodes are supported by converting them
 *     to a normal discrete node with a CPT containing ones
 *     and zeros only.</dd>    
 * </dl>
 * @param xml      the xdsl-file as a xml-dom object
 * @param xScale   x-scale factor
 * @param yScale   y-scale factor
 * @param xoffset  x-offset (before scaling)
 * @param yoffset  y-offset (before scaling)
 */
BayesNet.prototype.parseXDSL = function (xml, xScale, yScale, xOffset, yOffset) 
{
  if (!xScale)  xScale  = 1;
  if (!yScale)  yScale  = 1;
  if (!xOffset) xOffset = 0;
  if (!yOffset) yOffset = 0;
          
  // get main XML elements 
  var xmlSmile    = xml.getElementsByTagName('smile')[0];
  var xmlGenie    = xml.getElementsByTagName('genie')[0];
  var xmlNodes    = xml.getElementsByTagName('nodes')[0]; // one node containing a node-collection
  var xmlNodeExts = xml.getElementsByTagName('node');     // names and positions (array of nodes)
  
  // get id and name
  this.id   = xmlSmile.getAttribute('id');
  this.name = xmlGenie.getAttribute('name');
  
  // get node names and positions of all nodes
  var extensions = {};
  for (var i = 0; i < xmlNodeExts.length; i++) {
    var xmlNodeExt = xmlNodeExts[i];
    var nodeId     = xmlNodeExt.getAttribute('id');
    var pos        = xmlNodeExt.getElementsByTagName('position')[0].firstChild.nodeValue.split(' ');
    extensions[nodeId] = 
    {
      name : xmlNodeExt.getElementsByTagName('name')[0].firstChild.nodeValue,
      x    : toInt((parseInt(pos[0]) + xOffset) * xScale),
      y    : toInt((parseInt(pos[1]) + yOffset) * yScale)
    };
  }

  // create (discrete) nodes
  for (var xmlNode = xmlNodes.firstElementChild; xmlNode; xmlNode = xmlNode.nextElementSibling) {

    // get main elements
    var tag    = xmlNode.nodeName;
    var nodeId = xmlNode.getAttribute('id');
    
    // create nodes (dimensions) and links
    switch (tag) {
    case  'cpt': case 'deterministic':

      // get states
      var xmlStates = xmlNode.getElementsByTagName('state');
      var states    = new Array(); 
      for (var s = 0; s < xmlStates.length; s++) {
        var xmlState = xmlStates[s];
        var stateId  = xmlState.getAttribute('id');
        var state    = new State(states.length, stateId); 
        states.push(state);
      }

      // create node (set CPT later, see below)
      var node = new Node(nodeId, extensions[nodeId].name, states, 
                          extensions[nodeId].x, extensions[nodeId].y); 
      this.addNode(node);

      // get parents (i.e. get all dimensions of the CPT) and add links 
      var dims = new Array(node);
      var xmlParents = xmlNode.getElementsByTagName('parents');
      if (xmlParents.length) {
        var paIds = xmlParents[0].firstChild.nodeValue.split(' ');
        for (var p = paIds.length - 1; p >= 0; p--) {
          var pa = this.nodes[paIds[p]];
          if (pa) {               // skipping unkown types of nodes
            dims.push(pa);
            this.addLink(pa, node);
          }
        }
      }
      break;
      
    default:
      throw ('Sorry, currently nodes of type "' + tag + '" are not supported.');
    }
    
    // create CPT's
    switch (tag) {
    case  'cpt':
      
      // parse and set node CPT
      var probs = xmlNode.getElementsByTagName('probabilities')[0].firstChild.nodeValue.split(' ');
      var values = new Float32Array(probs.length);
      for (var j = 0; j < probs.length; j++) values[j] = parseFloat(probs[j]);
      node.cpt = new Float32Table(dims, values);
      break;
      
    case 'deterministic':
      
      // convert "resulting states" to a normal CPT
      var resStates = xmlNode.getElementsByTagName('resultingstates')[0].firstChild.nodeValue.split(' ');
      var values = new Float32Array(resStates.length * states.length);
      var i = 0;
      for (var j = 0; j < resStates.length; j++) {
        for (var k = 0; k < states.length; k++) {
          values[i++] = (states[k].name == resStates[j] ? 1 : 0);
        }
      }
      node.cpt = new Float32Table(dims, values);
      break;
      
    default:
      // TODO continuous, descision and utility nodes 
      // Notice I:   Descision and utility nodes are not yet supported by any propagator
      // Notice II:  GeNIe does support generic continuous distributions (using sampling 
      //             methods), while pn4web currently supports CG-nodes only
      // Notice III: GeNIe does not support (connected) hybrid networks (continuous + 
      //             discrete nodes) because there is no way to represent a table of
      //             multiple continuous distributions
    }
  }
  
};


/**
 * Create a representation of this bayesian network as 
 * a Smile/Genie-file (*.xdsl).
 * <dl><dt>Notice:</dt>
 * <dd>Some mandatory attributes of a xdsl-file do not exist
 *     in a 'BayesNet' and will be set to fixed values (just 
 *     to fit the format), e.g. colors, width and height of 
 *     nodes.</dd>    
 * <dt>Notice II:</dt>
 * <dd>If you read in a network by 'readXDSL()/parseXDSL()'
 *     which contains deterministic nodes, these nodes will
 *     be converted to normal CPT-nodes. Therfore you won't
 *     see deterministic nodes here anymore.</dd>
 * <dt>Notice III:</dt>
 * <dd>CG-nodes are not supported by Smile/Genie and will be
 *     skipped (that's safe because a continuous node cannot 
 *     be a parent of a discrete node).</dd>
 * <dt>Notice IV:</dt>
 * <dd>Dynamic networks are not supported by Genie/Smile. A
 *     'DynBayesNet' will be stored just as it was a static 
 *     bayesian network.</dd>
 * </dl>
 * @param xScale   x-scale factor
 * @param yScale   y-scale factor
 * @param xoffset  x-offset (before scaling)
 * @param yoffset  y-offset (before scaling)
 * @return         content of a xdsl file (xml) as a string
 */
BayesNet.prototype.toXDSL = function (xScale, yScale, xOffset, yOffset)
{
  var lines = new Array(); // array of strings (one for each line)
  
  var order = (new NaturalOrdering(this)).getOrderedNodes();
  
  lines.push('<?xml version="1.0" encoding="ISO-8859-1"?>');
  lines.push('<smile version="1.0" id="' + this.id + '">');  // fixed version
  
  // nodes (states, parents, CPT-values)
  lines.push('  <nodes>');
  for (var i = 0; i < order.length; i++) {
    var node = order[i];
    if (node.cpt) {
      lines.push('    <cpt id="' + node.id + '">');

      // states
      for (var k = 0; k < node.states.length; k++) {
        lines.push('      <state id="' + node.states[k].id + '" />');
      }

      // parents (reverse order as in 'node.cpt.dims')
      if (node.cpt.dims.length > 1) {                        // (first dim is the node itself)
        var paIds = new Array();
        for (var d = node.cpt.dims.length - 1; d > 0; d--) { // (first dim is the node itself)
          paIds.push(node.cpt.dims[d].id);
        }
        lines.push('      <parents>' + paIds.join(' ') + '</parents>');
      }

      // probabilities
      lines.push('      <probabilities>' + node.cpt.values.join(' ') + '</probabilities>');

      lines.push('    </cpt>');
    }
  }
  lines.push('  </nodes>');
  
  // nodes (names and coordinates)
  lines.push('  <extensions>');
  lines.push('    <genie version="1.0" name="' + this.name + '">');        // fixed version
  for (var i = 0; i < order.length; i++) {
    var node = order[i];
    if (node.cpt) {
      lines.push('      <node id="' + node.id + '">');
      lines.push('        <name>' + node.name + '</name>');
      lines.push('        <interior color="e5f6f7" />');                   // fixed background color
      lines.push('        <outline color="0000bb" />');                    // fixed border color
      lines.push('        <font color="000000" name="Arial" size="8" />'); // fixed font
      lines.push('        <position>' + toInt(node.x/xScale-xOffset) + ' ' + toInt(node.y/yScale-yOffset) + ' 95 59</position>'); // fixed width and height
      lines.push('      </node>');
    }
  }
  lines.push('    </genie>');
  lines.push('  </extensions>');
  
  lines.push('</smile>');
  return (lines.join("\n"));
};

