/*
 * -------------------------------------------------------------------------
 *
 *  pn4webDisplayExamples.js Copyright (C) 2011, 
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *   Author: Patrick Rammelt 
 *   E-Mail: patrick.rammelt@go4more.de
 *   Site:   http://www.patricks-seite.de
 * -------------------------------------------------------------------------
 *
 * Example networks. To be called in documents e.g. by:
 * <body onload="initPN4Web(1, false, 300,250, 4,4); 
 *               displayABCNet('pn1'); 
 *               displayXXXNet('pnX');"> ... </body>
 */

// INFERENCE ALGORITHMS =============================================

/** 
 * Map mode-id to node of those node for which the joint probability
 * table should be calculated using y-propagation.
 */
var yNodes = {};

/**
 * Inference-Algorithms-Example 
 * - static (not dynamic)
 * - purely discrete (not hybrid)
 * - propagators: FullJoint, JTree, LoopyBelief, LWSampler, GibbsSampler
 * - normal or max-propagation (not for sampling-algorithms)
 * - y-propagation (for junction-trees only)
 */
function displayInferenceNet (netId, propType, maxProp, yProp)
{
  //try {
    dy = 0;
    var sWeather = new Array(new State(0, "gut"),
                             new State(1, "schlecht"));
    var nWeather = new Node("weather", "Wetter", sWeather, 0,dy+15); 
    nWeather.cpt = new Float32Table(new Array(nWeather),
                                    new Float32Array([0.7, 0.3]));
    
    var sTime = new Array(new State(0, "Nacht"),
                          new State(1, "Morgen"),
                          new State(2, "Tag"),
                          new State(3, "Abend"));
    var nTime = new Node("time", "Tageszeit", sTime, 400,dy+0);
    nTime.cpt = new Float32Table(new Array(nTime),
                                 new Float32Array([0.25, 0.25, 0.25, 0.25]));
    
    var sView = new Array(new State(0, "gut"),
                          new State(1, "schlecht"));
    var nView = new Node("view", "Sicht", sView, 200,dy+15);
    nView.cpt = new Float32Table(new Array(nView, nWeather, nTime),
			   new Float32Array([0.60, 0.40,   // wetter=gut       time=nacht 
			                     0.30, 0.70,   // wetter=schlecht
			                     0.70, 0.30,   // wetter=gut       time=morgen 
			                     0.80, 0.20,   // wetter=schlecht
			                     0.95, 0.05,   // wetter=gut       time=tag 
			                     0.85, 0.15,   // wetter=schlecht
			                     0.70, 0.30,   // wetter=gut       time=abend
			                     0.80, 0.20])); // wetter=schlecht
    
    var sFocus = new Array(new State(0, "wach"),
                           new State(1, "müde"));
    var nFocus = new Node("focus", "Konzentration", sFocus, 400,dy+180);
    nFocus.cpt = new Float32Table(new Array(nFocus, nWeather, nTime),
			    new Float32Array([0.65, 0.35,    // wetter=gut       time=nacht 
			                      0.25, 0.75,    // wetter=schlecht
			                      0.70, 0.30,    // wetter=gut       time=morgen 
			                      0.85, 0.15,    // wetter=schlecht
			                      0.95, 0.05,    // wetter=gut       time=tag 
			                      0.80, 0.20,    // wetter=schlecht
			                      0.75, 0.25,    // wetter=gut       time=abend
			                      0.85, 0.15])); // wetter=schlecht

    var sStreet = new Array(new State(0, "griffig"),
                            new State(1, "rutschig"));
    var nStreet = new Node("street", "Straße", sStreet, 0,dy+180);
    nStreet.cpt = new Float32Table(new Array(nStreet, nWeather),
			     new Float32Array([0.95, 0.05,    // wetter=gut 
			                       0.30, 0.70])); // wetter=schlecht
    
    var sAccident = new Array(new State(0, "wenige"),
                              new State(1, "viele"));
    var nAccident = new Node("accident", "Unfälle", sAccident, 200,dy+240);
    nAccident.cpt = new Float32Table(new Array(nAccident, nStreet, nView, nFocus),
			       new Float32Array([0.95, 0.05,    // street=griffig   view=gut      focus=wach
			                         0.25, 0.75,    // street=rutschig
			                         0.30, 0.70,    // street=griffig   view=schlecht
			                         0.50, 0.50,    // street=rutschig
			                         0.45, 0.55,    // street=griffig   view=gut      focus=müde
			                         0.65, 0.35,    // street=rutschig
			                         0.85, 0.15,    // street=griffig   view=schlecht
			                         0.10, 0.90])); // street=rutschig
    
    var net = new BayesNet(netId, "Bayesian Network");
    net.addNode(nWeather);
    net.addNode(nTime);
    net.addNode(nView);
    net.addNode(nFocus);
    net.addNode(nStreet);
    net.addNode(nAccident);
    
    net.addLink(nWeather, nStreet);
    net.addLink(nWeather, nView);
    net.addLink(nWeather, nFocus);
    net.addLink(nTime,    nView);
    net.addLink(nTime,    nFocus);
    net.addLink(nView,    nAccident);
    net.addLink(nFocus,   nAccident);
    net.addLink(nStreet,  nAccident);

    var prop;
    switch (propType) {
    case 'fullJoint': 
      prop = new FullJoint(net, maxProp);
      decorate(prop.id, false, false, false, false, true, DECO.REDFRAME);
      PN4W.displayPTable(prop.fjpt, prop.id + '.table', document.getElementById(prop.id), PN4W.FJPTABLE);
      vscroll(prop.id + '.frame', prop.id + '.area', prop.id, SCROLL.BLUE); 
      hscroll(prop.id + '.frame', prop.id + '.area', prop.id, SCROLL.BLUE); 
      break;
    case 'junctionTree':
      if (yProp) {
        for (var nId in net.nodes) {
          var checkBox = document.getElementById('check.' + nId);
          if (checkBox.checked) yNodes[nId] = net.nodes[nId];
          mkYNodeCheckBoxSensitive(checkBox, net, nId, maxProp);
        }
        prop = new YJTree(net, {}, false, maxProp, yNodes); // network, restr, fillIn2Net, max-propagation, y-prop-nodes
        prop.layout(190, 600, true);
      } else {
        prop = new JTree(net, {}, false, maxProp); // network, restr, fillIn2Net, max-propagation
        prop.layout(190, 400, true);
      }
      decorate(prop.id, false, false, false, false, true, DECO.BLUEFRAME);
      PN4W.displayNet(prop, PN4W.JTREE);
      break;
    case 'loopyBelief':
      prop = new LoopyBelief(net, 10, maxProp);
      decorate(prop.id, false, false, false, false, true, DECO.BLUEFRAME);
      PN4W.displayNet(prop, PN4W.LOOPYBELIEF);
      break;
    case 'lwSampler':
      prop = new LWSampler(net, 100, true); // network, numOfSamples, keepSamples
      decorate(prop.id, false, false, false, false, true, DECO.REDFRAME);
      PN4W.displaySamples(prop.samples, prop.id + '.table', document.getElementById(prop.id), PN4W.SAMPLES);
      vscroll(prop.id + '.frame', prop.id + '.area', prop.id, SCROLL.BLUE); 
      hscroll(prop.id + '.frame', prop.id + '.area', prop.id, SCROLL.BLUE); 
      break;
    case 'gibbsSampler':
      prop = new GibbsSampler(net, 
			      100, 25, 10,   // numOfSamples, reSampleIn, burnIn, 
			      0, 50, true);  // minWeight, maxTrials, keepSamples
      decorate(prop.id, false, false, false, false, true, DECO.REDFRAME);
      decorate(prop.id + '.mbnet', false, false, false, false, true, DECO.BLUEFRAME);
      PN4W.displayNet(prop.mbNet, PN4W.MARKOVBLANKETNET);
      PN4W.displaySamples(prop.samples, prop.id + '.table', document.getElementById(prop.id), PN4W.SAMPLES);
      vscroll(prop.id + '.frame', prop.id + '.area', prop.id, SCROLL.BLUE); 
      hscroll(prop.id + '.frame', prop.id + '.area', prop.id, SCROLL.BLUE); 
      break;
    default:
      throw ("Unkown propagator type");
    }
    net.prop = prop;
    
    decorate(net.id, false, false, false, false, true, DECO.BLUEFRAME);
    PN4W.displayNet(net, PN4W.NET);
    //PN4W.displayCPT(net,n1);
    PN4W.displayPePopup(net, (net.prop && net.prop.max ? 'P(C|M)' : 'P(e|M)'), 1, PN4W.PE, 0,500);
    PN4W.propagate(net);

    //} catch (exception) {
    //alert("Accident Network: " + exception);
    //}
}

/**
 * Initialization, and incorporation of observations 
 * without collect/normalization/distribute/update.
 */
function initInferenceJTree (netId)
{
  var net = PN4W.nets[netId];
  var prop = net.prop;
  prop.init();
  prop.observe();
  PN4W.refresh(net);
}

/**
 * Initialization, incorporation of observations and 
 * collect-phase without normalization/distribute/update.
 */
function collectInferenceJTree (netId)
{
  var net = PN4W.nets[netId];
  var prop = net.prop;
  prop.init();
  prop.observe();
  for (var rId in prop.roots) {         // over all disconnected sub-trees
    prop.collect(prop.roots[rId]);  // collect phase
  }
  PN4W.refresh(net);
}

/**
 * Initialization, incorporation of observations,
 * collect-phase and normalization without distribute/update.
 */
function normInferenceJTree (netId)
{
  var net = PN4W.nets[netId];
  var prop = net.prop;
  prop.init();
  prop.observe();
  prop.pe = 1;
  for (var rId in prop.roots) {              // over all disconnected sub-trees
    prop.collect(prop.roots[rId]);           // collect phase
    prop.pe *= prop.roots[rId].jpt.norm();   // ...P(e|M)
  }
  PN4W.refresh(net);
}

/**
 * Initialization, incorporation of observations,
 * collect-phase, distribute-phase and update.
 */
function fullInferenceJTree (netId)
{
  var net = PN4W.nets[netId];  
  PN4W.propagate(net);
}


/**
 * Initialization, and incorporation of observations 
 * without propagation.
 */
function initInferenceLoopyBelief (netId)
{
  var net = PN4W.nets[netId];
  var prop = net.prop;
  prop.init();
  prop.observe();
  PN4W.refresh(net);
}

/**
 * Initialization, incorporation of observations and 
 * collect-phase without normalization/distribute/update.
 */
function propInferenceLoopyBelief (netId, loops)
{
  var net = PN4W.nets[netId];  
  var prop = net.prop;
  var l = prop.loops;
  prop.loops = loops;
  PN4W.propagate(net);
  prop.loops = l;
}

/**
 * Make the check-box element sensitive (re-create the {@link YJTree}
 * after the check-box has changed from checked to unchecked or vice versa. 
 */
function mkYNodeCheckBoxSensitive (checkBox, net, nId, maxProp) 
{ 
  addEventListener(checkBox, 'click', function(evnt){
    // preventDefault(evnt); don't prevent default behaviour toggling the state)
    if (yNodes[nId]) delete(yNodes[nId]); 
    else yNodes[nId] = net.nodes[nId];
    PN4W.clearNet(net.prop);
    net.prop = new YJTree(net, {}, false, maxProp, yNodes); // network, restr, fillIn2Net, max-propagation, y-prop-nodes
    net.prop.layout(190, 600, true);
    PN4W.displayNet(net.prop, PN4W.JTREE);
    PN4W.propagate(net);
  });
};
