/*
 * -------------------------------------------------------------------------
 *
 *  pn4webDisplayExamples.js Copyright (C) 2011, 
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *   Author: Patrick Rammelt 
 *   E-Mail: patrick.rammelt@go4more.de
 *   Site:   http://www.patricks-seite.de
 * -------------------------------------------------------------------------
 *
 * Example networks. To be called in documents e.g. by:
 * <body onload="initPN4Web(1, false, 300,250, 4,4); 
 *               displayABCNet('pn1'); 
 *               displayXXXNet('pnX');"> ... </body>
 */


// DISPLAY TREE DECOMPOSITION WITH EVOLUTIONARY ALGORITHMS ==========

/** 
 * Decomposition network (only recreated if the number 
 * of nodes and/or the connectivity has changed) 
 */ 
var decompNet = null;

/** 
 * Type parameter used for the current 'decompNet'
 */
var decompNetType = "";

/**
 * Stop optimization? (decompStop[netId] -> true/false)  
 */
var decompStop = {};

/**
 * Stop decomposition optimization
 */
function stopDecomp (netId)
{
  decompStop[netId] = true;
  document.getElementById(netId + '.stop').disabled  = true;
}

/**
 * Display decomposition optimization
 */
function displayDecomp (netId)
{
  try {
    
    // get settings
    var pConnect;
    var nrOfNodes = undefined, nrOfCauses = undefined, nrOfSensors = undefined;
    var netType = document.getElementById(netId + '.nettype').value;
    switch (netType) {
    case '10_1': nrOfNodes = 10; pConnect =  1.0/nrOfNodes; break;
    case '10_2': nrOfNodes = 10; pConnect =  5.0/nrOfNodes; break;
    case '20_1': nrOfNodes = 20; pConnect =  2.0/nrOfNodes; break;
    case '20_2': nrOfNodes = 20; pConnect = 10.0/nrOfNodes; break;
    case '50_1': nrOfNodes = 50; pConnect =  5.0/nrOfNodes; break;
    case '50_2': nrOfNodes = 50; pConnect = 15.0/nrOfNodes; break;
    case '10_5_1' : nrOfCauses = 10; nrOfSensors =  5; pConnect =  5.0/nrOfCauses; break; // XXX not used
    case '20_10_1': nrOfCauses = 20; nrOfSensors = 10; pConnect = 10.0/nrOfCauses; break; // XXX not used
    case '40_10_1': nrOfCauses = 40; nrOfSensors = 10; pConnect = 15.0/nrOfCauses; break; // XXX not used
    default: throw ("Network not supported");
    }
    
    var maxIndividuals;
    switch (document.getElementById(netId + '.maxindividuals').value) {
    case '10' : maxIndividuals =  10; break;
    case '20' : maxIndividuals =  20; break;
    case '50' : maxIndividuals =  50; break;
    case '100': maxIndividuals = 100; break;
    default: throw ("Number of individuals not supported");
    }
    
    var maxOffspring;
    switch (document.getElementById(netId + '.maxoffspring').value)
    {
    case '0.125' : maxOffspring = Math.floor(maxIndividuals * 0.125); break;
    case '0.25'  : maxOffspring = Math.floor(maxIndividuals * 0.25);  break;
    case '0.5'   : maxOffspring = Math.floor(maxIndividuals * 0.5);   break;
    case '1'     : maxOffspring = Math.floor(maxIndividuals * 1.0);   break;
    case '2'     : maxOffspring = Math.floor(maxIndividuals * 2.0);   break;
    case '4'     : maxOffspring = Math.floor(maxIndividuals * 4.0);   break;
    case '8'     : maxOffspring = Math.floor(maxIndividuals * 8.0);   break;
    default: throw ("Number of offspring individuals not supported");
    }
    
    var noise;
    switch (document.getElementById(netId + '.initialquality').value) {
    case 'greedy' : noise = 0.5;                          break; // < 1
    case 'noisy1' : noise = nrOfNodes / 2.0;              break;
    case 'noisy2' : noise = nrOfNodes * nrOfNodes / 10.0; break;
    case 'random' : noise = nrOfNodes * nrOfNodes;        break;
    default: throw ("Quality of Initial population not supported");
    }

    var maxGenerations;
    switch (document.getElementById(netId + '.maxgenerations').value) {
    case '1'  : maxGenerations =   1; break;
    case '2'  : maxGenerations =   2; break;
    case '5'  : maxGenerations =   5; break;
    case '10' : maxGenerations =  10; break;
    case '20' : maxGenerations =  20; break;
    case '50' : maxGenerations =  50; break;
    case '100': maxGenerations = 100; break;
    default: throw ("Number of generations not supported");
    }

    var mutator, rate = 0.5;
    switch (document.getElementById(netId + '.mutator').value) {
    case 'none': mutator = new ISM(0);    break;
    case 'ISM' : mutator = new ISM(rate); break;
    case 'EM'  : mutator = new ME(rate);  break;
    default: throw ("Mutation algorithm not supported");
    }

    var recombinator, success = 1.0;
    switch (document.getElementById(netId + '.recombinator').value) {
    case 'OX1': recombinator = new JT2ElimOrderRecombinator(new OX1(mutator, success)); break;
    case 'OX2': recombinator = new JT2ElimOrderRecombinator(new OX2(mutator, success)); break;
    case 'POS': recombinator = new JT2ElimOrderRecombinator(new POS(mutator, success)); break;
    case 'EOT': recombinator = new EOT(mutator, success, 0, 0.95, 0.99); break;
    default: throw ("Recombination algorithm not supported");
    }

    var evaluator = new FillInFitness();
    var agingPerGeneration = 0.0;
    var satisfyingFittness = 1000;
    var delay              = 0;
    
    decorate(netId + '.log', false, false, false, false, true, DECO.BLACKFRAME); // (clear)
    var logElem  = document.getElementById(netId + '.log'); // (after decorate) 
    var log = appendNewElement(logElem, 'div', netId + '.log.content', 'pn4web_log');

    var graphElem = document.getElementById(netId + '.graph');
    if (!graphElem || !graphElem.getContext) return;
    var context = graphElem.getContext('2d');
    if (!context) return;
    var x0 = 25;
    var y0 = 25;
    var w = getWidth(graphElem)  - 50 - x0;
    var h = getHeight(graphElem) - 50 - y0;
    // clear
    context.fillStyle = 'rgba(255,255,255,1.0)';
    context.clearRect(0, 0, w+1000, h+1000);
    //context.fillRect(0, 0, w+1000, h+1000);

    // re-use existing net or create a new one 
    var net;
    if (decompNetType == netType) {
      net = decompNet; 
      log.innerHTML += ('Re-using previous network with ' + net.getNodes().length + ' nodes and ' + countLinks4Decomp(net) + ' links<br>'); 
      vscroll(net.id + '.log.frame', netId + '.log.area', net.id + '.log', SCROLL.BLUE); 
    } else {
      if (nrOfNodes) {
        net = createDecompNet1(netId, nrOfNodes, pConnect);
      } else {
        net = createDecompNet2(netId, nrOfCauses, nrOfSensors, pConnect);
      }      
      log.innerHTML += ('New network created with ' + net.getNodes().length + ' nodes and ' + countLinks4Decomp(net) + ' links<br>'); 
      vscroll(net.id + '.log.frame', netId + '.log.area', net.id + '.log', SCROLL.BLUE); 
      decompNet = net;
      decompNetType = netType;
    }
    
    // initial set of individuals
    // ('individuals' will be filled just before optimization starts)
    var individuals = new Array();
        
    // create optimizer
    var optimizer = new EA(individuals, recombinator, evaluator,  
                           maxGenerations, maxIndividuals, maxOffspring, 
                           agingPerGeneration, satisfyingFittness, delay);
    
    var worst = 0;
    var listener = 
    {
     start : function () {
       document.getElementById(netId + '.stop').disabled  = false;
       document.getElementById(netId + '.start').disabled = true;
     },
     step : function (i) {
       if (i == 0) {
         worst = Math.floor(-optimizer.population[optimizer.population.length-1].fitness)+1;
         // paint coordinate system
         context.shadowColor  = 'transparent';
         context.strokeStyle  = 'rgba(0,0,0,1.0)';
         context.textBaseline = 'middle';
         context.textAlign    = 'right';
         context.fillStyle    = 'rgba(0,0,0,1.0)';
         context.font         = 'normal 9px sans-serif';
         context.beginPath(); context.moveTo(x0,0); context.lineTo(x0,h);   context.stroke(); // y-axis
         context.beginPath(); context.moveTo(x0,h); context.lineTo(w+x0,h); context.stroke(); // x-axis
         var s = Math.floor(worst/10)+1;
         for (var f = 0; f < worst; f += s) {
           var y = h - (h * f/worst);
           context.beginPath(); context.moveTo(x0-5,y); context.lineTo(x0,y); context.stroke(); // y-ticks
           context.fillText(f, x0-7, y); // y-tick-labels
         }
         context.textBaseline = 'top';
         context.textAlign    = 'center';
         for (var g = 0; g <= maxGenerations; g++) {
           var x = x0 + w * g/(maxGenerations);
           context.beginPath(); context.moveTo(x,h); context.lineTo(x,h+5); context.stroke(); // x-ticks
         }
         for (var g = 0; g <= maxGenerations; g++) {
           var x = x0 + w * (g + 0.5)/(maxGenerations);
           context.fillText(g, x, h + 7); // x-tick-labels
         }
         // settings for data points
         context.strokeStyle = 'rgba(0,0,0,0.5)';
         context.shadowOffsetX = 0;
         context.shadowOffsetY = 0;
         context.shadowBlur    = 10;
         context.shadowColor   = 'rgba(255,0,0,1.0)'; //'rgba(146,208,80,1.0)';
       }
       for (var k = 0; k < optimizer.population.length; k++) {
         var f = (-optimizer.population[k].fitness);
         var x1 = x0 + w * i/(maxGenerations), x2 = x0 + w * (i+1)/(maxGenerations), y = h - (h * f/worst);
         context.beginPath(); context.moveTo(x1,y); context.lineTo(x2,y); context.stroke();
       }
       log.innerHTML += ('Generation ' + i + ': ' + optimizer.individuals.length + ' individuals; fitness: [' + 
           Math.floor(-optimizer.population[0].fitness + 0.5) + '..' + 
           Math.floor(-optimizer.population[optimizer.population.length-1].fitness + 0.5) + '] fill-ins<br>'); 
       vscroll(net.id + '.log.frame', netId + '.log.area', net.id + '.log', SCROLL.BLUE);
       return (!decompStop[netId]);
     }, 
     stop : function (regular) {
       if (regular) {
         log.innerHTML += ('Optimization finished - best fitness: ' + Math.floor(-optimizer.population[0].fitness) + ' fill-ins<br>');
       } else {
         log.innerHTML += ('Optimization stopped - best fitness: ' + Math.floor(-optimizer.population[0].fitness) + ' fill-ins<br>');
       }
       vscroll(net.id + '.log.frame', netId + '.log.area', net.id + '.log', SCROLL.BLUE); 
       decompStop[netId] = false;
       document.getElementById(netId + '.stop').disabled = true;
       document.getElementById(netId + '.start').disabled = false;
     }
    };
    optimizer.addListener(listener);
    
    // create initial set of individuals and start optimization
    var optimize = function () {
      var individual = new JTree(net, {}, false, false, noise); 
      individuals.push(individual);
      if (individuals.length < maxIndividuals) { 
        setTimeout(optimize, 0);
      } else {
        optimizer.optimize();
      }
    };
    optimize();

  } catch (exception) {
    alert("Random network error: " + exception);
  }
}


/** 
 * Create a fully randomized network for 
 * tree-decomposition (with dummy CPTs).
 */
function createDecompNet1 (netId, size, connectivity)
{
  net = new BayesNet(netId, "Random Network");
  var ordering   = new NaturalOrdering(net); // helpers used to decide...
  var clustering = new Clustering(net);      // ...if two nodes are linkable

  // create nodes (without CPTs)
  for (var i = 0; i < size; i++) {
    var states = new Array(new State(0, "s1"));          // to save memory we use only one state per node
    var node   = new Node("n"+i, "Node"+i, states, 0,0);
    net.addNode(node);
  }

  // create links and CPTs // TODO max. cyclic
  var links = 0;
  for (var id1 in net.nodes) {
    var node  = net.nodes[id1];
    var nodes = new Array(node);
    for (var id2 in net.nodes) {
      var parent = net.nodes[id2];
      if (Math.random() > connectivity) {
        // do not connect
      } else if (parent.id == node.id) {
        // self-links are not allowed
      } else if (net.children[parent.id][node.id]) {
        // link exists
      } else if (!clustering.isConnected(node, parent)) {
        net.addLink(parent, node); // different clusters are always linkable
        nodes.push(parent);
        links++;
      } else if (!ordering.isAncestor(node, parent)) {
        net.addLink(parent, node);   // net will stay acyclic
        nodes.push(parent);
        links++;
      } else {
        // link would create a cycle
      }
    }
    node.cpt = new Float32Table(nodes, new Float32Array([1]));
  }
  return (net);
}

/** 
 * Create a diagnostic type network for 
 * tree-decomposition (with dummy CPTs).
 * @deprecated  This does create good examples for tree decomposition! 
 */
function createDecompNet2 (netId, causes, sensors, connectivity)
{
  net = new BayesNet(netId, "Random Diagnostic Type Network");

  // create cause nodes 
  var n = 0;
  var causeNodes = new Array();
  for (var i = 0; i < causes; i++) {
    var states = new Array(new State(0, "s1"));   // to save memory we use only one state per node
    var node   = new Node("n"+n, "node"+n, states, 0,0);
    net.addNode(node);
    node.cpt = new Float32Table(new Array(node), 
                                new Float32Array([1]));
    causeNodes.push(node);
    n++;
  }
  
  // create intermediate and sensor nodes
  for (var i = 0; i < sensors; i++) {
    
    var parents = new Array();
    for (var j = 0; j < causes; j++) {
      if (Math.random() <= connectivity) {
        var states = new Array(new State(0, "s1"));   // to save memory we use only one state per node
        var node   = new Node("n"+n, "node"+n, states, 0,0); // sensitivity node
        net.addNode(node);
        net.addLink(causeNodes[j], node); 
        node.cpt = new Float32Table(new Array(node, causeNodes[j]), 
                                    new Float32Array([1])); 
        parents.push(node);
        n++;
      }
    }
    
    while (parents.length > 1) {
      var children = new Array();
      for (var j = 1; j < parents.length; j += 2) {
        var states = new Array(new State(0, "s1"));   // to save memory we use only one state per node
        var node   = new Node("n"+n, "node"+n, states, 0,0); // or-node / sensor node
        net.addNode(node);
        net.addLink(parents[j-1], node);
        net.addLink(parents[j],   node);
        node.cpt = new Float32Table(new Array(node, parents[j-1], parents[j]), 
                                    new Float32Array([1]));
        children.push(node);
        n++;
      }
      if (j == parents.length) {
        children.push(parents[j-1]);
      }
      parents = children;
    } 
    
  }

  return (net);
}

/**
 * Count the number of links in a bayesian network
 */
function countLinks4Decomp (net)
{
  var links = 0;
  for (var paId in net.children) {
    for (var chId in net.children[paId]) {
      links++;
    }
  }
  return (links);
}
