/*
 * -------------------------------------------------------------------------
 *
 *  pn4webDynamic.js Copyright (C) 2011, 
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *   Author: Patrick Rammelt 
 *   E-Mail: patrick.rammelt@go4more.de
 *   Site:   http://www.patricks-seite.de
 * -------------------------------------------------------------------------
 *
 * All logic needed for dynamic bayesian networks.
 * (these types are fully independent of display functionalities)
 */

// DYNAMIC BAYESIAN NETWORK =========================================

/**
 * Constructor for a dynamic bayesian network.
 * @param id    id
 * @param name  name
 */
function DynBayesNet (id, name)
{
  if (id != undefined) {
    this.initDynBayesNet(id, name);
  }
}

/* Inheritance */
DynBayesNet.prototype             = new BayesNet();
DynBayesNet.prototype.parent      = BayesNet.prototype;
DynBayesNet.prototype.constructor = DynBayesNet;

/**
 * Initialize this dynamic bayesian network 
 * (see constructor {@link DynBayesNet}).
 */
DynBayesNet.prototype.initDynBayesNet = function (id, name) 
{
  this.initBayesNet(id, name);
  this.pred = {};  // map: pred[nodeId] = preNode 
  this.succ = {};  // map: succ[preNodeId] = node
  this.rate = 128; // [ms] time delay between two time slices
};

/**
 * Add a node and its predecessor.
 * @param preNode  predecessor (t-1) of the node
 * @param node     node (current time-slice t)
 */
DynBayesNet.prototype.addDynNode = function (preNode, node)
{
  this.addNode(preNode);
  this.addNode(node);
  this.pred[node.id]    = preNode;
  this.succ[preNode.id] = node;
};

/**
 * Remove a node and its predecessor.
 * @param node  node (current time-slice t)
 */
DynBayesNet.prototype.rmDynNode = function (node)
{
  var preNode = this.pred[node.id];
  this.rmNode(preNode);
  this.rmNode(node);
  delete(this.pred[node.id]);
  delete(this.succ[preNode.id]);
};


// DYNAMIC DATA =====================================================

/**
 * Constructor for a data collection (normally used for 
 * dynamic bayesian networks / dynamic propagators). 
 */
function TimeData ()
{
  this.initTimeData();
}

/* Inheritance */
// TimeData.prototype          = ;
// TimeData.prototype.parent   = ;
TimeData.prototype.constructor = TimeData;

/**
 * Re-initialize this data collection 
 * (see constructor {@link TimeData}).
 */
TimeData.prototype.initTimeData = function ()
{
  this.data = {};
  this.tmin = Number.POSITIVE_INFINITY;
  this.tmax = Number.NEGATIVE_INFINITY;
};

/**
 * Store one piece of data.
 * @param t   time slice (0,1,2,...)
 * @param id  id (normally a node id)
 * @param d   data to be stored (you might want
 *            to store a copy of the actual data!)
 */
TimeData.prototype.store = function (t, id, d)
{
  this.tmin = Math.min(this.tmin, t);
  this.tmax = Math.max(this.tmax, t);
  if (!this.data[t]) this.data[t] = {};
  this.data[t][id] = d;
};

/**
 * Remove data.
 * @param t   time slice (0,1,2,...)
 * @param id  id (normally a node id) [optional]
 */
TimeData.prototype.unstore = function (t, id)
{
  if (id == undefined)   delete (this.data[t]);
  else if (this.data[t]) delete (this.data[t][id]);
};

/**
 * Is there data stored?
 * @param t   time slice (0,1,2,...)
 * @param id  id (normally a node id) [optional]
 * @returns   true if there is data stored, false otherwise
 */
TimeData.prototype.isStored = function (t, id)
{
  if (!this.data[t])   return (false);
  if (id == undefined) return (this.data[t]     != undefined);
  else                 return (this.data[t][id] != undefined);
};

/**
 * Get one piece of stored data
 * @param t   time slice (0,1,2,...)
 * @param id  id (normally a node id)
 * @returns   piece of data or 'undefined' if nothing is
 *            stored for the given id in time slice t
 */
TimeData.prototype.stored = function (t, id)
{
  if (!this.data[t]) return (undefined);
  else               return (this.data[t][id]);
};


// DYNAMIC PROPAGATORS ==============================================
// DYNAMIC CONNECTOR FOR DYNAMIC PROPAGATORS ------------------------

/**
 * Constrcutor for a "dynamic connector" which is able to send 
 * a dynamic message about all dynamic nodes forward/backward
 * from a joint probability table into that very same joint
 * probability table (a dynamic connector is very similar to 
 * a separator in a junction tree).
 */
function DynConnector (net, jpt)
{
  if (net != undefined) {
    this.initDynConnector(net, jpt);
  }
}

/* Inheritance */
// DynConnector.prototype          = ;
// DynConnector.prototype.parent   = ;
DynConnector.prototype.constructor = DynConnector;

/**
 * Re-initialize this dynamic connector 
 * (see constructor {@link DynConnector}).
 */
DynConnector.prototype.initDynConnector = function (net, jpt)
{
  this.net = net;
  this.jpt = jpt;
  var predDims = new Array();  // predecessor dimensions (t-1)
  var succDims = new Array();  // successor dimensions   (t)
  for (var id in net.pred) {
    predDims[predDims.length] = net.pred[id];
    succDims[succDims.length] = net.nodes[id];
  }
  this.pred = new Float32Table(predDims); // predecessor node table
  this.succ = new Float32Table(succDims); // successor node table
};

/**
 * Receives a message from the joint probability 
 * table to be send forward/backward in time.
 * @param dt  -1 means backward, +1 means forward
 */
DynConnector.prototype.receive = function (dt) 
{
  if (dt > 0) {
    (this.net.prop.max 
     ? this.succ.mmarg(this.jpt) 
     : this.succ.marg(this.jpt));
  } else if (dt < 0) {
    (this.net.prop.max 
     ? this.pred.mmarg(this.jpt)
     : this.pred.marg(this.jpt));
  }
};

/**
 * Send the last received message forward/backward in time 
 * from the joint probability table to itself.
 * @param dt  -1 means backward,<br> 
 *            +1 means forward,<br>
 *             0 means do not send
 */
DynConnector.prototype.send = function (dt) 
{
  if (dt > 0) {
    (this.net.prop.max 
     ? this.pred.mmarg(this.jpt)
     : this.pred.marg(this.jpt));  // get info to be removed
    this.jpt.div(this.pred);       // remove info
    this.pred.exchange(this.succ); // transform information
    this.jpt.mult(this.pred);      // multiply in dynamic info
    this.pred.exchange(this.succ); // transform back
  } else if (dt < 0) {
    (this.net.prop.max 
     ? this.succ.mmarg(this.jpt)
     : this.succ.marg(this.jpt));  // get info to be removed
    this.jpt.div(this.succ);       // remove info
    this.pred.exchange(this.succ); // transform information
    this.jpt.mult(this.succ);      // multiply in dynamic info
    this.pred.exchange(this.succ); // transform back
  }
};

// FULL JOINT PROPAGATOR FOR DYNAMIC BAYESIAN NETWORKS ==============

/**
 * Constructor for a simple propagator which creates the 
 * full joint propability table for all nodes in a dynamic
 * bayesian network (see also 'FullJoint').
 * @param net     dynamic bayesian network ({@link DynBayesNet})
 * @param window  time window size (max number of steps for 
 *                back-propagation)
 * @param max     max-propagation? [optional]
 */
function DynFullJoint (net, window, max)
{
  if (net != undefined) {
    this.initDynFullJoint(net, window, max);
  }
}

/* Inheritance */
DynFullJoint.prototype             = new FullJoint();
DynFullJoint.prototype.parent      = FullJoint.prototype;
DynFullJoint.prototype.constructor = DynFullJoint;

/**
 * Compile / Recompile the given bayesian network 
 * (see constructor {@link DynFullJoint}).
 */
DynFullJoint.prototype.initDynFullJoint = function (net, window, max) 
{
  this.initFullJoint(net, max);

  // time slice and delta time
  this.window   = window;            // time window size (max. steps of back-propagation)
  this.t        = 0;                 // current time slice index (0,1,2,...)
  this.dt       = 0;                 // 1:forward, -1:backward, 0:no change in time
  this.timeData = new TimeData();    // fjpt value arrays stored for previous time slices
  this.conn     = new DynConnector(this.net, this.fjpt); // dynamic connector
};

/**
 * Re-load probabilities from the bayesian network
 * (probabilities of some nodes might have changed
 * or the number of states of one dimensions might 
 * have changed).
 */
DynFullJoint.prototype.reload = function () 
{
  this.parent.reload.call(this);
  this.t        = 0;                 // current time slice index (0,1,2,...)
  this.dt       = 0;                 // 1:forward, -1:backward, 0:no change in time
  this.timeData = new TimeData();    // fjpt value arrays stored for previous time slices
  this.conn     = new DynConnector(this.net, this.fjpt); // dynamic connector
};

/**
 * Fully recreate this propagator.
 */
DynFullJoint.prototype.recreate = function () 
{
  this.initDynFullJoint(this.net, this.window);
};

/**
 * Propagate (and update a-posteriori probability of all nodes).
 * @param observe  use current evidence?
 * @param update   update posterior probability tables of all nodes?
 * @returns        probability of evidence given the model:    P(e|M)<br>
 *                 or of the most probable config. (max-prop): P(C|e)
 */ 
DynFullJoint.prototype.propagate = function (observe, update) 
{
  this.init();                     // re-initialize full joint prop. table
  this.conn.send(this.dt);         // multiply in information from last/next time slice
  if (observe) this.observe();     // multiply in evidence
  this.pe = (this.max 
	     ? this.fjpt.mnorm()
	     : this.fjpt.norm());  // normalize and get probability of evidence
  if (update) this.update();       // update posteriors of all nodes
  return (this.pe);
};

/**
 * Re-initialize full joint probability table for next
 * propagation (time-slice).
 */ 
DynFullJoint.prototype.init = function () 
{
  // re-initialize full joint prop. table
  this.fjpt.values = (this.timeData.isStored(this.t, 'fjpt')
		      ? this.timeData.stored(this.t, 'fjpt')   // re-use stored configuration for backprop.
		      : this.initData.stored('fjpt').slice()); // initialize
};

/**
 * Move back to first slice. Information will 
 * be used for the next call to propagate().
 */ 
DynFullJoint.prototype.stop = function () 
{
  this.timeData.initTimeData();
  this.dt = 0;
  this.t  = 0;
};

/**
 * Move one step backward in time. Information 
 * will be used for the next call to propagate().
 */ 
DynFullJoint.prototype.prev = function () 
{
  this.timeData.unstore(this.t); // remove "future" info
  this.conn.receive(-1);
  this.t--;
  this.dt = -1;
};

/**
 * Move one step forward in time. Information 
 * will be used for the next call to propagate().
 */ 
DynFullJoint.prototype.next = function () 
{
  this.timeData.unstore(this.t - this.window);  // remove outdated info
  this.timeData.store(this.t, 'fjpt', this.fjpt.values.slice()); // store info
  this.conn.receive(1);
  this.t++;
  this.dt = 1;
};

// JUNCTION TREE PROPAGATOR FOR DYNAMIC BAYESIAN NETWORKS ===========

/**
 * Constructs a dynamic junction tree, which is usable for far
 * more complex networks than the dynamic full joint propagator.
 * @param net     bayesian network to work on
 * @param restr   minimum allowed order for all or some nodes 
 *                (restr[nodeId]) (either do not set restrictions 
 *                for dynamic nodes, or make sure that they are 
 *                eliminated after all other nodes)
 * @param window  time window size (max number of steps for 
 *                back-propagation)
 * @param fillin  put fill-in links for moralization and 
 *                triangulation into the original bayesian 
 *                network? [optional]
 * @param max     max-propagation?
 */
function DynJTree (net, restr, window, fillin, max)
{
  if (net != undefined) {
    this.initDynJTree(net, restr, window, fillin, max);
  }
}

/* Inheritance */
DynJTree.prototype             = new JTree();
DynJTree.prototype.parent      = JTree.prototype;
DynJTree.prototype.constructor = DynJTree;

/**
 * Compile / Recompile the given bayesian network 
 * (see constructor {@link DynJTree}).
 */
DynJTree.prototype.initDynJTree = function (net, restr, window, fillin, max) 
{
  // manipulate restrictions: all dynamic nodes should be 
  // eliminated after all other nodes (such that the root 
  // clique will contain all dynamic nodes (pred and succ))
  var o = 0; 
  for (var nId in net.nodes) o++;
  for (var nId in net.pred)  if (restr[nId] == undefined) o--;
  for (var nId in net.succ)  if (restr[nId] == undefined) o--;
  for (var nId in net.pred)  if (restr[nId] == undefined) restr[nId] = o;
  for (var nId in net.succ)  if (restr[nId] == undefined) restr[nId] = o;

  // init static junction tree
  this.initJTree(net, restr, fillin, max);

  // special dynamic attributes
  this.window   = window;            // time window size (max. steps of back-propagation)
  this.t        = 0;                 // current time slice index (0,1,2,...)
  this.dt       = 0;                 // 1:forward, -1:backward, 0:no change in time
  this.timeData = new TimeData();    // clique/separator table value arrays stored for previous time slices
  this.conn     = new DynConnector(this.net, this.root.jpt); // dynamic connector
};

/**
 * Re-load probabilities from the bayesian network
 * (probabilities of some nodes might have changed
 * or the number of states of one dimensions might 
 * have changed).
 */
DynJTree.prototype.reload = function () 
{
  this.parent.reload.call(this);
  this.t        = 0;                 // current time slice index (0,1,2,...)
  this.dt       = 0;                 // 1:forward, -1:backward, 0:no change in time
  this.timeData = new TimeData();    // clique/separator table value arrays stored for previous time slices
  this.conn     = new DynConnector(this.net, this.root.jpt); // dynamic connector
};

/**
 * Fully recreate this propagator.
 */
DynJTree.prototype.recreate = function () 
{
  this.initDynJTree(this.net, this.restr, this.window, this.fillin);
};

/**
 * Propagate (and update a-posteriori probability of all nodes).
 * @param observe  use current evidence?
 * @param update   update posterior probability tables of all nodes?
 * @returns        probability of evidence given the model:    P(e|M)<br>
 *                 or of the most probable config. (max-prop): P(C|e)
 */ 
DynJTree.prototype.propagate = function (observe, update) 
{
  // re-initialize and include evidence
  this.init();
  this.conn.send(this.dt);        // multiply in information from last/next time slice
  if (observe) this.observe();    // multiply in evidence

  // propagate (HUGIN scheme)
  this.collect(this.root);        // collect phase
  this.pe = this.root.jpt.norm(); // normalize root (gives p(e|M))
  this.distribute(this.root);     // distribute phase

  // update a-posteriori probabilities of all nodes
  if (update) this.update();

  return (this.pe);
};

/**
 * Re-initialize full joint probability table for next
 * propagation (time-slice).
 */ 
DynJTree.prototype.init = function () 
{
  if (this.timeData.isStored(this.t)) {

    // re-use stored configuration
    for (var cId in this.nodes) {
      var clique = this.nodes[cId];
      clique.jpt.values = this.timeData.stored(this.t, clique.id);
      for (var chId in this.children[cId]) {
        var sep = this.children[cId][chId];
        (this.max 
            ? sep.jpt.mmarg(clique.jpt)
            : sep.jpt.marg(clique.jpt));
      }
    }

  } else {

    // re-initialize (using initData)
    for (var cId in this.nodes) {
      var clique = this.nodes[cId];
      clique.jpt.values = this.initData.stored(clique.id).slice();
      for (var chId in this.children[cId]) {
        var sep = this.children[cId][chId];
        sep.jpt.init(1);
      }
    }

  }
};

/**
 * Move back to first slice. Information 
 * will be used for the next call to propagate().
 */ 
DynJTree.prototype.stop = function () 
{
  this.timeData.initTimeData();
  this.dt = 0;
  this.t  = 0;
};

/**
 * Move one step backward in time. Information 
 * will be used for the next call to propagate().
 */ 
DynJTree.prototype.prev = function () 
{
  // remove "future" info
  this.timeData.unstore(this.t); 

  // store dynamic info to be send to the next time slice
  this.conn.receive(-1);
  this.t--;
  this.dt = -1;
};

/**
 * Move one step forward in time. Information 
 * will be used for the next call to propagate().
 */ 
DynJTree.prototype.next = function () 
{
  // remove outdated info
  this.timeData.unstore(this.t - this.window); 

  // store joint probability tables of all cliques
  // (separator tables are restored by marginalization 
  //  from the jpt of any of the two connected cliques)
  for (var cId in this.nodes) {
    this.timeData.store(this.t, cId, this.nodes[cId].jpt.values.slice()); 
  }

  // store dynamic info to be send to the next time slice
  this.conn.receive(1);
  this.t++;
  this.dt = 1;
};
