/*
 * -------------------------------------------------------------------------
 *
 *  pn4webDisplayState.js Copyright (C) 2011, 
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *   Author: Patrick Rammelt 
 *   E-Mail: patrick.rammelt@go4more.de
 *   Site:   http://www.patricks-seite.de
 * -------------------------------------------------------------------------
 *
 * Displays states of (discrete) node in a bayesian network (equivalent to
 * "pn4webDisplayContent.js", which displays the names of nodes contained
 * in a clique of a junction tree).
 */


// DISPLAY STATES ===================================================

/**
 * Display or redisplay the states for the given node.
 * @param net   bayesian network
 * @param node  node in that network
 * @param type  e.g. this.STATE
 */
PN4Web.prototype.displayStates = function (net, node, type)
{
  // id for the node html element
  var id = net.id + '.' + node.id;

  // container elements
  var pbarsElem      = document.getElementById(id + '.state_pbars');
  var pvalsElem      = document.getElementById(id + '.state_pvals');
  var stateNamesElem = document.getElementById(id + '.state_names');
  var sensorsElem    = document.getElementById(id + '.state_sensors');

  // probability bar type for possible states
  var pbarNormType = (node.f.min() < 1.0 ? type.PBAR_ACTIVE : type.PBAR);

  for (var s = 0; s < node.states.length; s++) {
    var state = node.states[s];

    // get / create probability bar
    var pbarType = (node.f.values[s] ? pbarNormType : type.PBAR_INACTIVE);
    var pbarId   = id + '.state' + s + '.pbar';
    var pbarElem = document.getElementById(pbarId);
    if (!pbarElem) {
      pbarElem = appendNewElement(pbarsElem, 'div', pbarId, type.CLAZZ + '_pbar_' + s);
      this.mkStateSensitive(net, node, pbarElem, s); // (for IE sensors do not work)
    }

    // set border image
    setBorderImage(pbarElem, this.BASE_PATH + pbarType.BG, 
                   pbarType.SPLIT.N,  pbarType.SPLIT.E,  pbarType.SPLIT.S,  pbarType.SPLIT.W, 
                   pbarType.BORDER.N, pbarType.BORDER.E, pbarType.BORDER.S, pbarType.BORDER.W, 
                   pbarType.STRETCH);
    
    // set pbar width
    var w = toInt(type.MAX_PBAR_WIDTH * Math.max(0, Math.min(1, node.p.get(s))));
    if (!isNaN(w)) pbarElem.style.width = w;
    
    // get / create state name
    var stateNameId = id + '.state' + s + '.name';
    var stateNameElem = document.getElementById(stateNameId);
    if (!stateNameElem) {
      stateNameElem = appendNewElement(stateNamesElem, 'div', stateNameId, type.CLAZZ + '_name_' + s); 
      this.mkStateSensitive(net, node, stateNameElem, s); // (for IE sensors do not work)
    }
    stateNameElem.innerHTML = state.name.toHTML();
    
    // get / create probability value
    var pvalId = id + '.state' + s + '.pval';
    var pvalElem = document.getElementById(pvalId);
    if (!pvalElem) {
      pvalElem = appendNewElement(pvalsElem, 'div', pvalId, type.CLAZZ + '_pval_' + s); 
      this.mkStateSensitive(net, node, pvalElem, s); // (for IE sensors do not work)
    }
    pvalElem.innerHTML = node.p.get(s).toFixed(type.PRECISION);
    
    // get / create probability value
    var sensorId = id + '.state' + s + '.sensor';
    var sensorElem = document.getElementById(sensorId);
    if (!sensorElem) {
      sensorElem = appendNewElement(sensorsElem, 'div', sensorId, type.CLAZZ + '_sensor_' + s); 
      this.mkStateSensitive(net, node, sensorElem, s); // (couldn't be done "inline")
    }
  }

  // remove additional states not included anymore
  for (var s = node.states.length; ; s++) {
    var pbarElem      = document.getElementById(id + '.state' + s + '.pbar');
    var pvalElem      = document.getElementById(id + '.state' + s + '.pval');
    var stateNameElem = document.getElementById(id + '.state' + s + '.name');
    var sensorElem    = document.getElementById(id + '.state' + s + '.sensor');
    if (pbarElem || pvalElem || stateNameElem || sensorElem) {
      pbarElem.parentNode.removeChild(pbarElem);
      pvalElem.parentNode.removeChild(pvalElem);
      stateNameElem.parentNode.removeChild(stateNameElem);
      sensorElem.parentNode.removeChild(sensorElem);
    } else {
      break;
    }
  }
};

/**
 * Make sensor for state s sensitive for clicks
 * @param net         bayesian network
 * @param node        node
 * @param sensorElem  html sensor element for state s
 * @param s           state index
 */
PN4Web.prototype.mkStateSensitive = function (net, node, sensorElem, s)
{
  if (!(net.succ && net.succ[node.id])) { // not for predecessor nodes!
    var me = this;
    var set = function(evnt) {
      if (!evnt) evnt = window.event;
      preventDefault(evnt);
      
      // set evidence
      if (net.rate == undefined) {
        if (node.f.min() == 1) node.f.init(0);         // all possible -> all impossible
        node.f.values[s] = (node.f.values[s] ? 0 : 1); // toggle evidence for state s
        if (node.f.max() == 0) node.f.init(1);         // all impossible -> all possible
      } else {
        if (node.f.values[s] == 1 && node.f.min() < 1) node.f.init(1); // remove evidence
        else { node.f.init(0); node.f.values[s] = 1; } // set one state
      }

      // propagate and update
      me.propagate(net);
    };
    addEventListener(sensorElem, 'mousedown', set);
  }
};


