/*
 * -------------------------------------------------------------------------
 *
 *  pn4webDisplayPopup.js Copyright (C) 2011, 
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *   Author: Patrick Rammelt 
 *   E-Mail: patrick.rammelt@go4more.de
 *   Site:   http://www.patricks-seite.de
 * -------------------------------------------------------------------------
 *
 * Helper to display popup windows, e.g. used by "pn4webDisplayPTPopup.js".
 */


// DISPLAY POPUP WINDOW =============================================

/**
 * Create a popup window, which is invisible first. To make it 
 * visible and set its position use 'displayPopup2' after you
 * have added a content element to the mid-segment of this popup 
 * window.
 * @param id          id for the popup
 * @param title       title
 * @param parentElem  parent html element
 * @param type        popup type (e.g. this.POPUP)
 */
PN4Web.prototype.displayPopup1 = function (id, title, parentElem, type)
{
  // create popup window element and sub-elements
  var popupElem =   appendNewElement(parentElem, 'div', id,             type.CLAZZ,            true); 
  var shElem    =   appendNewElement(popupElem,  'div', id + '.shadow', type.CLAZZ + '_shadow');
  var bgElem    =   appendNewElement(popupElem,  'div', id + '.bg',     type.CLAZZ + '_bg');
  var titleElem =   appendNewElement(popupElem,  'div', id + '.title',  type.CLAZZ + '_title');
  var closeElem =   appendNewElement(popupElem,  'div', id + '.close',  type.CLAZZ + '_close');
  /*var areaElem=*/ appendNewElement(popupElem,  'div', id + '.area',   type.CLAZZ + '_area');
    
  // set border images (background image)
  setBorderImage(shElem, this.BASE_PATH + type.SHADOW, 
                 type.SPLIT.N,  type.SPLIT.E,  type.SPLIT.S,  type.SPLIT.W, 
                 type.BORDER.N, type.BORDER.E, type.BORDER.S, type.BORDER.W, 
                 type.STRETCH);
  setBorderImage(bgElem, this.BASE_PATH + type.BG, 
                 type.SPLIT.N,  type.SPLIT.E,  type.SPLIT.S,  type.SPLIT.W, 
                 type.BORDER.N, type.BORDER.E, type.BORDER.S, type.BORDER.W, 
                 type.STRETCH);

  // title and close link
  titleElem.innerHTML = title.toHTML();
  closeElem.innerHTML = '<a href="javascript:dummy();">[close]</a>';

  // make this popup grab and dragable
  this.mkPopupSensitive(id, new Array(bgElem, titleElem, closeElem));
  
  // react an close
  this.mkPopupCloseSensitive(id, closeElem.getElementsByTagName('a')[0]);
  
  return (popupElem);
};


/**
 * Set size and position of a popup window
 * @param id           id for the popup
 * @param contentElem  element(s) added to the mid-segment of this 
 *                     popup window (resize this popup accordingly)
 *                     either a single element or an array of elements
 * @param x, y         desired coordinates (might be adjusted to 
 *                     fit into the display area of the network)
 * @param type         popup type (e.g. this.POPUP)
 */
PN4Web.prototype.displayPopup2 = function (id, contentElem, x, y, type)
{
  // relevant elements
  var popupElem  = document.getElementById(id);
  var bgElem     = document.getElementById(id + '.bg');
  var shElem     = document.getElementById(id + '.shadow');
  var areaElem   = document.getElementById(id + '.area');
  var titleElem  = document.getElementById(id + '.title');
  var closeElem  = document.getElementById(id + '.close');

  // set initial position (will be adjusted, see below)
  //popupElem.style.left = 0;
  //popupElem.style.top  = 0;
  
  // set width and height
  var me = this;
  var setSize = function () {
    var w = 0, h = 0;
    if (contentElem instanceof Array) {
      if (contentElem.length == 0) { 
        w = type.MIN_WIDTH; 
        h = type.MIN_HEIGHT; 
      } else {
        for (var i = 0; i < contentElem.length; i++) {
          w  = Math.max(w, contentElem[i].offsetWidth); // maximum width
          h += contentElem[i].offsetHeight;             // cumulated height
        }
      }
    } else {
      w = contentElem.offsetWidth;
      h = contentElem.offsetHeight;
    }
    if (w > 0 && h > 0) {
      
      // set background shadow size and title/close widths
      w = Math.min(type.MAX_WIDTH,  w);
      h = Math.min(type.MAX_HEIGHT, h);
      bgElem.style.width    = w;
      bgElem.style.height   = h;
      shElem.style.width    = w;
      shElem.style.height   = h;
      areaElem.style.width  = w;
      areaElem.style.height = h;
      titleElem.style.width = w;
      closeElem.style.width = w;

      // set popup size 
      w += type.BORDER.W + type.BORDER.E;
      h += type.BORDER.N + type.BORDER.S;
      popupElem.style.width  = w;
      popupElem.style.height = h;
      
      // adjust position
      var frameElem = popupElem.parentNode;
      var xMax = frameElem.scrollWidth  - w - me.SHADOW.XOFFSET;
      var yMax = frameElem.scrollHeight - h - me.SHADOW.YOFFSET;
      popupElem.style.left = Math.max(0, Math.min(xMax, x));
      popupElem.style.top  = Math.max(0, Math.min(yMax, y));
      
      // add scrollbars (if necessary)
      if (areaElem.childNodes[0]) {
        vscroll(id, id + '.area', areaElem.childNodes[0].id, SCROLL.PN4W); 
        hscroll(id, id + '.area', areaElem.childNodes[0].id, SCROLL.PN4W); 
      }
      
      // finally make it visible
      popupElem.style.visibility = 'visible'; 
      return (true);
    } else {
      return (false);
    }
  };
  waitFor(setSize, 10, 200);
};

/**
 * Make popup window sensitive for grab and drag.
 * @param id     id of the popup element
 * @param elems  array of sensitive elements
 */
PN4Web.prototype.mkPopupSensitive = function (id, elems)
{
  var me = this;
  var grab = function(evnt) { 
    var outer = document.getElementById(id);
    var inner = document.getElementById(id + '.bg');
    me.DRAG.drag    = function(e) { me.drag(outer, inner, e); };
    me.DRAG.release = function(e) { me.release(outer, inner, e); };
    me.grab(outer, inner, evnt);
  };
  for (var i = 0; i < elems.length; i++) {
    addEventListener(elems[i], 'mousedown', grab);
  }
};

/**
 * Make closeElem react on clicks by closing the popup window.
 * @param id    id of the popup element
 * @param elem  sensitive element (react on clicks on that element)
 */
PN4Web.prototype.mkPopupCloseSensitive = function (id, elem)
{
  var close = function(evnt) { 
    if (!evnt) evnt = window.event;
    preventDefault(evnt);
    removeElement(id);
  };
  addEventListener(elem, 'mousedown', close);
};
