/*
 * -------------------------------------------------------------------------
 *
 *  pn4webDisplayPTable.js Copyright (C) 2011, 
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *   Author: Patrick Rammelt 
 *   E-Mail: patrick.rammelt@go4more.de
 *   Site:   http://www.patricks-seite.de
 * -------------------------------------------------------------------------
 *
 * Helper to display a probability table (conditional or joint probability  
 * tables), e.g. used by "pn4webDisplayPTPopup.js".
 */


// DISPLAY PROBABILITY TABLES =======================================

/**
 * Display a probability table.
 * @param pt          probability table ('PTable')
 * @param id          id for the table
 * @param parentElem  HTML parent element to add a new table
 * @param type        probability table type (e.g. this.PTABLE)
 * @param context     additional information which is the first 
 *                    parameter for any call to a callback function
 * @param callbacks   callback functions (e.g. CPT.CALLBACKS)
 * @param dims        map id -> dimension for all available dimensions
 *                    (only if dimensions should be addable, i.e.
 *                    if there is also a callback 'callbacks.DIM.ADD')
 */
PN4Web.prototype.displayPTable = function (pt, id, parentElem, type, context, callbacks, dims) 
{
  // create table
  var tabElem  = appendNewElement(parentElem, 'table', id, type.CLAZZ);
  tabElem.cellSpacing = 0;     // (IE ignores CSS border-spacing)
  var bodyElem = appendNewElement(tabElem, 'tbody');
  var rowElem, cellElem;
  var numOfStates = (pt.dims.length > 0 ? pt.dims[0].states.length : 1);
  
  // very first row: dummies to fix column widths + additional first/
  // last column for buttons to add parents and to add/remove states
  rowElem = appendNewElement(bodyElem, 'tr');
  if (dims && callbacks && callbacks.DIM && callbacks.DIM.ADD) {
    cellElem = appendNewElement(rowElem, 'td', undefined, type.CLAZZ + '_shape');
    cellElem.rowSpan = (pt.n / numOfStates) + 5; 
    this.displayButton(cellElem, id + '.dim.add', type.CLAZZ + '_dim_add', type.ADD_PARENT, type.ADD_PARENT_HL, "Add new parent");
    this.mkAddDimSensitive(context, pt, id, callbacks.DIM.ADD, dims, type); 
  }
  if (type.ISCOMPLEX) {
    for (var d = pt.dims.length-1; d >= 0; d--) { // for all dimensions
      appendNewElement(rowElem, 'td', undefined, type.CLAZZ + '_dummy');
      appendNewElement(rowElem, 'td', undefined, type.CLAZZ + '_vsep');
    }
    appendNewElement(rowElem, 'td', undefined, type.CLAZZ + '_dummy');
  } else {
    for (var d = pt.dims.length-1; d > 0; d--) {  // for all parent nodes
      appendNewElement(rowElem, 'td', undefined, type.CLAZZ + '_dummy');
      appendNewElement(rowElem, 'td', undefined, type.CLAZZ + '_vsep');
    }
    for (var s = 0; s < numOfStates; s++) {
      appendNewElement(rowElem, 'td', undefined, type.CLAZZ + '_dummy');
    }
  }
  if (callbacks && callbacks.STATE) {
    cellElem = appendNewElement(rowElem, 'td', undefined, type.CLAZZ + '_size');
    cellElem.rowSpan = (pt.n / numOfStates) + 5; 
    if (callbacks.STATE.ADD) {
      this.displayButton(cellElem, id + '.state.add', type.CLAZZ + '_state_add', type.ADD_STATE, type.ADD_STATE_HL, "Add new state");
      this.mkAddStateSensitive(context, pt, id, callbacks.STATE.ADD);
    }
    if (callbacks.STATE.ADD && callbacks.STATE.RM) appendNewElement(cellElem, 'br');
    if (callbacks.STATE.RM) {
      this.displayButton(cellElem, id + '.state.rm',  type.CLAZZ + '_state_rm',  type.RM_STATE, type.RM_STATE_HL, "Remove last state");
      this.mkRmStateSensitive(context, pt, id,  callbacks.STATE.RM);
    }
  }

  // first (visible) row: node/dimension names
  rowElem = appendNewElement(bodyElem, 'tr');
  if (type.ISCOMPLEX) {
    for (var d = pt.dims.length-1; d >= 0; d--) { // for all dimensions
      cellElem = appendNewElement(rowElem, 'td', undefined, type.CLAZZ + '_header');
      cellElem.innerHTML = pt.dims[d].name.toHTML();
      appendNewElement(rowElem, 'td', undefined, type.CLAZZ + '_vsep');
    }
    cellElem = appendNewElement(rowElem, 'td', id + '.dim', type.CLAZZ + '_header');
    cellElem.innerHTML = type.CONTENT_TYPE_NAME.toHTML();
  } else {
    for (var d = pt.dims.length-1; d > 0; d--) {  // for all parent nodes
      cellElem = appendNewElement(rowElem, 'td', undefined, type.CLAZZ + '_header');
      cellElem.innerHTML = pt.dims[d].name.toHTML();
      appendNewElement(rowElem, 'td', undefined, type.CLAZZ + '_vsep');
    }
    cellElem = appendNewElement(rowElem, 'td', id + '.dim', type.CLAZZ + '_header');
    cellElem.colSpan   = numOfStates;
    cellElem.innerHTML = (pt.dims.length > 0 ? pt.dims[0].name.toHTML() : "");
  }
  if (callbacks && callbacks.DIM && callbacks.DIM.NAME) {
    this.mkDimNameSensitive(context, pt, id, callbacks.DIM.NAME, type.MAX_DIM_NAME_LENGTH); 
  }

  // second (visible) row: state names
  rowElem = appendNewElement(bodyElem, 'tr');
  if (type.ISCOMPLEX) {
    // skip that row
  } else {
    for (var d = pt.dims.length-1; d > 0; d--) {  // for all parent nodes
      cellElem = appendNewElement(rowElem, 'td', undefined, type.CLAZZ + '_header');
      if (callbacks && callbacks.DIM && callbacks.DIM.RM) {
        this.displayButton(cellElem, id + '.dim_' + d + '.rm', type.CLAZZ + '_dim_rm', type.RM_PARENT, type.RM_PARENT_HL, "Disconnect this parent");
        this.mkRmDimSensitive(context, pt, d, id, callbacks.DIM.RM);
      }
      appendNewElement(rowElem, 'td', undefined, type.CLAZZ + '_vsep');
    }
    for (var s = 0; s < numOfStates; s++) { // s has to be a number here (in for (s in ...) s is a string)
      cellElem = appendNewElement(rowElem, 'td', id + '.state_' + s, type.CLAZZ + '_state');
      cellElem.innerHTML = (pt.dims.length > 0 ? pt.dims[0].states[s].name.toHTML() : "");
      if (callbacks && callbacks.STATE && callbacks.STATE.NAME) {
        this.mkStateNameSensitive(context, pt, s, id, callbacks.STATE.NAME, type.MAX_STATE_NAME_LENGTH); 
      }
    }
  }
  
  // third (visible) row: horizontal line
  if (type.ISCOMPLEX) {
    rowElem = appendNewElement(bodyElem, 'tr');
    for (var d = pt.dims.length-1; d >= 0; d--) { // for all dimensions
      appendNewElement(rowElem, 'td', undefined, type.CLAZZ + '_hsep'); 
      appendNewElement(rowElem, 'td', undefined, type.CLAZZ + '_sep');
    }
    sepElem = appendNewElement(rowElem, 'td', undefined, type.CLAZZ + '_hsep');
  } else {
    rowElem = appendNewElement(bodyElem, 'tr');
    for (var d = pt.dims.length-1; d > 0; d--) {  // for all parent nodes
      appendNewElement(rowElem, 'td', undefined, type.CLAZZ + '_hsep'); 
      appendNewElement(rowElem, 'td', undefined, type.CLAZZ + '_sep');
    }
    sepElem = appendNewElement(rowElem, 'td', undefined, type.CLAZZ + '_hsep');
    sepElem.colSpan = numOfStates;
  }
  
  // normal rows (parent states + values for all states of the current node)
  var d    = Math.max(0, pt.dims.length - 1);
  var conf = new Array(d);
  if (type.ISCOMPLEX) {
    this.displayPTableComplexRows(pt, d, 0, conf, id, bodyElem, null, type, context, callbacks);
  } else {
    this.displayPTableRows(pt, d, 0, conf, id, bodyElem, null, type, context, callbacks);
  }

  // last row: horizontal line
  if (type.ISCOMPLEX) {
    rowElem = appendNewElement(bodyElem, 'tr');
    for (var d = pt.dims.length-1; d >= 0; d--) { // for all dimensions
      appendNewElement(rowElem, 'td', undefined, type.CLAZZ + '_hsep'); 
      appendNewElement(rowElem, 'td', undefined, type.CLAZZ + '_sep');
    }
    sepElem = appendNewElement(rowElem, 'td', undefined, type.CLAZZ + '_hsep');
  } else {
    rowElem = appendNewElement(bodyElem, 'tr');
    for (var d = pt.dims.length-1; d > 0; d--) {  // for all parent nodes
      appendNewElement(rowElem, 'td', undefined, type.CLAZZ + '_hsep'); 
      appendNewElement(rowElem, 'td', undefined, type.CLAZZ + '_sep');
    }
    sepElem = appendNewElement(rowElem, 'td', undefined, type.CLAZZ + '_hsep');
    sepElem.colSpan = numOfStates;
  }

  return (tabElem);
};

/**
 * Display value-rows of a probability table, i.e. all values
 * are numeric (recursively working private helper function).
 * @param pt         probability table ('PTable')
 * @param d          current dimension index in this table 
 *                   (init: pt.dims.length-1)
 * @param i          current element index in this table (init: 0)
 * @param conf       parent state configuration
 * @param id         id of this table
 * @param bodyElem   HTML table element to add new rows
 * @param rowElem    current row element (init: null/undefined)
 * @param type       e.g. this.PTTABLE
 * @param context    first parameter for each callback-function
 * @param callbacks  callbacks
 */
PN4Web.prototype.displayPTableRows = function (pt, d, i, conf, id, 
                                               bodyElem, rowElem, type,
                                               context, callbacks) 
{
  var dim = pt.dims[d];
  if (d == 0) {
    var cellElem, valueElem;

    // start new row
    rowElem = appendNewElement(bodyElem, 'tr');

    // display parent state names
    for (d = pt.dims.length-1; d > 0; d--) {
      var s = conf[d];
      cellElem = appendNewElement(rowElem, 'td', undefined, type.CLAZZ + '_state');
      cellElem.innerHTML = pt.dims[d].states[s].name.toHTML();
      appendNewElement(rowElem, 'td', undefined, type.CLAZZ + '_vsep');
    }

    // display values for all states of this node/dimension
    var numOfStates = (dim ? dim.states.length : 1);
    for (var s = 0; s < numOfStates; s++) {
      cellElem  = appendNewElement(rowElem, 'td', undefined, type.CLAZZ + '_value');
      valueElem = appendNewElement(cellElem, 'div', id + '.cell_' + i, 'container');
      
      // probability bar
      if (type.PBAR) {
        var pbarElem  = appendNewElement(valueElem, 'div', id + '.cell_' + i + '.pbar',  type.PBAR.CLAZZ);
        setBorderImage(pbarElem, this.BASE_PATH + type.PBAR.BG, 
                       type.PBAR.SPLIT.N,  type.PBAR.SPLIT.E,  type.PBAR.SPLIT.S,  type.PBAR.SPLIT.W, 
                       type.PBAR.BORDER.N, type.PBAR.BORDER.E, type.PBAR.BORDER.S, type.PBAR.BORDER.W, 
                       type.PBAR.STRETCH);
        pbarElem.style.width = toInt(type.MAX_PBAR_WIDTH * Math.max(0, Math.min(1, pt.values[i]))) + 'px';
      }
      
      // value
      var valElem = appendNewElement(valueElem, 'div', id + '.cell_' + i + '.value', 'container');
      valElem.innerHTML = pt.values[i].toFixed(type.PRECISION);

      // make editable on double click
      if (callbacks && callbacks.CELL && callbacks.CELL.VALUE) {
        this.mkCellValueSensitive(context, pt, i, id, callbacks.CELL.VALUE, type.MAX_PBAR_WIDTH, type.PRECISION);
      }

      i += (dim ? pt.step(dim) : 0);
    }

  } else {

    // over all states of this (parent) node/dimension
    for (var s = 0; s < dim.states.length; s++) {    
      conf[d] = s;
      this.displayPTableRows(pt, d-1, i, conf, id, bodyElem, rowElem, 
                             type, context, callbacks);
      i += pt.step(dim);
    }

  }
};


/**
 * Display value-rows of a table containing complex objects
 * instead of simple numbers (recursively working "private"
 * helper function)
 * @param pt         probability table ('PTable')
 * @param d          current dimension index in this table 
 *                   (init: pt.dims.length-1)
 * @param i          current element index in this table (init: 0)
 * @param conf       parent state configuration
 * @param id         id of this table
 * @param bodyElem   HTML table element to add new rows
 * @param rowElem    current row element (init: null/undefined)
 * @param type       e.g. this.PTTABLE
 * @param context    first parameter for each callback-function
 * @param callbacks  callbacks
 */
PN4Web.prototype.displayPTableComplexRows = function (pt, d, i, conf, id, 
                                                      bodyElem, rowElem, type,
                                                      context, callbacks) 
{
  if (d < 0) {
    var cellElem, valueElem;

    // start new row
    rowElem = appendNewElement(bodyElem, 'tr');

    // display parent state names
    for (d = pt.dims.length-1; d >= 0; d--) {
      var s = conf[d];
      cellElem = appendNewElement(rowElem, 'td', undefined, type.CLAZZ + '_state');
      cellElem.innerHTML = pt.dims[d].states[s].name.toHTML();
      appendNewElement(rowElem, 'td', undefined, type.CLAZZ + '_vsep');
    }

    // display value in current cell
    cellElem  = appendNewElement(rowElem, 'td', undefined, type.CLAZZ + '_value');
    valueElem = appendNewElement(cellElem, 'div', id + '.cell_' + i, 'container'); 
    valueElem.innerHTML 
      = '<div class="container" id="' + id + '.cell_' + i + '.value">' 
      +   pt.values[i].toHTML()
      + '</div>';

    // make editable on double click
    if (callbacks && callbacks.CELL && callbacks.CELL.VALUE) {
      this.mkCellValueSensitive(context, pt, i, id, callbacks.CELL.VALUE, type.MAX_PBAR_WIDTH, type.PRECISION);
    }

  } else {

    // over all states of this (parent) node/dimension
    var dim = pt.dims[d];
    for (var s = 0; s < dim.states.length; s++) {    
      conf[d] = s;
      this.displayPTableComplexRows(pt, d-1, i, conf, id, bodyElem, rowElem, 
                                    type, context, callbacks);
      i += pt.step(dim);
    }

  }
};

// EDIT DIMENSION (NODE) NAMES --------------------------------------

/**
 * Makes dimension names sensitive on double clicks.
 * @param context   first parameter for the callback
 * @param pt        probability table
 * @param id        id-prefix of the element to make sensitive
 * @param callback  function to be called if a dimension-name 
 *                  has been changed by the user (editing)
 * @param maxChars  max allowed name length
 */
PN4Web.prototype.mkDimNameSensitive = function (context, pt, id, callback, maxChars) 
{
  var me   = this;
  var elem = document.getElementById(id + '.dim');
  var edit = function(evnt) {         // handler to make cell editable
    if (!evnt) evnt = window.event;
    preventDefault(evnt);
    me.editDimName(context, pt, id, callback, maxChars); 
  };
  addEventListener(elem, 'dblclick', edit);
};

/**
 * Edit dimension name
 * @param context   first parameter for the callback
 * @param pt        probability table
 * @param id        id-prefix of the element to make sensitive
 * @param callback  function to be called if a dimension-name 
 *                  has been changed by the user (editing)
 * @param maxChars  max allowed name length
 */
PN4Web.prototype.editDimName = function (context, pt, id, callback, maxChars) 
{
  var elem = document.getElementById(id + '.dim');

  // handler which sets the value, updates the 
  // probability bar and calls the callback function
  var setValue = function(evnt, silent) {
    if (!evnt) evnt = window.event;
    try { preventDefault(evnt); } catch (e) {}
    var v = elem.childNodes[0].value;
    if (pt.dims.length > 0) pt.dims[0].name = v;
    removeEventListener(elem.childNodes[0], 'blur', setValue); // (important for Mozilla)
    removeEventListener(elem.childNodes[0], 'keydown', leave);
    elem.innerHTML = (pt.dims.length > 0 ? pt.dims[0].name : ""); // replace input element by text
    if (!silent) {
      callback(context, pt);
    }
  };

  // handler which reacts on some keys by leaving the cell
  var leave = function(evnt) {
    if (!evnt) evnt = window.event;
    if (evnt.which == 13 || evnt.keyCode == 13) {                                   // return
      preventDefault(evnt);
      setValue(evnt, false);
    }
  };

  // create input element
  elem.innerHTML 
  = '<input type="text" maxLength="' + maxChars + '" '
  +         'value="' + (pt.dims.length > 0 ? pt.dims[0].name : "") + '">';
  elem.childNodes[0].style.width  = elem.offsetWidth;
  elem.childNodes[0].style.height = elem.offsetHeight;
  addEventListener(elem.childNodes[0], 'blur', setValue);
  addEventListener(elem.childNodes[0], 'keydown', leave);
  setTimeout(function(){ elem.childNodes[0].focus(); }, 10); // IE needs a delay
};


// EDIT STATE NAMES -------------------------------------------------

/**
 * Makes state-names sensitive on double clicks
 * @param context   first parameter for the callback
 * @param pt        probability table
 * @param s         index of the corresponding state
 * @param id        id-prefix of the element to make sensitive
 * @param callback  function to be called if a dimension-name 
 *                  has been changed by the user (editing)
 * @param maxChars  max allowed name length
 */
PN4Web.prototype.mkStateNameSensitive = function (context, pt, s, id, callback, maxChars) 
{
  var me = this;
  var elem = document.getElementById(id + '.state_' + s);
  var edit = function(evnt) {     // handler to make cell editable
    if (!evnt) evnt = window.event;
    preventDefault(evnt);
    me.editStateName(context, pt, s, id, callback, maxChars); 
  };
  addEventListener(elem, 'dblclick', edit);
};

/**
 * Edit state name.
 * @param context   first parameter for the callback
 * @param pt        probability table
 * @param s         index of the corresponding state
 * @param id        id-prefix of the element to make sensitive
 * @param callback  function to be called if a dimension-name 
 *                  has been changed by the user (editing)
 * @param maxChars  max allowed name length
 */
PN4Web.prototype.editStateName = function (context, pt, s, id, callback, maxChars) 
{
  var me   = this;
  var elem = document.getElementById(id + '.state_' + s);

  // handler which sets the value, updates the 
  // probability bar and calls the callback function
  var setValue = function(evnt, silent) {
    if (!evnt) evnt = window.event;
    try { preventDefault(evnt); } catch (e) {}
    var v = elem.childNodes[0].value;
    if (pt.dims.length > 0) pt.dims[0].states[s].name = v;
    removeEventListener(elem.childNodes[0], 'blur', setValue); // (important for Mozilla)
    removeEventListener(elem.childNodes[0], 'keydown', leave);
    elem.innerHTML = (pt.dims.length > 0 ? pt.dims[0].states[s].name : ""); // replace input element by text
    if (!silent) callback(context, pt, s);
  };

  // handler which reacts on some keys by leaving the cell
  var leave = function(evnt) {
    if (!evnt) evnt = window.event;
    var j = s;
    if (evnt.which == 13 || evnt.keyCode == 13) {                                   // return
      preventDefault(evnt);
      setValue(evnt, false);
    }
    else if (evnt.which ==  9 || evnt.keyCode ==  9) j += 1;                        // tab
    else if (evnt.which == 37 || evnt.keyCode == 37) j -= 1;                        // left
    else if (evnt.which == 39 || evnt.keyCode == 39) j += 1;                        // right
    if (j != s) {
      preventDefault(evnt);
      if (pt.dims.length > 0 && j >= 0 && j < pt.dims[0].states.length) {
        setTimeout(function() { 
          setValue(evnt, true); 
          me.editStateName(context, pt, j, id, callback, maxChars); }, 0);
      }
    }
  };

  // create input element
  elem.innerHTML 
  = '<input type="text" maxLength="' + maxChars + '" '
  +         'value="' + (pt.dims.length > 0 ? pt.dims[0].states[s].name : "") + '">';
  addEventListener(elem.childNodes[0], 'blur', setValue);
  addEventListener(elem.childNodes[0], 'keydown', leave);
  setTimeout(function(){ elem.childNodes[0].focus(); }, 10); // IE needs a delay
};

// EDIT PROBABILITY VALUES ------------------------------------------

/**
 * Makes cell sensitive on double clicks.
 * @param context       first parameter for the callback
 * @param pt            probability table
 * @param i             index of the corresponding table cell
 * @param id            id-prefix of the element to make sensitive
 * @param callback      function to be called if a dimension-name 
 *                      has been changed by the user (editing)
 * @param maxPBarWidth  max width of (the middle segment of) 
 *                      probability bars
 * @param precision     precision of values
 */
PN4Web.prototype.mkCellValueSensitive = function (context, pt, i, id, callback, maxPBarWidth, precision) 
{
  var me = this;
  var elem = document.getElementById(id + '.cell_' + i + '.value');

  // handler to make cell editable
  var edit = function(evnt) {   
    if (!evnt) evnt = window.event;
    preventDefault(evnt);
    me.editCellValue(context, pt, i, id, callback, maxPBarWidth, precision); 
  };
  addEventListener(elem, 'dblclick', edit);
};

/**
 * Edit cell.
 * @param context       first parameter for the callback
 * @param pt            probability table
 * @param i             index of the corresponding table cell
 * @param id            id-prefix of the element to make sensitive
 * @param maxPBarWidth  max width of (the middle segment of) 
 *                      probability bars
 * @param callback      function to be called if a value
 *                      has been changed by the user (editing)
 * @param precision     precision of values
 */
PN4Web.prototype.editCellValue = function (context, pt, i, id, callback, maxPBarWidth, precision) 
{
  var me = this;
  var barElem = document.getElementById(id + '.cell_' + i + '.pbar');
  var valElem = document.getElementById(id + '.cell_' + i + '.value');

  // handler which sets the value, updates the 
  // probability bar and calls the callback function
  var setValue = function(evnt, silent) {
    if (!evnt) evnt = window.event;
    try { preventDefault(evnt); } catch (e) {}
    var v = parseFloat(valElem.childNodes[0].value);
    if (!isNaN(v)) pt.values[i] = v;
    var w = toInt(maxPBarWidth * pt.values[i]);
    removeEventListener(valElem.childNodes[0], 'blur', setValue); // (important for Mozilla)
    removeEventListener(valElem.childNodes[0], 'keydown', leave);
    valElem.innerHTML = pt.values[i].toFixed(precision);          // replace input element by text
    if (barElem) barElem.style.width = w;
    if (!silent) callback(context, pt, i);
  };

  // handler which reacts on some keys by leaving the cell
  var leave = function(evnt) {
    if (!evnt) evnt = window.event;
    var j = i;
    var numOfStates = (pt.dims.length > 0 ? pt.dims[0].states.length : 1);
    if (evnt.which == 13 || evnt.keyCode == 13) {                      // return
      preventDefault(evnt);
      setValue(evnt, false);
    }
    else if (evnt.which ==  9 || evnt.keyCode ==  9) j += 1;           // tab
    else if (evnt.which == 37 || evnt.keyCode == 37) j -= 1;           // left
    else if (evnt.which == 39 || evnt.keyCode == 39) j += 1;           // right
    else if (evnt.which == 38 || evnt.keyCode == 38) j -= numOfStates; // up
    else if (evnt.which == 40 || evnt.keyCode == 40) j += numOfStates; // down
    if (j != i) {
      preventDefault(evnt);
      if (j >= 0 && j < pt.n) {
        setTimeout(function() { 
          setValue(evnt, true); 
          me.editCellValue(context, pt, j, id, callback, maxPBarWidth, precision); }, 0);
      }
    }
  };

  // create input element
  valElem.innerHTML 
  = '<input type="text" maxLength="' + (2 + precision) + '" '
  +         'value="' + pt.values[i].toFixed(precision) + '">';
  addEventListener(valElem.childNodes[0], 'blur', setValue);
  addEventListener(valElem.childNodes[0], 'keydown', leave);
  setTimeout(function(){ valElem.childNodes[0].focus(); }, 10); // IE needs a delay
};

// ADD / REMOVE STATES ----------------------------------------------

/**
 * Make buttons for adding states to the first dimension recating 
 * on clicks.<br> 
 * Notice: This method just calls the callback function!
 * @param context   first parameter for the callback
 * @param pt        probability table
 * @param id        id-prefix of the element to make sensitive
 * @param callback  function to be called to add a new state
 */
PN4Web.prototype.mkAddStateSensitive = function (context, pt, id, callback)
{
  var elem = document.getElementById(id + '.state.add');
  var addState = function(evnt) { 
    if (!evnt) evnt = window.event;
    preventDefault(evnt);
    callback(context, pt);
  };
  addEventListener(elem, 'mousedown', addState);
};

/**
 * Make buttons for removing states from the first dimension 
 * reacting on clicks.<br>
 * Notice: This method just calls the callback function!
 * @param context   first parameter for the callback
 * @param pt        probability table
 * @param id        id-prefix of the element to make sensitive
 * @param callback  function to be called to removes the last state
 */
PN4Web.prototype.mkRmStateSensitive = function (context, pt, id, callback)
{
  var elem = document.getElementById(id + '.state.rm');
  var rmState = function(evnt) { 
    if (!evnt) evnt = window.event;
    preventDefault(evnt);
    callback(context, pt);
  };
  addEventListener(elem,  'mousedown', rmState);
};

// ADD / REMOVE DIMENSIONS (PARENTS) --------------------------------

/**
 * Make buttons for adding a parent dimension reacating on clicks.<br> 
 * Notice: This method just calls the callback function but does 
 *         not actually add the dimension!
 * @param context   first parameter for the callback
 * @param pt        probability table
 * @param id        id-prefix of the element to make sensitive
 * @param callback  function to be called to add a new dimension
 * @param dims      map id -> dimension for all available dimensions
 * @param type      type (e.g. this.PT)
 */
PN4Web.prototype.mkAddDimSensitive = function (context, pt, id, callback, dims, type)
{
  var elem = document.getElementById(id + '.dim.add');
  var showList = function(evnt) { 
    if (!evnt) evnt = window.event;
    preventDefault(evnt);

    // create list
    var buttonElem = document.getElementById(id + '.dim.add');
    if (document.getElementById(id + '.dim.select')) return;
    var popupElem = appendNewElement(document.body, 'div', id + '.dim.select', type.CLAZZ + '_popupmenu');
    popupElem.style.left = Math.max(0, getLeft(buttonElem) - popupElem.offsetWidth);
    popupElem.style.top  = Math.max(0,  getTop(buttonElem));
    var listElem = appendNewElement(popupElem, 'select');
    listElem.size = 1; 

    // handler to hide list when leaving ("onblur") / cancel
    // TODO: autmatically close if CPT is closed
    var hideList = function (evnt) {
      if (!evnt) evnt = window.event;
      preventDefault(evnt);
      popupElem.parentNode.removeChild(popupElem);
    };
    addEventListener(listElem, 'blur', hideList);

    // handler to react on selection: add dimension and hide list again
    var addDim = function (evnt) {
      if (!evnt) evnt = window.event;
      preventDefault(evnt);
      if (listElem.value != "") {
        callback(context, pt, dims[listElem.value]);
      }
      popupElem.parentNode.removeChild(popupElem);
    };
    addEventListener(listElem, 'change', addDim);

    // fill list
    // TODO: show only dimensions which are allowed in a DAG (no cycles!)
    var entryElem = appendNewElement(listElem, 'option');
    entryElem.innerHTML = '---'.toHTML();
    entryElem.value     = '';
    for (var dimId in dims) {
      if (!pt.steps[dimId] && (context.node.isContinuous || !dims[dimId].isContinuous)) {
        var entryElem = appendNewElement(listElem, 'option');
        entryElem.innerHTML = dims[dimId].name.toHTML();
        entryElem.value     = dimId;
      }
    }
    var entryElem = appendNewElement(listElem, 'option');
    entryElem.innerHTML = '[cancel]'.toHTML();
    entryElem.value     = '';

    // set focus
    setTimeout(function(){ listElem.focus(); }, 10); // IE needs a delay
  };
  addEventListener(elem, 'mousedown', showList);
};

/**
 * Make buttons for removing a parent dimension reacting on clicks.<br> 
 * Notice: This method just calls the callback function, but does 
           not actually remove the dimension!
 * @param context   first parameter for the callback
 * @param pt        probability table
 * @param d         index of the dimension in pt which should be removed
 * @param id        id-prefix of the element to make sensitive
 * @param callback  function to be called to removes the dimension
 */
PN4Web.prototype.mkRmDimSensitive = function (context, pt, d, id, callback)
{
  var elem = document.getElementById(id + '.dim_' + d + '.rm');
  var rmDim = function(evnt) { 
    if (!evnt) evnt = window.event;
    preventDefault(evnt);
    callback(context, pt, d);
  };
  addEventListener(elem,  'mousedown', rmDim);
};

