/*
 * -------------------------------------------------------------------------
 *
 *  pn4webDisplayNode.js Copyright (C) 2011, 
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *   Author: Patrick Rammelt 
 *   E-Mail: patrick.rammelt@go4more.de
 *   Site:   http://www.patricks-seite.de
 * -------------------------------------------------------------------------
 *
 * Functionalities for displaying all types of nodes including normal nodes
 * of bayesian networks as well as cliques in junction trees.
 */


// DISPLAY NODES ====================================================

/**
 * Display (or redisplay) the given node
 * @param net   bayesian network / junction tree
 * @param node  one node/clique in that network
 * @param type  e.g. this.NODE, this.CLIQUE
 */
PN4Web.prototype.displayNode = function (net, node, type)
{
  // id for the node html element
  var id = net.id + '.' + node.id;

  // get elements
  var netElem = document.getElementById(net.id);
  var nodeElem = document.getElementById(id);

  // cannot display anything if there is no such element
  if (!netElem) return;

  // create html element
  if (!nodeElem) {
    
    // create divs for the main element, shadow, background and name
    nodeElem      =  appendNewElement(netElem,  'div', id,                  type.CLAZZ);
    var shElem    =  appendNewElement(nodeElem, 'div', id + '.shadow',      type.CLAZZ + '_shadow');
    var bgElem    =  appendNewElement(nodeElem, 'div', id + '.bg',          type.CLAZZ + '_bg');
    /*var nameElem=*/appendNewElement(nodeElem, 'div', id + '.name',        type.CLAZZ + '_name');
    
    // contents
    if (type.STATE) {
      appendNewElement(nodeElem, 'div', id + '.state_pbars',   'state_pbars');   // probability bars,... 
      appendNewElement(nodeElem, 'div', id + '.state_names',   'state_names');   // ...state names,...
      appendNewElement(nodeElem, 'div', id + '.state_pvals',   'state_pvals');   // ...probability values and...
    } else if (type.DISTRIBUTION) {
      appendNewElement(nodeElem, 'div', id + '.distribution',   'distribution'); // distribution for cont. nodes
    } else if (type.CONTENT) {
      appendNewElement(nodeElem, 'div', id + '.contents',       'contents');     // content area for cliques
    }

    // foregrounds
    var fgElem = appendNewElement(nodeElem, 'div', id + '.fg', type.CLAZZ + '_fg');

    // sensors
    if (type.STATE) {
      appendNewElement(nodeElem, 'div', id + '.state_sensors', 'state_sensors'); // ...sensors for discrete nodes      
    } else if (type.DISTRIBUTION) {
      appendNewElement(nodeElem, 'div', id + '.distribution_sensor', 'distribution_sensor'); // ...sensors for continuous nodes
    } else if (type.CONTENT) {
      // no sensors
    }
    
    // open popup
    var openElem;
    if (type.STATE) {
      openElem = appendNewElement(nodeElem, 'div', id + '.popup_open', 'popup_open');    
      openElem.innerHTML = '<a href="javascript:dummy();">[cpt]</a>';
    } else if (type.DISTRIBUTION) {
      openElem = appendNewElement(nodeElem, 'div', id + '.popup_open', 'popup_open');    
      openElem.innerHTML = '<a href="javascript:dummy();">[cgt]</a>';
    } else if (type.CONTENT && node.jpt) {
      openElem = appendNewElement(nodeElem, 'div', id + '.popup_open', 'popup_open');    
      openElem.innerHTML = '<a href="javascript:dummy();">[probs]</a>';
    } else if (type.CONTENT) {
      openElem = appendNewElement(nodeElem, 'div', id + '.popup_open', 'popup_open');    
      openElem.innerHTML = '[<a href="javascript:dummy();">lp</a> | '
                         +   '<a href="javascript:dummy();">bag</a>]';
    } else {
      openElem = null;
    }
    
    // set border images (foreground, background, shadow)
    setBorderImage(shElem, this.BASE_PATH + type.SHADOW, 
                   type.SPLIT.N,  type.SPLIT.E,  type.SPLIT.S,  type.SPLIT.W, 
                   type.BORDER.N, type.BORDER.E, type.BORDER.S, type.BORDER.W, 
                   type.STRETCH);
    setBorderImage(bgElem, this.BASE_PATH + type.BG, 
                   type.SPLIT.N,  type.SPLIT.E,  type.SPLIT.S,  type.SPLIT.W, 
                   type.BORDER.N, type.BORDER.E, type.BORDER.S, type.BORDER.W, 
                   type.STRETCH);
    setBorderImage(fgElem, this.BASE_PATH + type.FG, 
                   type.SPLIT.N,  type.SPLIT.E,  type.SPLIT.S,  type.SPLIT.W, 
                   type.BORDER.N, type.BORDER.E, type.BORDER.S, type.BORDER.W, 
                   type.STRETCH);

    // add dragging functionality 
    this.mkNodeSensitive(net, node, nodeElem, type);

    // make popup open "button" sensitive
    if (openElem) {
      var linkElem = openElem.getElementsByTagName('a')[0];
      this.mkPTOpenSensitive(net, node, linkElem, type.PT);
      var linkElem2 = openElem.getElementsByTagName('a')[1];
      if (linkElem2) {
        this.mkPTOpenSensitive(net, node, linkElem2, type.PTS);
      }
    }
  }

  // set position
  nodeElem.style.left = node.x;   // set x 
  nodeElem.style.top  = node.y;   // set y

  // set additional height needed for this node
  var h = (type.STATE 
           ? Math.max(1, type.HEIGHT_PER_ROW * Math.max(0, node.states.length - type.MIN_NUM_OF_ROWS))
           : (type.DISTRIBUTION 
              ? type.DISTRIBUTION.NODE_HEIGHT 
              : (type.CONTENT 
                 ? Math.max(1, type.HEIGHT_PER_ROW * Math.max(0, node.size() - type.MIN_NUM_OF_ROWS))
                 : 0)));
  var shElem = document.getElementById(id + '.shadow');
  var bgElem = document.getElementById(id + '.bg');
  var fgElem = document.getElementById(id + '.fg');
  shElem.style.height = h;
  bgElem.style.height = h;
  fgElem.style.height = h;

  // set height of the main element
  nodeElem.style.height = h + type.BORDER.N + type.BORDER.S;
  
  // set name
  var nameElem = document.getElementById(id + '.name');
  nameElem.innerHTML = node.name.toHTML();

  // display states
  if      (type.STATE)        this.displayStates(net, node, type.STATE);
  else if (type.DISTRIBUTION) this.displayDistribution(net, node, type.DISTRIBUTION);
  else if (type.CONTENT)      this.displayContent(net, node, type.CONTENT);
};

/**
 * Undisplay the given node.<br> 
 * Notice: links to/from that node are not undisplayed automatically!
 * @param net   bayesian network
 * @param node  one node in that network
 */
PN4Web.prototype.undisplayNode = function (net, node)
{
  removeElement(net.id + '.' + node.id);
};

/**
 * Make node sensitive for grab and drag
 * @param net       bayesian network
 * @param node      node to dragged
 * @param nodeElem  html node element
 * @param type      node type (e.g. this.NODE)
 */
PN4Web.prototype.mkNodeSensitive = function (net, node, nodeElem, type)
{
  var me = this;
  var grab = function(evnt) { 
    if (!evnt) evnt = window.event;
    var outer = document.getElementById(net.id + '.' + node.id);
    var inner = document.getElementById(net.id + '.' + node.id + '.bg');
    me.DRAG.drag    = function(e) { me.dragNode(outer, inner, net, node, type, e); };
    me.DRAG.release = function(e) { me.release(e); };
    me.grab(outer, inner, evnt);
  };
  addEventListener(nodeElem, 'mousedown', grab);
};

/**
 * Make openElem react on clicks by opening the 
 * corresponding CPT/JPT/CGT/LP-Potential popup.
 * @param net       bayesian network / junction tree
 * @param node      node/clique for which the CPT/JPT should be opened
 * @param openElem  make this html element sensitive
 * @param type      probability table type (e.g. this.CPT or this.JPT)
 */
PN4Web.prototype.mkPTOpenSensitive = function (net, node, openElem, type)
{
  var me = this;
  var open = function(evnt) { 
    if (!evnt) evnt = window.event;
    preventDefault(evnt);
    if (me.getPTPopup(net, node)) {
      me.undisplayPTPopup(net, node);
    } else {
      me.displayPTPopup(net, node, type);
    }
  };
  addEventListener(openElem, 'mousedown', open);
};
