/*
 * -------------------------------------------------------------------------
 *
 *  pn4webDisplayNet.js Copyright (C) 2011, 
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *   Author: Patrick Rammelt 
 *   E-Mail: patrick.rammelt@go4more.de
 *   Site:   http://www.patricks-seite.de
 * -------------------------------------------------------------------------
 *
 * Functionalities for displaying all types of networks (bayesian networks 
 * and junction trees). A frame around the network area could be created
 * separately (see "pn4webDisplayFame.js").
 */


// DISPLAY NETWORK ==================================================

/**
 * Display bayesian network (all nodes and links)
 * @param net   bayesian network / junction tree
 * @param type  e.g. this.NET
 */
PN4Web.prototype.displayNet = function (net, type)
{
  // store network
  this.nets[net.id] = net;

  // cannot display anything if there is no such element
  if (!document.getElementById(net.id)) return;

  // display graphs 
  if (net.rate != undefined) {
    this.posteriors[net.id] = new TimeData();
    this.updateGraphs(net);
  }

  // display nodes
  for (var nId in net.nodes) { 
    var node = net.nodes[nId];
    if (node.isContinuous) {                   // continuous node
      this.displayNode(net, node, type.CGNODE); 
    } else if (net.succ && net.succ[nId]) {    // node has successor -> its a predecessor node
      this.displayNode(net, node, type.PRE_NODE); 
    } else {                                   // discrete node
      this.displayNode(net, node, type.NODE); 
    }
  }

  // display directed links
  for (var paId in net.children) {
    var parent = net.nodes[paId];
    for (var chId in net.children[paId]) {
      var child = net.nodes[chId];
      if (net.succ && net.succ[paId]) { // parent has successor -> dynamic link
        this.displayLink(net, parent, child, type.NODE.DYNLINK, type.NODE.LINKDATA);
      } else {
        this.displayLink(net, parent, child, type.NODE.LINK, type.NODE.LINKDATA);
      }
    }
  }

  // display undirected links
  for (var n1Id in net.siblings) {
    var node1 = net.nodes[n1Id];
    for (var n2Id in net.siblings[n1Id]) {
      var node2 = net.nodes[n2Id];
      if (n1Id < n2Id) {
        this.displayLink(net, node1, node2, type.NODE.RELATION, type.NODE.LINKDATA);
      }
    }
  }

  // fix network display area size
  var netElem = document.getElementById(net.id);
  var bgMElem = document.getElementById(net.id + '.area');
  if (bgMElem) {
    netElem.style.width  = getWidth(bgMElem); 
    netElem.style.height = getHeight(bgMElem);
  }

  // display popups
  for (var nId in net.nodes) {
    var node = net.nodes[nId];
    var nodeType = (node.states ? type.NODE : type.CGNODE);
    var nodePopupElem = document.getElementById(net.id + '.' + nId + '.pt');
    if (nodePopupElem) {
      this.displayPTPopup(net, node, nodeType.PT);
    }
    for (var chId in net.children[nId]) {
      var link = net.children[nId][chId];
      if (link.id != undefined) {
        var linkPopupElem = document.getElementById(net.id + '.' + link.id + '.pt');
        if (linkPopupElem) {
          this.displayPTPopup(net, link, this.NODE.LINKDATA.PT);	  
        }
      }
    }
  }
};

/**
 * Undisplay bayesian network (all nodes and links)
 * @param net  bayesian network / junction tree
 */
PN4Web.prototype.undisplayNet = function (net)
{
  // check
  var netElem = document.getElementById(net.id);
  if (!netElem) return;

  // remove directed links
  for (var paId in net.children) {
    for (var chId in net.children[paId]) {
      this.undisplayLink(net, net.getNode(paId), net.getNode(chId));
    }
  }

  // remove undirected links
  for (var n1Id in net.siblings) {
    for (var n2Id in net.siblings[n1Id]) {
      if (n1Id > n2Id) { 
        this.undisplayLink(net, net.getNode(n1Id), net.getNode(n2Id));
      }
    }
  }

  // remove node
  for (var nId in net.nodes) { 
    this.undisplayNode(net, net.getNode(nId)); 
  }

  // forget network
  delete (this.nets[net.id]);
};

/**
 * Clear bayesian network: removes all html-elements
 * from the div-element used to display the given network.
 * @param net  bayesian network, junction-tree, ...
 */
PN4Web.prototype.clearNet = function (net)
{
  // check
  var netElem = document.getElementById(net.id);
  if (!netElem) return;

  // clear
  netElem.innerHTML = "";

  // forget network
  delete (this.nets[net.id]);
};
