/*
 * -------------------------------------------------------------------------
 *
 *  pn4webDisplayLink.js Copyright (C) 2011, 
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *   Author: Patrick Rammelt 
 *   E-Mail: patrick.rammelt@go4more.de
 *   Site:   http://www.patricks-seite.de
 * -------------------------------------------------------------------------
 *
 * Display all kinds of links (directed and undirected). CSS3 attribute 
 * "transform" ("-moz-transform", "-webkit-transform", "-o-transform") 
 * is used to rotate links such that they point into the right direction. 
 * This is not yet supported by all browsers.
 */


// DISPLAY LINKS ====================================================

/**
 * Display (or redisplay) a link from the first given node to
 * the second given node, 
 * @param net   bayesian network / junction tree
 * @param from  first node/clique
 * @param to    second node/clique
 * @param type  link type (e.g. this.LINK or this.RELATION)
 * @param data  data type (e.g. this.SEPARATOR) [optional]
 * @param anim  animation [optional]
 */
PN4Web.prototype.displayLink = function (net, from, to, type, data, anim)
{
  // id of html link element 
  var id = net.id + '.' + from.id + '__' + net.id + '.' + to.id; 

  var netElem  = document.getElementById(net.id);
  var linkElem = document.getElementById(id);
  if (!linkElem) {

    // create link element
    linkElem   = appendNewElement(netElem,  'div', id,               type.CLAZZ,            true); 
    var shElem = appendNewElement(linkElem, 'div', id + '.shadow',   type.CLAZZ + '_shadow');
    var bgElem = appendNewElement(linkElem, 'div', id + '.bg',       type.CLAZZ + '_bg');
    
    // set border images (background image)
    setBorderImage(shElem, this.BASE_PATH + type.SHADOW, 
                   type.SPLIT.N,  type.SPLIT.E,  type.SPLIT.S,  type.SPLIT.W, 
                   type.BORDER.N, type.BORDER.E, type.BORDER.S, type.BORDER.W, 
                   type.STRETCH);
    setBorderImage(bgElem, this.BASE_PATH + type.BG, 
                   type.SPLIT.N,  type.SPLIT.E,  type.SPLIT.S,  type.SPLIT.W, 
                   type.BORDER.N, type.BORDER.E, type.BORDER.S, type.BORDER.W, 
                   type.STRETCH);
         
  }

  // set position and size
  if (anim) {
    var me      = this;
    var wrapper = function (s) { me.layoutLink(net, from, to, type, data, s); };
    anim.start(wrapper);
  } else {
    this.layoutLink(net, from, to, type, data, 1.0);
  }
};

/**
 * Set the position of a link from the first given node to
 * the second given node. 
 * @param net   bayesian network / junction tree
 * @param from  first node/clique
 * @param to    second node/clique
 * @param type  link type (e.g. this.LINK or this.RELATION)
 * @param data  data type (e.g. this.SEPARATOR) [optional]
 * @param size  relative size (used for animation)
 */
PN4Web.prototype.layoutLink = function (net, from, to, type, data, size)
{
  // id of html link element and of relevant elements in connected nodes
  var id = net.id + '.' + from.id + '__' + net.id + '.' + to.id; 
  var fromId   = net.id + '.' + from.id;
  var fromBgId = net.id + '.' + from.id + '.bg';
  var toId     = net.id + '.' + to.id;
  var toBgId   = net.id + '.' + to.id   + '.bg';

  // get elements
  var linkElem   = document.getElementById(id);
  var fromElem   = document.getElementById(fromId);
  var fromBgElem = document.getElementById(fromBgId);
  var toElem     = document.getElementById(toId);
  var toBgElem   = document.getElementById(toBgId);

  // center points
  var x0 = fromElem.offsetLeft + (fromBgElem.offsetWidth  / 2);
  var y0 = fromElem.offsetTop  + (fromBgElem.offsetHeight / 2);
  var x1 = toElem.offsetLeft   + (toBgElem.offsetWidth    / 2);
  var y1 = toElem.offsetTop    + (toBgElem.offsetHeight   / 2);

  // angle
  var a = Math.atan2(y1-y0, x1-x0);

  // points to connect
  var cos = Math.cos(a);
  var sin = Math.sin(a);
  var sqrtCos = Math.sqrt(Math.abs(cos)) * (cos < 0 ? -1 : 1);
  var sqrtSin = Math.sqrt(Math.abs(sin)) * (sin < 0 ? -1 : 1);
  x0 += (fromBgElem.offsetWidth  / 2) * sqrtCos;
  y0 += (fromBgElem.offsetHeight / 2) * sqrtSin;
  x1 -= (toBgElem.offsetWidth    / 2) * sqrtCos; // (x1,y1) might node be reached...
  y1 -= (toBgElem.offsetHeight   / 2) * sqrtSin; // ...due to minimum link length

  // link length (width)
  var dx = (x1-x0);
  var dy = (y1-y0);
  var wa = parseInt(Math.max(type.MIN_LENGTH, Math.sqrt(dx*dx + dy*dy) * size)); // overall
  var wm = wa - type.MIN_LENGTH;                                                 // mid-segment

  // shadow offsets
  var ox = this.SHADOW.XOFFSET * cos + this.SHADOW.YOFFSET * sin;
  var oy = this.SHADOW.XOFFSET * cos - this.SHADOW.YOFFSET * sin;

  // display separator (for junction trees)
  var sep = net.children[from.id][to.id];
  if (data && sep && sep.jpt) {
    x1 = x0 + Math.cos(a) * (wm + type.MIN_LENGTH/2);  // real link end point 
    y1 = y0 + Math.sin(a) * (wm + type.MIN_LENGTH/2);
    this.displaySeparator(net, from, to, (x0+x1)/2, (y0+y1)/2, data);
  }

  // (don't) set position (webkit bug 60689: 
  // left/top isn't updated after rotation has been used)
  var left = toInt(x0 + (sin * (type.THICKNESS / 2 + Math.max(0,  ox)))); 
  var top  = toInt(y0 - (cos * (type.THICKNESS / 2 + Math.max(0, -oy))));
  linkElem.style.left = 0; // left; // set x 
  linkElem.style.top  = 0; // top;  // set y

  // set orientation (and position, see above)
  var transform = 'translate(' + left + 'px, ' + top + 'px) rotate(' + a + 'rad)';
  setTransform(linkElem, transform);

  // set shadow offset
  var shElem = document.getElementById(id + '.shadow');
  var bgElem = document.getElementById(id + '.bg');
  shElem.style.left = Math.max(0,  ox);
  shElem.style.top  = Math.max(0,  oy);
  bgElem.style.left = Math.max(0, -ox);
  bgElem.style.top  = Math.max(0, -oy);

  // set length
  shElem.style.width = wm;
  bgElem.style.width = wm;
  
  // make sure this element is visible...
  // (used multiple times on links that are animated)
  linkElem.style.visibility = 'visible';
};

/**
 * Undisplay the given link.
 * @param net   bayesian network / junction tree
 * @param from  first node/clique
 * @param to    second node/clique
 */
PN4Web.prototype.undisplayLink = function (net, from, to)
{
  removeElement(net.id + '.' + from.id + '__' + net.id + '.' + to.id); 
  this.undisplaySeparator(net, from, to);
};


// DISPLAY SEPARATOR ================================================

/**
 * Display separator
 * @param net   junction tree
 * @param from  source clique
 * @param to    target clique
 * @param x     center x-coordinate
 * @param y     center y-coordinate
 * @param type  link-data type (e.g. this.SEPARATOR)
 */
PN4Web.prototype.displaySeparator = function (net, from, to, x, y, type)
{
  var id = net.id + '.' + from.id + '__' + to.id; 

  var netElem = document.getElementById(net.id);
  var sepElem = document.getElementById(id);
  if (!sepElem) {

    // create link element and sub-elements
    sepElem    = appendNewElement(netElem, 'div', id,               type.CLAZZ); 
    var shElem = appendNewElement(sepElem, 'div', id + '.shadow',   type.CLAZZ + '_shadow');
    var bgElem = appendNewElement(sepElem, 'div', id + '.bg',       type.CLAZZ + '_bg');
    var opElem = appendNewElement(sepElem, 'div', id + '.sep_open', 'sep_open');
    
    // set border images (background image)
    setBorderImage(shElem, this.BASE_PATH + type.SHADOW, 
                   type.SPLIT.N,  type.SPLIT.E,  type.SPLIT.S,  type.SPLIT.W, 
                   type.BORDER.N, type.BORDER.E, type.BORDER.S, type.BORDER.W, 
                   type.STRETCH);
    setBorderImage(bgElem, this.BASE_PATH + type.BG, 
                   type.SPLIT.N,  type.SPLIT.E,  type.SPLIT.S,  type.SPLIT.W, 
                   type.BORDER.N, type.BORDER.E, type.BORDER.S, type.BORDER.W, 
                   type.STRETCH);
    
    // set link to open the separator popup
    var lnElem = appendNewElement(opElem, 'a');
    lnElem.href = 'javascript:dummy();';
    lnElem.innerHTML = '[p]';

    this.mkSepOpenSensitive(net, from, to, lnElem, type.PT);
  }

  // set position 
  sepElem.style.left = parseInt(x - (type.WIDTH  / 2));  // set x 
  sepElem.style.top  = parseInt(y - (type.HEIGHT / 2));  // set y
};

/**
 * Make openElem react on clicks by opening the JPT popup
 * @param net       junction tree
 * @param from      source clique of the separator link 
 * @param to        target clique of the separator link
 * @param openElem  make this html element sensitive
 * @param type      probability table type (e.g. this.JPT)
 */
PN4Web.prototype.mkSepOpenSensitive = function (net, from, to, openElem, type)
{
  var me = this;
  var sep = (net.children[from.id][to.id] 
	     ? net.children[from.id][to.id] 
	     : net.siblings[from.id][to.id]);
  var open = function(evnt) {
    if (!evnt) evnt = window.event;
    preventDefault(evnt);
    if (me.getPTPopup(net, sep)) {
      me.undisplayPTPopup(net, sep);
    } else {
      me.displayPTPopup(net, sep, type);
    }
  };
  addEventListener(openElem, 'mousedown', open);
};

/**
 * Undisplay the given separator
 * @param net   junction tree
 * @param from  first clique
 * @param to    second clique
 */
PN4Web.prototype.undisplaySeparator = function (net, from, to)
{
  removeElement(net.id + '.' + from.id + '__' + net.id + '.' + to.id + '.sep'); 
};

