/*
 * -------------------------------------------------------------------------
 *
 *  pn4webDisplayGraph.js Copyright (C) 2011, 
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *   Author: Patrick Rammelt 
 *   E-Mail: patrick.rammelt@go4more.de
 *   Site:   http://www.patricks-seite.de
 * -------------------------------------------------------------------------
 *
 * Displaying graphs for dynamic bayesian networks (posteriors over time)
 */


// DISPLAY GRAPHS FOR DYNAMIC BAYESIAN NETWORKS =====================

/**
 * Display graphs.
 * @param net         bayesian network
 * @param posteriors  recorded posterior data (TimeData)
 * @param t           current time
 * @param type        e.g. this.GRAPH
 */
PN4Web.prototype.displayGraphs = function (net, posteriors, t, type)
{
  for (var nId in net.nodes) {
    var node = net.nodes[nId];

    // get canvas element and 2d-context
    var graphElem = document.getElementById(net.id + '.' + nId + '.graph');
    if (!graphElem || !graphElem.getContext) continue;
    var context = graphElem.getContext('2d');
    if (!context) continue;

    // widths and heights
    var w = getWidth(graphElem);
    var h = getHeight(graphElem);
    var v = type.HEADER_HEIGHT + type.HEADER_YOFFSET;
    var u = h-v;

    // define shadows
    context.shadowOffsetX = type.SHADOW_XOFFSET;
    context.shadowOffsetY = type.SHADOW_YOFFSET;
    context.shadowBlur    = type.SHADOW_BLUR;
    context.shadowColor   = type.SHADOW_COLOR;
    
    // clear and set background
    context.fillStyle = type.BG_COLOR;
    context.clearRect(0, 0, w, h);
    context.fillRect(0, 0, w, h);

    // display node name
    context.textBaseline = 'bottom';
    context.fillStyle    = type.NODE_COLOR;
    context.font         = type.NODE_FONT;
    context.fillText(node.name, 0, type.HEADER_HEIGHT);

    // display state names (in respective colors)
    context.font         = type.STATE_FONT;
    for (var s = 0; s < node.states.length; s++) {
      context.fillStyle = type.LINE_COLORS[s % type.LINE_COLORS.length];
      context.fillText(node.states[s].name, s*type.STATE_XOFFSET+type.NODE_XOFFSET, type.HEADER_HEIGHT);
    }

    // display graph lines
    context.lineWidth = type.LINE_WIDTH;
    var x, y;
    if (posteriors.tmin < posteriors.tmax) {

      // over all states
      for (var s = 0; s < node.states.length; s++) {
        context.strokeStyle = type.LINE_COLORS[s % type.LINE_COLORS.length];
        var ti = posteriors.tmin;
        var p = posteriors.stored(ti, nId); // first time-slice
        x = 0;
        y = toInt(u - (p[s] * u) + v);
        context.beginPath();
        context.moveTo(x, y);
        for (ti++; ti <= posteriors.tmax; ti++) {
          p = posteriors.stored(ti, nId);
          x = toInt((ti - posteriors.tmin) / (posteriors.tmax - posteriors.tmin) * w);
          y = toInt(u - (p[s] * u) + v);
          context.lineTo(x, y);
        }
        context.stroke();
      }

      // display marker for current time
      context.lineWidth = type.TIMEMARKER_WIDTH;
      context.strokeStyle = type.TIMEMARKER_COLOR;
      context.beginPath();
      x = toInt((t - posteriors.tmin) / (posteriors.tmax - posteriors.tmin) * w); 
      context.moveTo(x, v);
      context.lineTo(x, h);
      context.stroke();

    }

  }
};

