/*
 * -------------------------------------------------------------------------
 *
 *  pn4webDisplayContent.js Copyright (C) 2011, 
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *   Author: Patrick Rammelt 
 *   E-Mail: patrick.rammelt@go4more.de
 *   Site:   http://www.patricks-seite.de
 * -------------------------------------------------------------------------
 *
 * Displays a distribution of a continuous node (like pn4webDisplayState.js 
 * splays states of discrete nodes of a bayesian network).
 */


// DISPLAY A DISTRIBUTION OF A CONTINUOUS NODE (GRAPH) ==============

/**
 * Display or redisplay the content of the given node
 * @param net   bayesian network
 * @param node  node in that network
 * @param type  e.g. this.DISTRIBUTION
 */
PN4Web.prototype.displayDistribution = function (net, node, type)
{
  // id for the node html element
  var id = net.id + '.' + node.id;

  // container elements
  var distrElem  = document.getElementById(id + '.distribution');
  var sensorElem = document.getElementById(id + '.distribution_sensor');
  
  var w = type.WIDTH; 
  var h = type.HEIGHT; 
  var graphElem = distrElem.childNodes[0];
  if (!graphElem) {
    distrElem.innerHTML 
      = '<canvas id="' + id + '.graph" class="' + type.CLAZZ + '" '
      +         'width="' + w + '" height="' + h + '">'
      +   '<div style="width:' + w + 'px; height:' + h + 'px; background-color:black;">'
      +     'Sorry, graphs<br>are not supported<br>by your browser'
      +   '</div>'
      + '</canvas>';
    graphElem = distrElem.childNodes[0];
    sensorElem.style.width  = w;
    sensorElem.style.height = h;
    this.mkDistributionSensitive(net, node, sensorElem);
  }

  // get context
  if (!graphElem.getContext) return;
  var context = graphElem.getContext('2d');
  if (!context) return;

  // clear graph
  context.clearRect(0, 0, w, h);

  if (node.v != null) {

    var r = (node.v - node.min) / (node.max - node.min);
    var y = 5.5 + ((h - 10) * r);

    // display "graph" (just one line)
    context.lineWidth    = type.AXIS_WIDTH;
    context.strokeStyle  = type.FIXED_COLOR;
    context.beginPath();
    context.moveTo(2.5, y);
    context.lineTo(w,   y);
    context.stroke();

    // display marker and position
    context.textBaseline = 'middle';
    context.textAlign    = 'right';
    context.strokeStyle  = type.AXIS_COLOR;
    context.fillStyle    = type.TEXT_COLOR;
    context.font         = type.TEXT_FONT;
    context.beginPath();
    context.moveTo(0, y);
    context.lineTo(5, y);
    context.stroke();
    context.fillText(node.v.toFixed(4), w-2.5, y);

  } else {

    // get values
    var max  = 0;
    var vals = new Array(type.RESOLUTION);
    for (var s = 0; s < type.RESOLUTION; s++) {
      var r = s / (type.RESOLUTION - 1); 
      var v = node.min + ((node.max - node.min) * r);
      vals[s] = 0;
      for (var i in node.p) {
        vals[s] += node.p[i].value(v); // get weighted value
      }
      max = Math.max(max, vals[s]);
    }
    
    // display graph
    if (max > 0) {
      context.fillStyle = type.GRAPH_COLOR;
      context.beginPath();
      context.moveTo(2.5, 5.5);
      for (var s = 0; s < type.RESOLUTION; s++) {
        var r = s / (type.RESOLUTION - 1); 
        var y = 5.5 + ((h - 10) * r);
        var x = 2.5 + ((vals[s] / max) * (w - 2.5));
        context.lineTo(x, y);
      }
      context.lineTo(2.5, h-4.5);
      context.fill();
    }
  }

  // coordinate system
  context.textBaseline = 'middle';
  context.textAlign    = 'left';
  context.fillStyle    = type.TEXT_COLOR;
  context.font         = type.TEXT_FONT;
  context.lineWidth    = type.AXIS_WIDTH;
  context.strokeStyle  = type.AXIS_COLOR;

  // y-axis
  context.beginPath();
  context.moveTo(2.5, 0);
  context.lineTo(2.5, h);
  context.stroke();
  
  // markers
  for (var i = 0; i <= 5; i++) {
    var r = i/5;
    var y = 5.5 + ((h - 10) * r);
    var v = (r * (node.max-node.min) + node.min);
    context.beginPath();
    context.moveTo(0, y);
    context.lineTo(5, y);
    context.stroke();
    context.fillText(v.toFixed(0), 6, y);
  }
};


/**
 * Make graph sensitive on clicks
 * @param net        bayesian network
 * @param node       node
 * @param sensorElem sensor element
 */
PN4Web.prototype.mkDistributionSensitive = function (net, node, sensorElem)
{
  if (!(net.succ && net.succ[node.id])) { // not for predecessor nodes!
    var me = this;
    var set = function(evnt) {
      if (!evnt) evnt = window.event;
      preventDefault(evnt);
      
      // set/unset evidence
      var y = (pageY(evnt) - getTop(sensorElem));
      var r = (y - 5.5) / (sensorElem.offsetHeight - 10);
      var v = r * (node.max - node.min) + node.min;
      node.v = (v < node.min || v > node.max ? null : v);
      
      // propagate and update
      me.propagate(net);  
    };
    addEventListener(sensorElem, 'click',    set);
  }
};
