/*
 * -------------------------------------------------------------------------
 *
 *  pn4webDisplayLink.js Copyright (C) 2011, 
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *   Author: Patrick Rammelt 
 *   E-Mail: patrick.rammelt@go4more.de
 *   Site:   http://www.patricks-seite.de
 * -------------------------------------------------------------------------
 *
 * Animations. An animation is an object which has methods: 'start(fnc)' 
 * and 'stop()', where 'fnc' is a (wrapper) function which takes one
 * parameter (the animated value). Furthermore each animation takes care
 * that is stoppes whenever another animation on the same object 
 * (identified by some kind of Id) is started.
 */


// ANIMATIONS =======================================================

/* 
 * Holds all currently running animations 
 * ('animations[id]' = animation).
 */
var animations = {};

// LINEAR ANIMATION -------------------------------------------------

/*
 * Constructor for a linear animation
 * id:    id; this animation is stopped if there is
 *        another animation started with the same id
 * steps: number of steps - 1
 * delay: delay between two steps
 */
function LinearAnimation (id, steps, delay)
{
  if (animations[id]) animations[id].stop();
  animations[id] = this;

  this.id    = id;
  this.steps = steps;
  this.delay = delay;
  this.i     = 0;
}

/* Inheritance */
// LinearAnimation.prototype          = 
// LinearAnimation.prototype.parent   = 
LinearAnimation.prototype.constructor = LinearAnimation;

/*
 * Start this animation
 * fnc: function to be called with one parameter
 *      which is a value in [0..1]
 */
LinearAnimation.prototype.start = function (fnc) 
{
  this.i   = 0;
  var me   = this;
  var anim = function () {
    if (me.i <= me.steps) {
      var v = me.i/me.steps; 
      fnc(v);
      me.i++;
      setTimeout(anim, me.delay);
    } 
  };
  anim();
};

/*
 * Stop this animation
 */
LinearAnimation.prototype.stop = function ()
{
  this.i = this.steps + 1; 
};

// SINUS-ANIMATION --------------------------------------------------

/*
 * Constructor for a sinus animation
 * id:    id; this animation is stopped if there is
 *        another animation started with the same id
 * steps: number of steps - 1
 * delay: delay between two steps
 */
function SinusAnimation (id, steps, delay)
{
  if (animations[id]) animations[id].stop();
  animations[id] = this;

  this.id    = id;
  this.steps = steps;
  this.delay = delay;
  this.i     = 0;
}

/*
 * Start this animation
 * fnc: function to be called with one parameter
 *      which is a value in [0..1]
 */
SinusAnimation.prototype.start = function (fnc) 
{
  this.i   = 0;
  var me   = this;
  var anim = function () {
    if (me.i <= me.steps) {
      var v = 1.0 - ((Math.cos(me.i/me.steps * Math.PI) + 1.0) / 2.0);
      fnc(v);
      me.i++;
      setTimeout(anim, me.delay);
    } 
  };
  anim();
};

/*
 * Stop this animation
 */
SinusAnimation.prototype.stop = function ()
{
  this.i = this.steps + 1; 
};
