/*
 * -------------------------------------------------------------------------
 *
 *  pn4webDisplay.js Copyright (C) 2011, 
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *   Author: Patrick Rammelt 
 *   E-Mail: patrick.rammelt@go4more.de
 *   Site:   http://www.patricks-seite.de
 * -------------------------------------------------------------------------
 *
 * General display functionalities for bayesian networks. 
 * All initializations are done here (definining images, sizes, etc.)
 * Also grab, drag and drop functionalities for nodes and popup windows
 * (e.g. to show probability tables) are implemented here.
 */

/* 
 * Display class (container for all global 
 * constants and parameters and display methods) 
 */
var PN4W = {};


// INITIALIZATION ===================================================

/**
 * Initialization
 * <dl><dt>Notice:</dt><dd>
 * Since we are using border-images the callback-parameter should 
 * be used to provide a function that actually displays one or more 
 * networks. Otherwise sometimes the nodes won't have their correct
 * size when this is needed to display links.</dd></dl>
 * @param subLevels       number of sub-directories to the root directory
 * @param maxPopupWidth   max width for popup windows (middle area) 
 * @param maxPopupHeight  max height for popup windows (middle area)
 * @param shadowXOffset   x-offset for shadows (needed because links rotate)
 * @param shadowYOffset   y-offset for shadows (needed because links rotate)
 * @param callback        function to be called after everything has been 
 *                        initialized [optional] (although recommendet)
 */
function initPN4Web (subLevel, maxPopupWidth, maxPopupHeight,
                     shadowXOffset, shadowYOffset, callback)
{
  PN4W = new PN4Web(subLevel, maxPopupWidth, maxPopupHeight, 
                    shadowXOffset, shadowYOffset, callback);
}


// CONSTRUCTOR ======================================================

/**
 * Constructs the display object (its ment to be a singleton - i.e. 
 * call/instantiate this only once!)
 * <dl><dt>Notice:</dt><dd>
 * Since we are using border-images the callback-parameter should 
 * be used to provide a function that actually displays one or more 
 * networks. Otherwise sometimes the nodes won't have their correct
 * size when this is needed to display links.</dd></dl>
 * @param subLevels       number of sub-directories to the root directory
 * @param maxPopupWidth   max width for popup windows (middle area) 
 * @param maxPopupHeight  max height for popup windows (middle area)
 * @param shadowXOffset   x-offset for shadows (needed because links rotate)
 * @param shadowYOffset   y-offset for shadows (needed because links rotate)
 * @param callback        function to be called after everything has been 
 *                        initialized [optional] (although recommendet)
 */
function PN4Web (subLevel, maxPopupWidth, maxPopupHeight, 
                 shadowXOffset, shadowYOffset, callback)
{
  var me = this;

  // networks
  this.nets       = {}; // map: id to net
  this.posteriors = {}; // map: id to posterior data for each dynamic network
  this.running    = {}; // map: id to flag indicating running dynamic networks

  // attributes needed for node dragging
  this.DRAG = {};
  this.DRAG.drag    = dummyEventHandler; // drag function
  this.DRAG.release = dummyEventHandler; // release function
  this.DRAG.x0      = 0;                 // start x-coordinate of dragging
  this.DRAG.y0      = 0;                 // start y-coordinate of dragging
  this.DRAG.xOffset = -1;                // offset grab position to left border
  this.DRAG.yOffset = -1;                // offset grab position to top border

  // get base path
  this.BASE_PATH = "";
  for (var i = 0; i < subLevel; i++) this.BASE_PATH += "../";

  // callback counting loaded images
  var miss = 1;                          // counter for missing images (+1)
  var unloaded   = function (img) { miss++; };
  var loaded     = function (img) { if (--miss <= 0 && callback) callback(); };
  
  // controls for dynamic bayesian networks
  this.CTRL = {};
  this.CTRL.CLAZZ     = 'ctrl';
  this.CTRL.PLAY      = "js/pn4web/play.png";
  this.CTRL.PLAYBACK  = "js/pn4web/playback.png";
  this.CTRL.NEXT      = "js/pn4web/step.png";
  this.CTRL.PREV      = "js/pn4web/stepback.png";
  this.CTRL.PAUSE     = "js/pn4web/pause.png";
  this.CTRL.STOP      = "js/pn4web/stop.png";
  this.CTRL.FASTER    = "js/pn4web/faster.png";
  this.CTRL.SLOWER    = "js/pn4web/slower.png";

  // graphs (for dynamic bayesian networks)
  this.GRAPH                  = {};
  this.GRAPH.CLAZZ            = 'graph';
  this.GRAPH.BG_COLOR         = 'white';
  this.GRAPH.NODE_FONT        = 'bold 10px sans-serif';
  this.GRAPH.NODE_COLOR       = 'black';
  this.GRAPH.STATE_FONT       = 'bold 10px sans-serif';
  this.GRAPH.NODE_XOFFSET     = 100;  // x-dist. of first state name to node name
  this.GRAPH.STATE_XOFFSET    = 50;   // x-dist. between state names
  this.GRAPH.HEADER_HEIGHT    = 10;
  this.GRAPH.HEADER_YOFFSET   = 5;    // distance between header and graph
  this.GRAPH.TIMEMARKER_COLOR = '#777';
  this.GRAPH.TIMEMARKER_WIDTH = 3;
  this.GRAPH.LINE_COLORS      = new Array('red', 'green', 'blue', 'magenta', 'cyan', 'yellow');
  this.GRAPH.LINE_WIDTH       = 1;
  this.GRAPH.SHADOW_XOFFSET   = 5; 
  this.GRAPH.SHADOW_YOFFSET   = 5;
  this.GRAPH.SHADOW_BLUR      = 4;
  this.GRAPH.SHADOW_COLOR     = 'rgba(0,0,0,0.5)';

  // popup
  this.POPUP            = {};
  this.POPUP.CLAZZ      = 'popup';
  this.POPUP.MAX_WIDTH  = maxPopupWidth;
  this.POPUP.MAX_HEIGHT = maxPopupHeight;
  this.POPUP.MIN_WIDTH  = 100; // only used for empty popups (multiple contents)
  this.POPUP.MIN_HEIGHT = 10;  // only used for empty popups (multiple contents)
  this.POPUP.STRETCH    = true;    // stretch shadow and background images?
  this.POPUP.BORDER     = { E:23, W:23, N:23, S:23 };   
  this.POPUP.SPLIT      = { E:92, W:92, N:92, S:92 };   
  this.POPUP.BG         = loadImg(this.BASE_PATH, "js/pn4web/popup.png",        unloaded, loaded);
  this.POPUP.SHADOW     = loadImg(this.BASE_PATH, "js/pn4web/popup_shadow.png", unloaded, loaded);

  // normal probability bars (for nodes)
  this.PBAR         = {};
  this.PBAR.CLAZZ   = 'pbar';
  this.PBAR.STRETCH = true;    // stretch shadow and background images?
  this.PBAR.BORDER  = { E:4,  W:4,  N:0, S:0 }; 
  this.PBAR.SPLIT   = { E:16, W:16, N:0, S:0 };  
  this.PBAR.BG      = loadImg(this.BASE_PATH, "js/pn4web/pbar.png", unloaded, loaded);

  // active probability bars (for nodes)
  this.PBAR_ACTIVE         = {};
  this.PBAR_ACTIVE.CLAZZ   = 'pbar';
  this.PBAR_ACTIVE.STRETCH = true;    // stretch shadow and background images?
  this.PBAR_ACTIVE.BORDER  = { E:4,  W:4,  N:0, S:0 };
  this.PBAR_ACTIVE.SPLIT   = { E:16, W:16, N:0, S:0 };
  this.PBAR_ACTIVE.BG      = loadImg(this.BASE_PATH, "js/pn4web/pbar_active.png", unloaded, loaded); 

  // inactive probability bars (for nodes)
  this.PBAR_INACTIVE         = {};
  this.PBAR_INACTIVE.CLAZZ   = 'pbar';
  this.PBAR_INACTIVE.STRETCH = true;    // stretch shadow and background images?
  this.PBAR_INACTIVE.BORDER  = { E:4,  W:4,  N:0, S:0 };
  this.PBAR_INACTIVE.SPLIT   = { E:16, W:16, N:0, S:0 };
  this.PBAR_INACTIVE.BG      = loadImg(this.BASE_PATH, "js/pn4web/pbar_inactive.png", unloaded, loaded);

  // normal probability bars (for pre-nodes in dynamic networks)
  this.PRE_PBAR         = {};
  this.PRE_PBAR.CLAZZ   = 'pbar';
  this.PRE_PBAR.STRETCH = true;    // stretch shadow and background images?
  this.PRE_PBAR.BORDER  = { E:4,  W:4,  N:0, S:0 };
  this.PRE_PBAR.SPLIT   = { E:16, W:16, N:0, S:0 };
  this.PRE_PBAR.BG      = loadImg(this.BASE_PATH, "js/pn4web/pre_pbar.png", unloaded, loaded);

  // active probability bars (for pre-nodes in dynamic networks)
  this.PRE_PBAR_ACTIVE         = {};
  this.PRE_PBAR_ACTIVE.CLAZZ   = 'pbar';
  this.PRE_PBAR_ACTIVE.STRETCH = true;    // stretch shadow and background images?
  this.PRE_PBAR_ACTIVE.BORDER  = { E:4,  W:4,  N:0, S:0 };
  this.PRE_PBAR_ACTIVE.SPLIT   = { E:16, W:16, N:0, S:0 };
  this.PRE_PBAR_ACTIVE.BG      = loadImg(this.BASE_PATH, "js/pn4web/pre_pbar_active.png", unloaded, loaded);

  // inactive probability bars (for pre-nodes in dynamic networks)
  this.PRE_PBAR_INACTIVE         = {};
  this.PRE_PBAR_INACTIVE.CLAZZ   = 'pbar';
  this.PRE_PBAR_INACTIVE.STRETCH = true;    // stretch shadow and background images?
  this.PRE_PBAR_INACTIVE.BORDER  = { E:4,  W:4,  N:0, S:0 };   // CSS3
  this.PRE_PBAR_INACTIVE.SPLIT   = { E:16, W:16, N:0, S:0 };   // CSS3
  this.PRE_PBAR_INACTIVE.BG      = loadImg(this.BASE_PATH, "js/pn4web/pre_pbar_inactive.png", unloaded, loaded);

  // probability bars (for probability tables, such as cpts)
  this.PTPBAR         = {};
  this.PTPBAR.CLAZZ   = 'pt_pbar';
  this.PTPBAR.STRETCH = true;    // stretch shadow and background images?
  this.PTPBAR.BORDER  = { E:4,  W:4,  N:0, S:0 };
  this.PTPBAR.SPLIT   = { E:16, W:16, N:0, S:0 };
  this.PTPBAR.BG      = loadImg(this.BASE_PATH, "js/pn4web/pbar.png", unloaded, loaded);

  // probability bars (for probability tables, such as cpts)
  this.FJPTPBAR         = {};
  this.FJPTPBAR.CLAZZ   = 'fjpt_pbar';
  this.FJPTPBAR.STRETCH = true;    // stretch shadow and background images?
  this.FJPTPBAR.BORDER  = { E:4,  W:4,  N:0, S:0 };
  this.FJPTPBAR.SPLIT   = { E:16, W:16, N:0, S:0 };
  this.FJPTPBAR.BG      = loadImg(this.BASE_PATH, "js/pn4web/pbar.png", unloaded, loaded);

  // probability bars (for P(e|M))
  this.PEPBAR         = {};
  this.PEPBAR.CLAZZ   = 'pe_pbar';
  this.PEPBAR.STRETCH = true;    // stretch shadow and background images?
  this.PEPBAR.BORDER  = { E:4,  W:4,  N:0, S:0 };
  this.PEPBAR.SPLIT   = { E:16, W:16, N:0, S:0 };
  this.PEPBAR.BG      = loadImg(this.BASE_PATH, "js/pn4web/pbar.png", unloaded, loaded);

  // state
  this.STATE                = {};
  this.STATE.CLAZZ          = 'state';
  this.STATE.PRECISION      = 4;
  this.STATE.MAX_PBAR_WIDTH = 78;  // middle segment only
  this.STATE.PBAR           = this.PBAR;
  this.STATE.PBAR_ACTIVE    = this.PBAR_ACTIVE;
  this.STATE.PBAR_INACTIVE  = this.PBAR_INACTIVE;

  // state for predecessor nodes in dynamic bayesian networks
  this.PRE_STATE                = {};
  this.PRE_STATE.CLAZZ          = 'state';
  this.PRE_STATE.PRECISION      = 4;
  this.PRE_STATE.MAX_PBAR_WIDTH = 78;  // middle segment only
  this.PRE_STATE.PBAR           = this.PRE_PBAR;
  this.PRE_STATE.PBAR_ACTIVE    = this.PRE_PBAR_ACTIVE;
  this.PRE_STATE.PBAR_INACTIVE  = this.PRE_PBAR_INACTIVE;

  // state
  this.DISTRIBUTION             = {};
  this.DISTRIBUTION.CLAZZ       = 'distribution_graph';
  this.DISTRIBUTION.NODE_HEIGHT = 50;
  this.DISTRIBUTION.WIDTH       = 80;
  this.DISTRIBUTION.HEIGHT      = 75;
  this.DISTRIBUTION.GRAPH_COLOR = 'rgb(0,162,0)';
  this.DISTRIBUTION.FIXED_COLOR = 'rgb(0,127,255)';
  this.DISTRIBUTION.TEXT_FONT   = 'normal 9px sans-serif';
  this.DISTRIBUTION.TEXT_COLOR  = 'rgb(192,192,192)';
  this.DISTRIBUTION.AXIS_COLOR  = 'rgb(192,192,192)';
  this.DISTRIBUTION.AXIS_WIDTH  = 1;
  this.DISTRIBUTION.RESOLUTION  = 50;

  // content (e.g. nodes contained in a clique)
  this.CONTENT       = {};
  this.CONTENT.CLAZZ = 'content';

  // links (directed)
  this.LINK            = {};
  this.LINK.CLAZZ      = 'link';
  this.LINK.THICKNESS  = 24;    // all three tiles must have this height (incl. borders)
  this.LINK.MIN_LENGTH = 36;    // width of left + right tile
  this.LINK.MAX_LENGTH = 254;   // middle segment only
  this.LINK.STRETCH    = false; // stretch shadow and background images?
  this.LINK.BORDER     = { E:26,  W:10, N:12, S:12 };
  this.LINK.SPLIT      = { E:104, W:39, N:48, S:48 };
  this.LINK.BG         = loadImg(this.BASE_PATH, "js/pn4web/link.png",        unloaded, loaded);
  this.LINK.SHADOW     = loadImg(this.BASE_PATH, "js/pn4web/link_shadow.png", unloaded, loaded); 

  // dynamic links (directed)
  this.DYNLINK            = {};
  this.DYNLINK.CLAZZ      = 'link';
  this.DYNLINK.THICKNESS  = 24;   // all three tiles must have this height
  this.DYNLINK.MIN_LENGTH = 36;   // width of left + right tile
  this.DYNLINK.MAX_LENGTH = 254;  // middle segment only
  this.DYNLINK.STRETCH    = false;   // stretch shadow and background images?
  this.DYNLINK.BORDER     = { E:26,  W:10, N:12, S:12 };
  this.DYNLINK.SPLIT      = { E:104, W:39, N:48, S:48 };
  this.DYNLINK.BG         = loadImg(this.BASE_PATH, "js/pn4web/dlink.png",       unloaded, loaded);
  this.DYNLINK.SHADOW     = loadImg(this.BASE_PATH, "js/pn4web/link_shadow.png", unloaded, loaded); 

  // links (undirected)
  this.RELATION            = {};
  this.RELATION.CLAZZ      = 'relation';
  this.RELATION.THICKNESS  = 24;   // all three tiles must have this height
  this.RELATION.MIN_LENGTH = 32;   // width of left + right tile
  this.RELATION.MAX_LENGTH = 254;  // middle segment only
  this.RELATION.STRETCH    = false;   // stretch shadow and background images?
  this.RELATION.BORDER     = { E:16, W:16, N:12, S:12 };     // CSS3
  this.RELATION.SPLIT      = { E:64, W:64, N:48, S:48 };     // CSS3
  this.RELATION.BG         = loadImg(this.BASE_PATH, "js/pn4web/ulink.png",        unloaded, loaded);
  this.RELATION.SHADOW     = loadImg(this.BASE_PATH, "js/pn4web/ulink_shadow.png", unloaded, loaded);

  // probability of evidence
  this.TIME                = {};
  this.TIME.CLAZZ          = 'time';
  this.TIME.PRECISION      = 4;
  this.TIME.POPUP          = this.POPUP;

  // probability of evidence
  this.PE                = {};
  this.PE.CLAZZ          = 'pe';
  this.PE.PRECISION      = 4;
  this.PE.MAX_PBAR_WIDTH = 42;  // middle segment only
  this.PE.PBAR           = this.PEPBAR;
  this.PE.POPUP          = this.POPUP;

  // table of samples
  this.SAMPLES                = {};
  this.SAMPLES.CLAZZ          = 'samples';
  this.SAMPLES.PRECISION      = 4;    // for weights only

  // probability table
  this.PTABLE                       = {};
  this.PTABLE.CLAZZ                 = 'pt';
  this.PTABLE.PRECISION             = 4;
  this.PTABLE.MAX_PBAR_WIDTH        = 40;   // middle segment only
  this.PTABLE.MAX_DIM_NAME_LENGTH   = 15;
  this.PTABLE.MAX_STATE_NAME_LENGTH = 10;
  this.PTABLE.PBAR                  = this.PTPBAR;
  this.PTABLE.ADD_STATE             = "js/pn4web/state_add.png";
  this.PTABLE.ADD_STATE_HL          = "js/pn4web/state_add_hl.png";
  this.PTABLE.RM_STATE              = "js/pn4web/state_rm.png";
  this.PTABLE.RM_STATE_HL           = "js/pn4web/state_rm_hl.png";
  this.PTABLE.ADD_PARENT            = "js/pn4web/parent_add.png";
  this.PTABLE.ADD_PARENT_HL         = "js/pn4web/parent_add_hl.png";
  this.PTABLE.RM_PARENT             = "js/pn4web/parent_rm.png";
  this.PTABLE.RM_PARENT_HL          = "js/pn4web/parent_rm_hl.png";

  // parameter table
  this.PARAMTABLE                       = {};
  this.PARAMTABLE.CLAZZ                 = 'pt';
  this.PARAMTABLE.PRECISION             = 4;
  this.PARAMTABLE.MAX_DIM_NAME_LENGTH   = 15;
  this.PARAMTABLE.MAX_STATE_NAME_LENGTH = 10;
  this.PARAMTABLE.ADD_PARENT            = "js/pn4web/parent_add.png";
  this.PARAMTABLE.ADD_PARENT_HL         = "js/pn4web/parent_add_hl.png";
  this.PARAMTABLE.RM_PARENT             = "js/pn4web/parent_rm.png";
  this.PARAMTABLE.RM_PARENT_HL          = "js/pn4web/parent_rm_hl.png";
  this.PARAMTABLE.RM_STATE              = "js/pn4web/state_rm.png";    // (removes cont. dimensions)
  this.PARAMTABLE.RM_STATE_HL           = "js/pn4web/state_rm_hl.png"; // (removes cont. dimensions)

  // LP-potential table
  this.LPTABLE                       = {};
  this.LPTABLE.CLAZZ                 = 'pt';
  this.LPTABLE.MAX_DIM_NAME_LENGTH   = 15;
  this.LPTABLE.MAX_STATE_NAME_LENGTH = 10;
  this.LPTABLE.ISCOMPLEX             = true;
  this.LPTABLE.CONTENT_TYPE_NAME     = "LP-Potential";

  // LP-potential table
  this.POSTBAGTABLE                       = {};
  this.POSTBAGTABLE.CLAZZ                 = 'pt';
  this.POSTBAGTABLE.MAX_DIM_NAME_LENGTH   = this.LPTABLE.MAX_DIM_NAME_LENGTH;
  this.POSTBAGTABLE.MAX_STATE_NAME_LENGTH = this.LPTABLE.MAX_STATE_NAME_LENGTH;
  this.POSTBAGTABLE.ISCOMPLEX             = true;
  this.POSTBAGTABLE.CONTENT_TYPE_NAME     = "Postbag";

  // full joint probability table (not shown in a popup)
  this.FJPTABLE                = {};
  this.FJPTABLE.CLAZZ          = 'fjpt';
  this.FJPTABLE.PRECISION      = 4;
  this.FJPTABLE.MAX_PBAR_WIDTH = 40;   // middle segment only
  this.FJPTABLE.PBAR           = this.FJPTPBAR;

  // cpt popup
  this.CPT                      = {};
  this.CPT.CLAZZ                = 'cpt';
  this.CPT.PT_ATTR              = 'cpt';   // name of the node attribute holding the table
  this.CPT.POPUP                = this.POPUP;
  this.CPT.PTABLE               = this.PTABLE;
  this.CPT.CALLBACKS            = {};
  this.CPT.CALLBACKS.DIM        = {};
  this.CPT.CALLBACKS.DIM.NAME   = function(context, pt) { 
    var net = context.net;
    me.repaint(net); 
  };
  this.CPT.CALLBACKS.DIM.ADD    = function(context, pt, dim) { 
    var net = context.net, node = context.node;
    me.undisplay(net);
    net.addLink(dim, node);
    node.adaptCPT(net.getParents(node, true)); 
    if (net.prop) net.prop.recreate(); 
    me.repaint(net); 
    me.propagate(net); 
  };
  this.CPT.CALLBACKS.DIM.RM     = function(context, pt, d) { 
    var net = context.net, node = context.node;
    me.undisplay(net);
    net.rmLink(pt.dims[d], node);
    node.adaptCPT(net.getParents(node, true));
    if (net.prop) net.prop.recreate(); 
    me.propagate(net, true);  
  };
  this.CPT.CALLBACKS.STATE      = {};
  this.CPT.CALLBACKS.STATE.NAME = function(context, pt, index) { 
    var net = context.net, node = context.node;
    node.name = pt.dims[0].name; // (e.g. for CGNodes pt.dims[0] != node)
    me.repaint(net);
  };
  this.CPT.CALLBACKS.STATE.ADD  = function(context, pt) { 
    var net = context.net, node = context.node;
    var s = node.states.length; 
    if (s < 8) {
      node.states.push(new State(s, "State " + (s+1))); 
      node.adaptCPT(net.getParents(node, true));
      for (var chId in net.children[node.id]) {
        var child = net.nodes[chId];
        child.adaptCPT(net.getParents(child, true));
      }
      if (net.prop) net.prop.reload(); 
      me.propagate(net, true); 
    } 
  };
  this.CPT.CALLBACKS.STATE.RM   = function(context, pt) { 
    var net = context.net, node = context.node;
    var s = node.states.length; 
    if (s > 2) {
      node.states.pop();
      node.adaptCPT(net.getParents(node, true));
      for (var chId in net.children[node.id]) {
        var child = net.nodes[chId];
        child.adaptCPT(net.getParents(child, true));
      }
      if (net.prop) net.prop.reload(); 
      me.propagate(net, true); 
    }
  };
  this.CPT.CALLBACKS.CELL       = {};
  this.CPT.CALLBACKS.CELL.VALUE = function(context, pt, index) { 
    var net = context.net;
    if (net.prop) net.prop.reload(); 
    me.propagate(net); 
  };

  // conditional continuous parameter table popup
  this.CGT                      = {};
  this.CGT.CLAZZ                = 'cpt';
  this.CGT.PT_ATTR              = 'cgt';   // name of the node attribute holding the table
  this.CGT.POPUP                = this.POPUP;
  this.CGT.PTABLE               = this.PARAMTABLE;
  this.CGT.CALLBACKS            = {};
  this.CGT.CALLBACKS.DIM        = {};
  this.CGT.CALLBACKS.DIM.NAME   = this.CPT.CALLBACKS.DIM.NAME;
  this.CGT.CALLBACKS.DIM.ADD    = this.CPT.CALLBACKS.DIM.ADD;
  this.CGT.CALLBACKS.DIM.RM     = this.CPT.CALLBACKS.DIM.RM;
  this.CGT.CALLBACKS.CELL       = {};
  this.CGT.CALLBACKS.CELL.VALUE = this.CPT.CALLBACKS.CELL.VALUE;
  this.CGT.CALLBACKS.STATE      = {};
  this.CGT.CALLBACKS.STATE.RM   = function(context, pt) { 
    var net = context.net, node = context.node;
    var s = pt.dims[0].states.length; 
    if (s > 2) {                         // (first two states are "mean" and "variance")
      var dim = pt.dims[0].states.pop(); // (this state is in fact a continuous dimension)
      me.undisplay(net);
      net.rmLink(dim, node);
      node.adaptCPT(net.getParents(node, true));
      if (net.prop) net.prop.recreate(); 
      me.propagate(net, true);  
    }
  };

  // jpt popup for cliques
  this.JPT           = {};
  this.JPT.CLAZZ     = 'jpt';
  this.JPT.PT_ATTR   = 'jpt';   // name of the node attribute holding the table
  this.JPT.POPUP     = this.POPUP;
  this.JPT.PTABLE    = this.PTABLE;
  this.JPT.CALLBACKS = null;    // jpt's are not editable

  // LP-Potential table popup for CG-clusters
  this.LPT           = {};
  this.LPT.CLAZZ     = 'lpt';
  this.LPT.PT_ATTR   = 'lpt';   // name of the node attribute holding the table
  this.LPT.POPUP     = this.POPUP;
  this.LPT.PTABLE    = this.LPTABLE;
  this.LPT.CALLBACKS = null;    // lp-potentials are not editable

  // Postbag tables popup for CG-clusters
  this.POSTBAG           = {};
  this.POSTBAG.CLAZZ     = 'post';
  this.POSTBAG.PT_ATTR   = 'post';  // name of the node attribute holding the table(s)
  this.POSTBAG.POPUP     = this.POPUP;
  this.POSTBAG.PTABLE    = this.POSTBAGTABLE;
  this.POSTBAG.CALLBACKS = null;    // postbag-tables are not editable

  // separators
  this.SEPARATOR        = {};
  this.SEPARATOR.CLAZZ  = 'sep';
  this.SEPARATOR.WIDTH  = 40;
  this.SEPARATOR.HEIGHT = 40;
  this.SEPARATOR.BORDER = { E:20, W:20, N:20, S:20 }; 
  this.SEPARATOR.SPLIT  = { E:80, W:80, N:80, S:80 };
  this.SEPARATOR.BG     = loadImg(this.BASE_PATH, "js/pn4web/separator.png",        unloaded, loaded);
  this.SEPARATOR.SHADOW = loadImg(this.BASE_PATH, "js/pn4web/separator_shadow.png", unloaded, loaded);
  this.SEPARATOR.PT     = this.JPT;

  // node
  this.NODE                 = {};
  this.NODE.CLAZZ           = 'node';
  this.NODE.MIN_NUM_OF_ROWS = 2;     // number of states that could be shown without additional height
  this.NODE.HEIGHT_PER_ROW  = 15;    // additional height per state needed if #states > MIN_NUM_OF_ROWS
  this.NODE.STRETCH         = false; // stretch shadow and background images?
  this.NODE.BORDER          = { E:58,  W:58,  N:40,  S:40 }; 
  this.NODE.SPLIT           = { E:232, W:232, N:160, S:160 };
  this.NODE.BG              = loadImg(this.BASE_PATH, "js/pn4web/node.png",        unloaded, loaded);
  this.NODE.FG              = loadImg(this.BASE_PATH, "js/pn4web/node_fg.png",     unloaded, loaded);
  this.NODE.SHADOW          = loadImg(this.BASE_PATH, "js/pn4web/node_shadow.png", unloaded, loaded);

  this.NODE.LINK            = this.LINK;
  this.NODE.DYNLINK         = this.DYNLINK;
  this.NODE.RELATION        = this.RELATION;
  this.NODE.STATE           = this.STATE;
  this.NODE.PT              = this.CPT;

  // CG-node (continuous node)
  this.CGNODE                 = {};
  this.CGNODE.CLAZZ           = 'node';
  this.CGNODE.STRETCH         = false;  // stretch shadow and background images?
  this.CGNODE.BORDER          = { E:58,  W:58,  N:40,  S:40 };
  this.CGNODE.SPLIT           = { E:232, W:232, N:160, S:160 };
  this.CGNODE.BG              = loadImg(this.BASE_PATH, "js/pn4web/node.png",        unloaded, loaded);
  this.CGNODE.FG              = loadImg(this.BASE_PATH, "js/pn4web/node_fg.png",     unloaded, loaded);
  this.CGNODE.SHADOW          = loadImg(this.BASE_PATH, "js/pn4web/node_shadow.png", unloaded, loaded);
  this.CGNODE.LINK            = this.LINK;
  this.CGNODE.DYNLINK         = this.DYNLINK;
  this.CGNODE.RELATION        = this.RELATION;
  this.CGNODE.DISTRIBUTION    = this.DISTRIBUTION;
  this.CGNODE.PT              = this.CGT;

  // predecessor node for dynamic bayesian networks
  this.PRE_NODE                 = {};
  this.PRE_NODE.CLAZZ           = 'node';
  this.PRE_NODE.MIN_NUM_OF_ROWS = 2;     // number of states that could be shown without additional height
  this.PRE_NODE.HEIGHT_PER_ROW  = 15;    // additional height per state needed if #states > MIN_NUM_OF_ROWS
  this.PRE_NODE.STRETCH         = false; // stretch shadow and background images?
  this.PRE_NODE.BORDER          = { E:58,  W:58,  N:40,  S:40 };
  this.PRE_NODE.SPLIT           = { E:232, W:232, N:160, S:160 };
  this.PRE_NODE.BG              = loadImg(this.BASE_PATH, "js/pn4web/prenode.png",     unloaded, loaded);
  this.PRE_NODE.FG              = loadImg(this.BASE_PATH, "js/pn4web/node_fg.png",     unloaded, loaded);
  this.PRE_NODE.SHADOW          = loadImg(this.BASE_PATH, "js/pn4web/node_shadow.png", unloaded, loaded);
  this.PRE_NODE.LINK            = this.LINK;
  this.PRE_NODE.DYNLINK         = this.DYNLINK;
  this.PRE_NODE.RELATION        = this.RELATION;
  this.PRE_NODE.STATE           = this.PRE_STATE;
  this.PRE_NODE.PT              = this.CPT;

  // clique
  this.CLIQUE                 = {};
  this.CLIQUE.CLAZZ           = 'node';
  this.CLIQUE.MIN_NUM_OF_ROWS = 2;     // number of nodes that could be shown without additional height
  this.CLIQUE.HEIGHT_PER_ROW  = 15;    // additional height per state needed if #nodes > MIN_NUM_OF_ROWS
  this.CLIQUE.STRETCH         = false; // stretch shadow and background images?
  this.CLIQUE.BORDER          = { E:58,  W:58,  N:40,  S:40 };  // CSS3
  this.CLIQUE.SPLIT           = { E:232, W:232, N:160, S:160 }; // CSS3
  this.CLIQUE.BG              = loadImg(this.BASE_PATH, "js/pn4web/node.png",        unloaded, loaded);
  this.CLIQUE.FG              = loadImg(this.BASE_PATH, "js/pn4web/node_fg.png",     unloaded, loaded);
  this.CLIQUE.SHADOW          = loadImg(this.BASE_PATH, "js/pn4web/node_shadow.png", unloaded, loaded);
  this.CLIQUE.LINK            = this.LINK;
  this.CLIQUE.DYNLINK         = this.DYNLINK;
  this.CLIQUE.RELATION        = this.RELATION;
  this.CLIQUE.LINKDATA        = this.SEPARATOR;
  this.CLIQUE.CONTENT         = this.CONTENT;
  this.CLIQUE.PT              = this.JPT;

  // CG-clusters (hybrid networks)
  this.CLUSTER                 = {};
  this.CLUSTER.CLAZZ           = 'node';
  this.CLUSTER.MIN_NUM_OF_ROWS = 2;     // number of nodes that could be shown without additional height
  this.CLUSTER.HEIGHT_PER_ROW  = 15;    // additional height per state needed if #nodes > MIN_NUM_OF_ROWS
  this.CLUSTER.STRETCH         = false; // stretch shadow and background images?
  this.CLUSTER.BORDER          = { E:58,  W:58,  N:40,  S:40 };
  this.CLUSTER.SPLIT           = { E:232, W:232, N:160, S:160 };
  this.CLUSTER.BG              = loadImg(this.BASE_PATH, "js/pn4web/node.png",        unloaded, loaded);
  this.CLUSTER.FG              = loadImg(this.BASE_PATH, "js/pn4web/node_fg.png",     unloaded, loaded);
  this.CLUSTER.SHADOW          = loadImg(this.BASE_PATH, "js/pn4web/node_shadow.png", unloaded, loaded);
  this.CLUSTER.LINK            = this.LINK;
  this.CLUSTER.DYNLINK         = this.DYNLINK;
  this.CLUSTER.RELATION        = this.RELATION;
  this.CLUSTER.LINKDATA        = this.SEPARATOR;
  this.CLUSTER.CONTENT         = this.CONTENT;
  this.CLUSTER.PT              = this.LPT;
  this.CLUSTER.PTS             = this.POSTBAG;

  // markov blanket node (gibbs sampler)
  this.MARKOVBLANKET                 = {};
  this.MARKOVBLANKET.CLAZZ           = 'node';
  this.MARKOVBLANKET.MIN_NUM_OF_ROWS = 2;     // number of nodes that could be shown without additional height
  this.MARKOVBLANKET.HEIGHT_PER_ROW  = 15;    // additional height per state needed if #nodes > MIN_NUM_OF_ROWS
  this.MARKOVBLANKET.STRETCH         = false; // stretch shadow and background images?
  this.MARKOVBLANKET.BORDER          = { E:58,  W:58,  N:40,  S:40 };
  this.MARKOVBLANKET.SPLIT           = { E:232, W:232, N:160, S:160 };
  this.MARKOVBLANKET.BG              = loadImg(this.BASE_PATH, "js/pn4web/node.png",        unloaded, loaded);
  this.MARKOVBLANKET.FG              = loadImg(this.BASE_PATH, "js/pn4web/node_fg.png",     unloaded, loaded);
  this.MARKOVBLANKET.SHADOW          = loadImg(this.BASE_PATH, "js/pn4web/node_shadow.png", unloaded, loaded);
  this.MARKOVBLANKET.LINK            = this.LINK;
  this.MARKOVBLANKET.DYNLINK         = this.DYNLINK;
  this.MARKOVBLANKET.RELATION        = this.RELATION;
  this.MARKOVBLANKET.CONTENT         = this.CONTENT;
  this.MARKOVBLANKET.PT              = this.JPT;

  // bayesian network
  this.NET          = {};
  this.NET.CLAZZ    = 'net';
  this.NET.NODE     = this.NODE;
  this.NET.CGNODE   = this.CGNODE;
  this.NET.PRE_NODE = this.PRE_NODE;

  // junction tree
  this.JTREE        = {};
  this.JTREE.CLAZZ  = 'net';
  this.JTREE.NODE   = this.CLIQUE;
  this.JTREE.CGNODE = this.CLUSTER;

  // loopy belief
  this.LOOPYBELIEF = this.JTREE; // using the same style as a junction-tree

  // markov blanket network (gibbs sampler)
  this.MARKOVBLANKETNET        = {};
  this.MARKOVBLANKETNET.CLAZZ  = 'net';
  this.MARKOVBLANKETNET.NODE   = this.MARKOVBLANKET;

  // shadows (general settings)
  this.SHADOW         = {};
  this.SHADOW.XOFFSET = shadowXOffset;
  this.SHADOW.YOFFSET = shadowYOffset;

  // add dragging functionality 
  var drag    = function(evnt) { me.DRAG.drag(evnt);    }; // wrappers (otherwise "this" inside
  var release = function(evnt) { me.DRAG.release(evnt); }; // drag/release would be the document)
  addEventListener(document, "mousemove", drag);
  addEventListener(document, "mouseup",   release);
  
  // first trial
  loaded(null);
}


// BASIC ELEMENTS ===================================================

/**
 * Create a button which is highlighted while the mouse is over it.
 * @param parent  parent element
 * @param id      element-id
 * @param clazz   element-class
 * @param img     normal image source
 * @param imgHL   highlighted image soure
 * @returns       image element
 */
PN4Web.prototype.displayButton = function (parent, id, clazz, img, imgHL, toolTip)
{
  var elem   = appendNewElement(parent, 'img', id, clazz);
  elem.src   = this.BASE_PATH + img;
  elem.title = toolTip;
  var me     = this;
  var enter  = function(evnt) { elem.src = me.BASE_PATH + imgHL; };
  var leave  = function(evnt) { elem.src = me.BASE_PATH + img;   };
  addEventListener(elem, "mouseover", enter);
  addEventListener(elem, "mouseout",  leave);
  return (elem);
};


// REPAINT / REFRESH ================================================

/**
 * Undisplay networks (necessary if nodes/links should be removed)
 * @param net  bayesian network
 */
PN4Web.prototype.undisplay = function (net)
{
  // nothing to do
  if (!net) return;

  // clear net
  this.undisplayNet(net);

  // no propagator
  if (!net.prop) return;

  // clear junction tree
  if (document.getElementById(net.id + '.jtree') ||
      document.getElementById(net.id + '.lbp')) {
    this.clearNet(net.prop);
  }

  // clear markov blanket net (gibb sampler)
  if (document.getElementById(net.id + '.gibbs.mbnet')) {
    this.clearNet(net.prop.mbNet);
  }
};

/**
 * Repaint network and propagator / samples 
 * (if a propagator or samples are displayed)
 * @param net  bayesian network
 */
PN4Web.prototype.repaint = function (net)
{
  // display net
  this.displayNet(net, this.NET);

  // no propagator
  if (!net.prop) return;

  // display probability of evidence
  this.displayPePopup(net, (net.prop.max ? 'P(C|e)' : 'P(e|M)'), net.prop.pe, this.PE, 0,0);

  // display time for dynamic networks / propagators
  if (net.prop.t != undefined) {
    this.displayPePopup(net, 'Time', net.prop.t, this.TIME, 10000,10000);
  }

  // update full joint probability table
  if (document.getElementById(net.id + '.fjp')) {
    document.getElementById(net.prop.id).innerHTML = ""; // clear
    this.displayPTable(net.prop.fjpt, net.prop.id + '.table', 
		       document.getElementById(net.prop.id), PN4W.FJPTABLE);
  }

  // update junction tree
  if (document.getElementById(net.id + '.jtree') ||
      document.getElementById(net.id + '.lbp')) {
    this.displayNet(net.prop, this.JTREE);
  }
      
  // update samples
  if (document.getElementById(net.id + '.lws') ||
      document.getElementById(net.id + '.gibbs')) {
    document.getElementById(net.prop.id).innerHTML = ""; // clear
    this.displaySamples(net.prop.samples, net.prop.id + '.table', 
			document.getElementById(net.prop.id), PN4W.SAMPLES);
  }

  // update markov blanket network (gibbs-sampler)
  if (document.getElementById(net.id + '.gibbs.mbnet')) {
    this.displayNet(net.prop.mbNet, this.MARKOVBLANKETNET);
  }
};

/**
 * Update probabilities only.
 * @param net  bayesian network
 */
PN4Web.prototype.refresh = function (net)
{
  // node states
  for (var nId in net.nodes) {
    var node = net.nodes[nId];
    if (node.isContinuous) {
      this.displayDistribution(net, node, this.NET.CGNODE.DISTRIBUTION);
    } else if (net.succ && net.succ[nId]) { // node has successor -> its a predecessor node
      this.displayStates(net, node, this.NET.PRE_NODE.STATE);
    } else {
      this.displayStates(net, node, this.NET.NODE.STATE);
    }
  }
  
  // no propagator
  if (!net.prop) return;

  // P(e|m)
  this.displayPePopup(net, (net.prop.max ? 'P(C|e)' : 'P(e|M)'), net.prop.pe, this.PE, 0,0);

  // display time for dynamic networks / propagators
  if (net.prop.t != undefined) {
    this.displayPePopup(net, 'Time', net.prop.t, this.TIME, 10000,10000);
  }

  // update full joint probability table
  if (document.getElementById(net.id + '.fjp')) {
    document.getElementById(net.prop.id).innerHTML = ""; // clear
    this.displayPTable(net.prop.fjpt, net.prop.id + '.table', 
		       document.getElementById(net.prop.id), PN4W.FJPTABLE);
  }

  // update junction tree
  if (document.getElementById(net.id + '.jtree') ||
      document.getElementById(net.id + '.lbp')) {
    for (var cId in net.prop.nodes) {
      var clique = net.prop.nodes[cId];
      var type = (clique.isContinuous ? this.JTREE.CGNODE : this.JTREE.NODE);
      var clPopupElem = document.getElementById(net.prop.id + '.' + cId + '.pt');
      if (clPopupElem) {
        this.displayPTPopup(net.prop, clique, type.PT);
      }
      for (var chId in net.prop.children[cId]) {
        var sep = net.prop.children[cId][chId];
        var sepPopupElem = document.getElementById(net.prop.id + '.' + sep.id + '.pt');
        if (sepPopupElem) {
          this.displayPTPopup(net.prop, sep, type.LINKDATA.PT);	  
        }
      }
    }
  }
      
  // update samples
  if (document.getElementById(net.id + '.lws') ||
      document.getElementById(net.id + '.gibbs')) 
  {
    document.getElementById(net.prop.id).innerHTML = ""; // clear
    this.displaySamples(net.prop.samples, net.prop.id + '.table', 
			document.getElementById(net.prop.id), PN4W.SAMPLES);
  }

  // (no need to update markov blanket network (gibbs-sampler))
};


// PROPAGATE AND UPDATE =============================================

/**
 * Propagate and update the given bayesian network
 * @param net      bayesian network ('net.prop' set)
 * @param repaint  complete repaint?
 */
PN4Web.prototype.propagate = function (net, repaint)
{
  if (net.prop) {
    net.prop.propagate(true, true);
  }
  (repaint ? this.repaint(net) : this.refresh(net)); // (even if there is no propagator!)
};

/**
 * Update graphs
 */
PN4Web.prototype.updateGraphs = function (net) 
{
  this.displayGraphs(net, this.posteriors[net.id], net.prop.t, this.GRAPH);
};

/**
 * Forward propagation using the rate given by net.rate
 * @param net  dynamic bayesian network (having a dynamic 
 *             propagator associated in net.prop)
 */
PN4Web.prototype.play = function (net) 
{
  if (!this.running[net.id]) {
    this.running[net.id] = true;
    var me = this;
    var run = function (){
      if (me.running[net.id]) { 
        setTimeout(run, net.rate); // (queues itself, it's not recursive!)
        me.storePosteriors(net);   // before moving to the next slice
        net.prop.next();
        me.propagate(net);
        me.updateGraphs(net);
      }
    };
    run();
  }
};

/**
 * Backward propagation using the rate given by net.rate
 * @param net  dynamic bayesian network (having a dynamic 
 *             propagator associated in net.prop)
 */
PN4Web.prototype.playback = function (net) 
{ 
  if (!this.running[net.id]) {
    this.running[net.id] = true;
    net.rmEvidence();
    var me = this;
    var run = function (){
      if (net.prop.t <= 0) {
        delete (me.running[net.id]);
      } else if (me.running[net.id]) {
        setTimeout(run, net.rate); // (queues itself, it's not recursive!)
        me.storePosteriors(net);   // before moving to the previous slice
        net.prop.prev();
        me.propagate(net);
        me.updateGraphs(net);
      }
    };
    run();
  }
};

/**
 * Pause running forward or backward propagation.
 * @param net  dynamic bayesian network 
 */
PN4Web.prototype.pause = function (net) 
{ 
  delete (this.running[net.id]); 
};

/**
 * Stop running forward or backward propagation
 * (re-initializes propagator and clears collected evidence)
 * @param net  dynamic bayesian network (having a dynamic 
 *             propagator associated in net.prop)
 */
PN4Web.prototype.stop = function (net) 
{ 
  delete (this.running[net.id]);
  net.prop.stop();
  this.posteriors[net.id].initTimeData(); // clear data (recreate object)
  this.propagate(net);
  this.updateGraphs(net);
};


/**
 * Single step forward propagation (stops running 
 * forward/backward propagation)
 * @param net  dynamic bayesian network (having a dynamic 
 *             propagator associated in net.prop)
 */
PN4Web.prototype.next = function (net) 
{ 
  delete (this.running[net.id]);
  this.storePosteriors(net); // before moving to the next slice
  net.prop.next();
  this.propagate(net);
  this.updateGraphs(net);
};

/**
 * Single step backward propagation (stops running 
 * forward/backward propagation)
 * @param net  dynamic bayesian network (having a dynamic 
 *             propagator associated in net.prop)
 */
PN4Web.prototype.prev = function (net) 
{ 
  delete (this.running[net.id]);
  if (net.prop.t <= 0) return;
  this.storePosteriors(net); // before moving to the previous slice
  net.prop.prev();
  net.rmEvidence();
  this.propagate(net);
  this.updateGraphs(net);
};

/**
 * Double speed (net.rate = net.rate/2).
 * @param net  dynamic bayesian network
 */
PN4Web.prototype.faster = function (net) 
{
  if (this.running[net.id] && net.rate > 1) {
    net.rate = net.rate >> 2;
  }
};

/**
 * Half speed (net.rate = net.rate * 2)
 * @param net  dynamic bayesian network
 */
PN4Web.prototype.slower = function (net) 
{
  if (this.running[net.id] && net.rate < 2048) {
    net.rate = net.rate << 2;
  }
};

/**
 * Store posteriors for the given network
 * @param net  dynamic bayesian network (having a dynamic 
 *             propagator associated in net.prop)
 */
PN4Web.prototype.storePosteriors = function (net) 
{ 
  for (var nId in net.nodes) {
    this.posteriors[net.id].store(net.prop.t, nId, net.nodes[nId].p.values.slice());
  }
};


// DRAGGING =========================================================

/**
 * Grab element for dragging (mouse down) 
 * @param outer  outer html element (the element to be dragged)
 * @param inner  inner html element (used to get the actual size)
 * @param evnt   event
 */
PN4Web.prototype.grab = function (outer, inner, evnt)
{
  if (!evnt) evnt = window.event;
  preventDefault(evnt);
  this.DRAG.x0      = pageX(evnt);
  this.DRAG.y0      = pageY(evnt);
  this.DRAG.xOffset = this.DRAG.x0 - getLeft(outer);
  this.DRAG.yOffset = this.DRAG.y0 -  getTop(outer);
};

/**
 * Drag element (mouse move while button pressed) 
 * @param outer  outer html element (the element to be dragged)
 * @param inner  inner html element (used to get the actual size)
 * @param evnt   event
 */
PN4Web.prototype.drag = function (outer, inner, evnt)
{
  if (!evnt) evnt = window.event;
  preventDefault(evnt);

  // notice: if outer is already removed then it has no parent anymore
  var frameElem = outer.parentNode;
  if (frameElem == null) { this.release(evnt); return; }

  // drag (set position)
  var xMax  = frameElem.scrollWidth  - inner.offsetWidth  - this.SHADOW.XOFFSET;
  var yMax  = frameElem.scrollHeight - inner.offsetHeight - this.SHADOW.YOFFSET;
  var x     = Math.max(0, Math.min(xMax, pageX(evnt) - this.DRAG.xOffset - getLeft(frameElem)));
  var y     = Math.max(0, Math.min(yMax, pageY(evnt) - this.DRAG.yOffset -  getTop(frameElem)));
  outer.style.left = x;
  outer.style.top  = y; 
};

/**
 * Drag node (mouse move while button pressed)
 * @param outer  outer html element (the element to be dragged)
 * @param inner  inner html element (used to get the actual size)
 * @param net    bayesian network
 * @param node   grabbed node 
 * @param type   node type, e.g. this.NODE
 * @param evnt   event
 */
PN4Web.prototype.dragNode = function (outer, inner, net, node, type, evnt)
{
  if (!evnt) evnt = window.event;
  this.drag(outer, inner, evnt);

  // store coordinates
  node.x = outer.offsetLeft;
  node.y = outer.offsetTop;

  // adjust links
  for (var paId in net.parents[node.id]) {
    if (net.succ && net.succ[paId]) {    // this parent is a predecessor node -> dynamic link
      this.displayLink(net, net.getNode(paId), node, type.DYNLINK, type.LINKDATA);
    } else {
      this.displayLink(net, net.getNode(paId), node, type.LINK, type.LINKDATA);
    }
  }
  for (var chId in net.children[node.id]) {
    if (net.succ && net.succ[node.id]) { // dragged node is a predecessor node -> dynamic link
      this.displayLink(net, node, net.getNode(chId), type.DYNLINK, type.LINKDATA);
    } else {
      this.displayLink(net, node, net.getNode(chId), type.LINK, type.LINKDATA);
    }
  }
  for (var neId in net.siblings[node.id]) {
    var other = net.getNode(neId);
    (node.id < other.id 
     ? this.displayLink(net, node, other, type.RELATION, type.LINKDATA) 
     : this.displayLink(net, other, node, type.RELATION, type.LINKDATA));
  }
};

/** 
 * Release (mouse up) 
 */
PN4Web.prototype.release = function (evnt)
{  
  if (!evnt) evnt = window.event;
  preventDefault(evnt);
  this.DRAG.drag    = dummyEventHandler;
  this.DRAG.release = dummyEventHandler;
};
