/*
 * -------------------------------------------------------------------------
 *
 *  pn4webDecomposition.js Copyright (C) 2012, 
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *   Author: Patrick Rammelt 
 *   E-Mail: patrick.rammelt@go4more.de
 *   Site:   http://www.patricks-seite.de
 * -------------------------------------------------------------------------
 *
 * Graph decomposition with evolutionary algorithms.
 * (These types are fully independent of display functionalities)
 */

// EVOLUTIONARY OPTIMIZATION ========================================

/**
 * Constructor for an evolutionary optimizer.
 * <dl><dt>The recombinator needs at least these methods:</dt>
 *     <dd>c = recombinator.conjugate(a, b): creates a new child 
 *         individual c from two parent individuals a and b</dd>
 *     <dd>recombinator.isSymmetric(): Is conjugate(a,b) the same
 *         as conjugate(b,a)?</dd>
 *     <dt>The evaluator needs at least these methods:</dt>
 *     <dd>evaluator.fitness(a): Returns the fitness value for the
 *         given individual a.
 * </dl>
 * @param individuals         initial set of individuals
 * @param recombinator        recombination operator used to 
 *                            create offspring individuals by
 *                            the combination of two parent
 *                            individuals (including mutations)
 * @param evaluator           fitness rating operator which is
 *                            used to associate a fitness value
 *                            to each individual
 * @param maxGenerations      max number of generations 
 * @param maxIndividuals      max number of individuals (worst 
 *                            individuals will be removed from 
 *                            the population)
 * @param maxOffspring        max number of new individuals
 *                            created per generation
 * @param agingPerGeneration  aging value added to the fitness of
 *                            each individual for each generation
 *                            it survives
 * @param satisfyingFittness  satisfying fitness value (stop 
 *                            process if the best individual(s)
 *                            have at least this fitness value
 * @param delay:              (min) delay in [ms] between two iterations;
 */
function EA (individuals, recombinator, evaluator, 
             maxGenerations, maxIndividuals, maxOffspring, 
             agingPerGeneration, satisfyingFittness, delay)
{
  if (recombinator != undefined) {
    this.initEA(individuals, recombinator, evaluator, 
                maxGenerations, maxIndividuals, maxOffspring, 
                agingPerGeneration, satisfyingFittness, delay);
  }
}

/* Inheritance */
//EA.prototype             = new <ParentClass>();
//EA.prototype.parent      = <ParentClass>.prototype;
EA.prototype.constructor = EA;

/**
 * (Re-)initialize this evolutionary optimizer
 * (see constructor of this {@link EA})
 */
EA.prototype.initEA = function (individuals, recombinator, evaluator, 
                                maxGenerations, maxIndividuals, maxOffspring, 
                                agingPerGeneration, satisfyingFittness, delay)
{
  // optimization parameters / helper objects
  this.recombinator       = recombinator;        // recombinator + mutator
  this.evaluator          = evaluator;           // fitness evaluator
  this.maxGenerations     = maxGenerations;      // max. number of generations
  this.maxIndividuals     = maxIndividuals;      // max. number of individuals surviving in each generation
  this.maxOffspring       = maxOffspring;        // max. number of offspring individuals per generation
  this.agingPerGeneration = agingPerGeneration;  // fitness offset per generation for surviving individuals
  this.satisfyingFittness = satisfyingFittness;  // min. satisfying fitness (stop if reached)
  
  // control parameters
  this.delay = delay;      // time delay in [ms] between two generations (>= 0 means async; < 0 sync)
  this.generation = 0;     // current generation
  
  // internals
  this.individuals = individuals;  // array of individuals newly created in the current generation
  this.population  = null;         // population rated and sorted array of individuals (in a wrapper structure)
  
  // listeners
  this.listeners = {};
};

/**
 * Add a new listener.
 * @param listener  listener with methods:<br> 
 *                  'start()' called before the algorithm starts<br>
 *                  'step(i)' called after a new (the i-th) generation has
 *                            been created; returns 'false' if the process
 *                            should stop, true otherwise (if at least one 
 *                            listener returns 'false' the process stoppes)<br>
 *                  'stop(f)' called just before finishing ('f=true' means
 *                            process terminated regular - 'f=false' means
 *                            process has been cancelled i.e. one listener
 *                            returned 'false' for 'step(i)'<br>
 */
EA.prototype.addListener = function (listener)
{
 this.listeners[listener] = listener;
};

/**
 * Remove a listener.
 */
EA.prototype.rmListener = function (listener)
{
 delete (this.listeners[listener]);
};

/**
 * Optimzation. 
 */
EA.prototype.optimize = function ()
{
  // init
  this.generation = 0;
  this.population = new Array();
  
  // call listeners
  for (var l in this.listeners) {
    this.listeners[l].start();
  }
  
  // run optimization process
  this.run(); 
};

/**
 * Get the best individual after optimization has finished.
 * In async mode attach a listener and wait for the 'stop()' event.
 */
EA.prototype.getFittest = function()
{
  // return best individual
  return (this.population[0].individual);
};

/**
 * Actually run the optimization process (helper for 'optimize()').
 */
EA.prototype.run = function ()
{  
  // include new individuals into population
  this.assimilate();

  // check: satisfying fitness or max number of generations reached?
  if (this.generation >= this.maxGenerations ||
      this.population[0].fitness > this.satisfyingFittness) 
  {
    
    // inform listeners
    for (var l in this.listeners) {
      this.listeners[l].stop(true);
    }

  } else {

    // survival of the fittest (filter out worst individuals)
    this.filter();

    // callback called after all offspring individuals have been created  
    var me = this;
    var callback = function() {
      
      // call listeners
      var go = true;
      for (var l in me.listeners) {
        go &= me.listeners[l].step(me.generation);
      }

      // iterate
      me.generation++;
      if (go) {
        setTimeout(function(){ me.run(); }, me.delay); // (wrapper needed)
      } else {
        for (var l in me.listeners) {
          me.listeners[l].stop(false);
        }
      }
    };
    
    // create new offspring individuals
    this.individuals = new Array();
    this.proliferate(callback, 0,0);
    
  }
};

/**
 * Assimilate new 'individuals' into the 'population'
 */
EA.prototype.assimilate = function ()
{
  // incorporate individuals
  for (var i = 0; i < this.individuals.length; i++)
  {
    var fitness = this.evaluator.fitness(this.individuals[i]);
    this.population.push({ individual : this.individuals[i],
                           fitness    : fitness,
                           generation : this.generation });
  }
  
  // (re-)sort population  var me = this;
  var me = this;
  comp = function (a, b) 
  { 
    return ((b.fitness - b.generation * me.agingPerGeneration) - 
            (a.fitness - a.generation * me.agingPerGeneration));
  };
  this.population.sort(comp);
};

/** 
 * Create new individuals in 'this.individuals' by recombinating two 
 * individuals from 'this.population', also adding mutations to 
 * such new individuals. Because this could be time-consuming it
 * is done asnchronously if 'this.delay' &lt; 0.
 * @param callback    function called finished
 * @param i           iteration parameter (initially 0)
 * @param j           iteration parameter (initially 0)
 */
EA.prototype.proliferate = function (callback, i, j)
{
  // create new offspring from the i-th and the j-th individual in 'population'
  if (i != j) {
    var child = this.recombinator.conjugate(this.population[i].individual, 
                                            this.population[j].individual);
    if (child) {
      this.individuals.push(child);
    }
  }

  // iterate (async. emulation of a for(i...) { for(j...) { ... }} loop)
  j++;
  if (this.recombinator.isSymmetric() && j >= i)    { j = 0; i++; }      
  else if (j >= this.population.length)             { j = 0; i++; }
  
  var me = this;
  if (this.individuals.length >= this.maxOffspring) { callback(); }      // stop: enough children
  else if (i >= this.population.length)             { callback(); }      // stop "iteration"
  else { setTimeout(function(){ me.proliferate(callback, i, j); }, 0); } // (wrapper needed)
};

/**
 * "Survival of the fittest": remove worst individuals
 * such that the 'population' has at most 'maxIndividuals'.
 */
EA.prototype.filter = function ()
{
  this.population.splice(this.maxIndividuals, this.population.length - this.maxIndividuals);
};


// FITNESS EVALUATORS ===============================================
// FILL-IN ----------------------------------------------------------

/**
 * Constructor for a fitness evaluator for junction trees,
 * where the fitness is evaluated by the number of fill-ins.
 */
function FillInFitness ()
{
  this.initFillInFitness(); 
}

/* Inheritance */
//FillInFitness.prototype           = new <ParentClass>();
//FillInFitness.prototype.parent    = <ParentClass>.prototype;
FillInFitness.prototype.constructor = FillInFitness;

/** 
 * (Re-)Initialize this {@link FillInFitness}
 * (see constructor {@link FillInFitness})  
 */
FillInFitness.prototype.initFillInFitness = function () {};

/**
 * Evaluate the fitness of a junction tree by the number of fill-ins added 
 * to the original bayesian network in order to create that junction tree.
 */
FillInFitness.prototype.fitness = function (jt) 
{
  return (-jt.fillins); // - Math.random()*0.25); // (add some random offset)
};

// RECOMBINATORS ====================================================
// REAL JT RECOMBINATOR ---------------------------------------------

// TODO

// WRAPPERS JT <-> ELIM-ORDER ---------------------------------------

/**
 * Wrapper class for the use of elimination order 
 * recombinators/mutators on junction trees (see
 * {@link JTree}).
 * <dl><dt>Notice:</dt>
 * <dd>If we only use the number of fill-ins to rate one tree-
 *     decomposition over another, then it's an unnecessary
 *     overhead to create full junction trees. Nevertheless 
 *     we do always create full junction tree objects for 
 *     generality and simplicity here.
 * </dd></dl>
 * @param elimOrderRecombinator  which works on (elimination) 
 *                               orders of elements (nodes)
 */
function JT2ElimOrderRecombinator (elimOrderRecombinator) 
{ 
  this.initJT2ElimOrderRecombinator(elimOrderRecombinator); 
}

/* Inheritance */
//JT2ElimOrderRecombinator.prototype             = new <ParentClass>();
//JT2ElimOrderRecombinator.prototype.parent      = <ParentClass>.prototype;
JT2ElimOrderRecombinator.prototype.constructor = JT2ElimOrderRecombinator;

/** 
 * (Re-)initialize this JT2ElimOrderRecombWrapper object.
 * (See constructor {@link JT2ElimOrderRecombinator})
 */
JT2ElimOrderRecombinator.prototype.initJT2ElimOrderRecombinator = function (elimOrderRecombinator) 
{
  this.elimOrderRecombinator = elimOrderRecombinator;
};

/** 
 * Recombinates two individuals and returns an offspring.
 * <dl><dt>Notice:</dt>
 * <dd>{@link BayesNet} jt1.net should be the same as jt2.net.
 * </dd></dl>
 * @param jt1  first {@link JTree}
 * @param jt2  second {@link JTree}
 * @returns    offspring {@link JTree}
 */
JT2ElimOrderRecombinator.prototype.conjugate = function (jt1, jt2)
{
  // get elimination-orders (arrays of node IDs)
  var o1 = new Array(jt1.order.length);
  var o2 = new Array(jt2.order.length);
  for (var id in jt1.order) o1[jt1.order[id]] = id; 
  for (var id in jt2.order) o2[jt2.order[id]] = id;
  
  // create offspring elimination order
  var o12 = this.elimOrderRecombinator.conjugate(o1, o2);
  if (!o12) return (null);
  
  // create a new offspring junction tree - with 
  // a fully restricted elimination order ('restr') 
  var restr = {};
  for (var i = 0; i < o12.length; i++) restr[o12[i]] = i;
  var jt = new JTree(jt1.net, restr, jt1.fillin, jt1.max);
  return (jt);
};

/** 
 * Means 'conjugate(a,b)' the same as 'conjugate(b,a)'?
 */
JT2ElimOrderRecombinator.prototype.isSymmetric = function ()
{
  return (this.elimOrderRecombinator.isSymmetric());
};

// OX1 --------------------------------------------------------------

/**
 * OX1 algorithm for recombination of (elimination) 
 * orders of elements (nodes). 
 * <dl><dt>Notice:</dt>
 * <dd>This recombinator cannot be used directly 
 *     on junction trees, use the wrapper class 
 *     {@link JT2ElimOrderRecombinator} for that.
 * </dd></dl>
 * @param mutator  mutator with method 'mutator.mutate()'
 * @param success  probability for an offspring to be created
 */
function OX1 (mutator, success) 
{ 
  this.initOX1(mutator, success); 
}

/* Inheritance */
//OX1.prototype           = new <ParentClass>();
//OX1.prototype.parent    = <ParentClass>.prototype;
OX1.prototype.constructor = OX1;

/** 
 * (Re-)initialize this OX1 object.
 * (See constructor {@link OX1})
 */
OX1.prototype.initOX1 = function (mutator, success) 
{
  this.mutator = mutator;
  this.success = success;
};

/** 
 * Recombinates two individuals and returns an offspring.
 * @param a  first (elimination) order
 * @param b  second (elimination) order
 * @returns  offspring elimination order
 */
OX1.prototype.conjugate = function (a, b)
{
  var c = null;
  if (Math.random() <= this.success) {
    c = a.slice(0, a.length);                     // make offspring 'c' a copy of 'a'
    var n = a.length;
    
    var used = {};                                // the elements used in the offspring 'c'    
    var beg  = Math.floor(Math.random() * n);     // create the starting range border (inclusive)...
    var end  = Math.floor(Math.random() * n);     // ...and the end range border (exclusive)

    var i, j, equal = true;
    for (i = beg; i != end; i = (i+1)%n) {        // go from begin to end (end might be < beg!)...
      var elem = a[i];                            // ...get the element (taken from 'a')...
      var comp = b[i];                            // ...take the element at the same position from...
      equal &= (elem == comp);                    // ...the other parent and compare both elements...
      used[elem] = true;                          // ...and finally mark that element as used
    } 
    
    if (equal) return (null);                     // all equal -> no offspring
    
    for (i = end, j = i; i != beg; i = (i+1)%n) { // go from end to begin...
      var elem = b[j];                            // starting at end (initial j),...  
      while (used[elem]) {                        // ...find the first element in ...
        j = (j+1)%n;                              // ...'b' which is not used...
        elem = b[j];                              // ...by the offspring and...
      }
      c[i] = elem;                                // ...add that element to the offspring...
      used[elem] = true;                          // ...finally mark it as used 
    }
    
    this.mutator.mutate(c);                       // mutation 
  }
  return (c);
};

/** 
 * Means 'conjugate(a,b)' the same as 'conjugate(b,a)'?
 */
OX1.prototype.isSymmetric = function ()
{
  return (true);
};

// OX2 --------------------------------------------------------------

/**
 * OX2 algorithm for recombination of (elimination) 
 * orders of elements (nodes). 
 * <dl><dt>Notice:</dt>
 * <dd>This recombinator cannot be used directly 
 *     on junction trees, use the wrapper class 
 *     {@link JT2ElimOrderRecombinator} for that.
 * </dd></dl>
 * @param mutator  mutator with method 'mutator.mutate()'
 * @param success  probability for an offspring to be created
 */
function OX2 (mutator, success) 
{ 
  this.initOX2(mutator, success); 
}

/* Inheritance */
//OX2.prototype           = new <ParentClass>();
//OX2.prototype.parent    = <ParentClass>.prototype;
OX2.prototype.constructor = OX2;

/** 
 * (Re-)initialize this OX2 object.
 * (See constructor {@link OX2})
 */
OX2.prototype.initOX2 = function (mutator, success) 
{
  this.mutator = mutator;
  this.success = success;
};

/** 
 * Recombinates two individuals and returns an offspring.
 * @param a  first (elimination) order
 * @param b  second (elimination) order
 * @returns  offspring elimination order
 */
OX2.prototype.conjugate = function (a, b)
{
  var c = null;
  if (Math.random() <= this.success) {
    c = b.slice(0, b.length);                     // make offspring 'c' a copy of 'b'
    var n = a.length;

    var elem2index = {};           // maps elements in 'a' to their positions in 'a'
    var del2index  = {};           // maps deleted elements in 'b' to their positions in 'a'!
    var del        = new Array();  // deleted positions in 'b'
    for (var i = 0; i < a.length; a++) { elem2index[a[i]] = i; }
    
    var equal = true;
    for (var i = 0; i < n; i++) {  // go over all elements...
      if (Math.random() > 0.5) {   // ...select positions by tossing a coin...
        var elem = c[i];           // ...get that element (same as in 'b')...
        var j = elem2index[elem];  // ...get the index of this element in 'a'...
        equal &= (i == j);         // ...check the positions to be equal...
        del2index[elem] = j;       // ...save its index in 'a' and...
        del.push(i);               // ...save the index in 'c' (same as in 'b')
      }
    }
    
    if (equal) return (null);      // do not create an offspring if all selected elements are equal

    var add = new Array();         // sorted elements by their indices in 'a'
    for (var i in del2index) { add.push(i); }
    add.sort(function(u,v){ del2index[u] - del2index[v]; });
    for (var i = 0, j = 0; j < del.length; j++) {  // go over all deleted indices
      c[del[j]] = add[i++];        // and re-insert the elements (in the order of 'a')
    }
    
    this.mutator.mutate(c);        // mutation 
  }
  return (c);
};

/** 
 * Means 'conjugate(a,b)' the same as 'conjugate(b,a)'?
 */
OX2.prototype.isSymmetric = function ()
{
  return (true);
};

// POS --------------------------------------------------------------

/**
 * POS algorithm for recombination of (elimination) 
 * orders of elements (nodes). 
 * <dl><dt>Notice:</dt>
 * <dd>This recombinator cannot be used directly 
 *     on junction trees, use the wrapper class 
 *     {@link JT2ElimOrderRecombinator} for that.
 * </dd></dl>
 * @param mutator  mutator with method 'mutator.mutate()'
 * @param success  probability for an offspring to be created
 */
function POS (mutator, success) 
{ 
  this.initPOS(mutator, success); 
}

/* Inheritance */
//POS.prototype           = new <ParentClass>();
//POS.prototype.parent    = <ParentClass>.prototype;
POS.prototype.constructor = POS;

/** 
 * (Re-)initialize this POS object.
 * (See constructor {@link POS})
 */
POS.prototype.initPOS = function (mutator, success) 
{
  this.mutator = mutator;
  this.success = success;
};

/** 
 * Recombinates two individuals and returns an offspring.
 * @param a  first (elimination) order
 * @param b  second (elimination) order
 * @returns  offspring elimination order
 */
POS.prototype.conjugate = function (a, b)
{
  var c = null;
  if (Math.random() <= this.success) {
    c = a.slice(0, a.length);             // make offspring 'c' a copy of 'a'
    var n = a.length;

    var added   = {};                    // nodes added to 'c' (taken from 'b')
    var removed = {};                    // nodes remved from 'c' 
    var skip    = new Array(n);          // marks positions to skip in 'a'

    var equal = true;
    for (var i = 0; i < n; i++) {         // go over all elements...
      if (Math.random() > 0.5) {          // ...select elements by tossing a coin...
        var elem1 = a[i];                 // ...take the elements from 'a' and...
        var elem2 = b[i];                 // ...from 'b' at the current position...
        equal &= elem1 == elem2;          // ...check both elements to be equal anyway...
        c[i] = elem2;                     // ...exchange these elements in 'c'...
        added[elem2] = true;              // ...mark these nodes as added 'c'...
        removed[elem1] = true;            // ...and to a (virtual) second offspring...
        skip[i] = true;                   // ...also mark the position
      }
    }
    
    if (equal) return (null);             // do not create an 'c' if all selected elements are equal

    for (var i = 0, j = 0; i < n; i++) {  // go over all elements...
      if (!skip[i]) {                     // ...but skip the exchanged positions...
        var elem1 = c[i];                 // ...get the element at the current position...
        if (added[elem1]) {               // ...it is doubled because we know it wasn't added at this position (->skip)...
          var elem2 = b[j++];             // ...find the first missing element from other the parent...
          while (!( removed[elem2] &&     // ...i.e. the element must be added to 'c'...
                   !added[elem2]))        // ...and must not be added to 'c'
          {
            elem2 = b[j++];
          }
          c[i] = elem2;                   // ...take that element...
          added[elem2] = true;            // ...and and mark it as added (don't forget)
        }
      }
    }
    
    this.mutator.mutate(c);               // mutation 
  }
  return (c);
};

/** 
 * Means 'conjugate(a,b)' the same as 'conjugate(b,a)'?
 */
POS.prototype.isSymmetric = function ()
{
  return (false);
};

// EOT --------------------------------------------------------------

/**
 * Creates recombinations of decompositions / triangulations
 * (i.e. unlike OX1, OX2 and POS which all work on elimination
 * orders this recombinator works directly on the junction tree)
 * <ul>
 * <li> taking fill-in-links which exists in only one 
 *      parent-triangulation with chance p1 (p1 might be 0)
 * <li> taking all fill-in-links which are part of both 
 *      parent-triangulations with chance p2 (p2 might be 0)
 * <li> find remaining fill-ins for a full triangulation 
 *      by a one-step-lookahead algorithm (not done here, 
 *      to be done by the mutator!)
 * </ul> 
 * While {@link ElimOrderRecombinator} just recombinates
 * elimination orders, this {@link Recombinator} recombinates
 * structures - i.e. it recombinates the fill-ins used in two
 * different triangulations. 
 * 
 * <dl><dt>Notice</dt>
 * <dd>The best setting for the attribute 'takeFrom1' seems to be 0, 
 *     while for 'takeFrom2' the best value seems to be 1 (or at least 
 *     very close to one). They are still included for evaluation reasons.
 * </dd></dl>
 * 
 * @param mutator     the mutator used on the elimination order
 * @param success     the probability that two parents create a child
 * @param takeFrom1   the probability to take a fill-in which is part
 *                    of one of the parent triangulations only
 * @param takeFrom2   the probability to take a fill-in which is part
 *                    of both parent triangulations
 * @param noise       noise level used on the ranks of nodes
 *                    while creating initial solutions (population);
 *                    normally it is best to have noise in the range 
 *                    [0,1[ - i.e. 1 is a good choice...  
 */
function EOT (mutator, success, takeFrom1, takeFrom2, noise) 
{ 
  this.initEOT(mutator, success, takeFrom1, takeFrom2, noise); 
}

/* Inheritance */
//EOT.prototype           = new <ParentClass>();
//EOT.prototype.parent    = <ParentClass>.prototype;
EOT.prototype.constructor = EOT;

/** 
 * (Re-)initialize this EOT object.
 * (See constructor {@link EOT})
 */
EOT.prototype.initEOT = function (mutator, success, takeFrom1, takeFrom2, noise) 
{
  this.mutator    = mutator;
  this.success    = success;
  // since for every fill-in there are two links (i.e. one for 
  // each direction) the probability to take one is given by:
  // pTakeBoth = 1 - pTakeNone = 1 - (1-pTakeAny)^2 => 
  // pTakeAny  = 1 - sqrt(1 - pTakeBoth)
  this.takeFrom1  = 1 - Math.sqrt(1 - takeFrom1);
  this.takeFrom2  = 1 - Math.sqrt(1 - takeFrom2);
  this.noise      = noise; 
};

/** 
 * Recombinates two individuals and returns an offspring.
 * @param a  first junction tree
 * @param b  second junction tree
 * @returns  offspring junction tree
 */
EOT.prototype.conjugate = function (a, b)
{
  var c = null;
  if (Math.random() <= this.success) {
    
    // network of pre-defined fill-ins
    var preNet = new Net("preNet", "predefined fill-ins"); 
    for (var nId in a.net.nodes) {
      preNet.addNode(a.net.nodes[nId]);
    }

    // use all fill-ins which are part of 'a' _and_ 'b' with probability 'takeFrom2'
    // use fill-ins which are part of either 'a' or 'b' with probability 'takeFrom1'
    for (var nId1 in a.filledNet.siblings) {
      for (var nId2 in a.filledNet.siblings[nId1]) {
        if (a.filledNet.siblings[nId1][nId2] == "fillin") {
          if (b.filledNet.siblings[nId1][nId2]) {
            if (Math.random() < this.takeFrom2) {
              preNet.addRelation(a.filledNet.nodes[nId1], a.filledNet.nodes[nId2], "fillin");
            }
          } else {
            if (Math.random() < this.takeFrom1) { 
              preNet.addRelation(a.filledNet.nodes[nId1], a.filledNet.nodes[nId2], "fillin");
            }
          }
        }
      }
    }
    for (var nId1 in b.filledNet.siblings) {
      for (var nId2 in b.filledNet.siblings[nId1]) {
        if (b.filledNet.siblings[nId1][nId2] == "fillin") {
          if (a.filledNet.siblings[nId1][nId2]) {
            // already included with probability = takeFrom2
          } else {
            if (Math.random() < this.takeFrom1) { 
              preNet.addRelation(b.filledNet.nodes[nId1], b.filledNet.nodes[nId2], "fillin");
            }              
          }
        }
      }
    }      

    // now create the new JTree (if the elim-order gets
    // mutated the whole JTree must be re-created again)
    c = new JTree(a.net, {}, a.fillin, a.max, this.noise, preNet);
    var order = new Array(c.order.length);
    for (var id in c.order) order[c.order[id]] = id;              // convert elim-order to a list
    if (this.mutator.mutate(order)) {                             // mutate that order
      var restr = {};
      for (var i = 0; i < order.length; i++) restr[order[i]] = i; // convert order to restriction
      c = new JTree(a.net, restr, a.fillin, a.max, 0);
    } 
  }   
  return (c);
};

/** 
 * Means 'conjugate(a,b)' the same as 'conjugate(b,a)'?
 */
EOT.prototype.isSymmetric = function ()
{
  return (true);
};

// MUTATORS =========================================================
// ISM --------------------------------------------------------------

/**
 * ISM algorithm for mutation of (elimination) 
 * orders of elements (nodes). 
 * <dl><dt>Notice:</dt>
 * <dd>To be used in conjuction with recombinators  
 *     for (elimination) orders of elements (nodes).
 * </dd></dl>
 * @param rate  mutation rate
 */
function ISM (rate) 
{ 
  this.initISM(rate); 
}

/* Inheritance */
//ISM.prototype           = new <ParentClass>();
//ISM.prototype.parent    = <ParentClass>.prototype;
ISM.prototype.constructor = ISM;

/** 
 * (Re-)initialize this ISM object.
 * (See constructor {@link ISM})
 */
ISM.prototype.initISM = function (rate) 
{
  this.rate = rate;
};

/** 
 * Introduces mutations on an individual.
 * @param a  (elimination) order of elements (nodes)
 * @returns  true iff a mutation has been applied
 */
ISM.prototype.mutate = function (a)
{
  if (Math.random() <= this.rate) {
    var i = Math.floor(Math.random() * a.length); 
    var j = Math.floor(Math.random() * a.length); 
    if (i < j) {
      var sub = a.slice(i, j+1);
      sub.unshift(sub.pop());
      a.splice(i, j-i+1, sub);
    } else if (j < i) {
      var sub = a.slice(j+1, i);
      sub.push(sub.shift());
      a.splice(j, i-j+1, sub);
    } else {
      // even though nothing is done in this case we return 'true'
    }
    return (true);
  } else {
    return (false);
  }
};

// EM (ME) ----------------------------------------------------------

/**
 * ME algorithm for mutation of (elimination) 
 * orders of elements (nodes). 
 * <dl><dt>Notice:</dt>
 * <dd>To be used in conjuction with recombinators  
 *     for (elimination) orders of elements (nodes).
 * </dd>
 * <dd>Called ME instead of EM because EM is already
 *     used for expectation maximization.
 * </dd></dl>
 * @param rate  mutation rate
 */
function ME (rate) 
{ 
  this.initME(rate); 
}

/* Inheritance */
//ME.prototype           = new <ParentClass>();
//ME.prototype.parent    = <ParentClass>.prototype;
ME.prototype.constructor = ME;

/** 
 * (Re-)initialize this ME object.
 * (See constructor {@link ME})
 */
ME.prototype.initME = function (rate) 
{
  this.rate = rate;
};

/** 
 * Introduces mutations on an individual.
 * @param a  (elimination) order of elements (nodes)
 * @returns  true iff a mutation has been applied
 */
ME.prototype.mutate = function (a)
{
  if (Math.random() <= this.rate) {
    var i = Math.floor(Math.random() * a.length); 
    var j = Math.floor(Math.random() * a.length); 
    var x = individual[i];
    individual[i] = individual[j];
    individual[j] = x;
    return (true);
  } else {
    return (false);
  }
};
