/*
 * -------------------------------------------------------------------------
 *
 *  pn4webDynamic.js Copyright (C) 2011, 
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *   Author: Patrick Rammelt 
 *   E-Mail: patrick.rammelt@go4more.de
 *   Site:   http://www.patricks-seite.de
 * -------------------------------------------------------------------------
 *
 * Continuous distributions, continuous dimensions, continuous nodes for
 * hybrid bayesian networks (these types are fully independent of display 
 * functionalities).
 */


// CONTINUOUS CONDITIONAL DISTRIBUTIONS =============================
// CONDITIONAL GAUSSIAN ---------------------------------------------

/**
 * Constructor of a conditional gaussian distribution
 * @param head     id of the head dimension
 * @param factors  'factors[dim.id]' = linear factor of 
 *                 dependency on tail-dimension 'dim'
 * @param mean     mean value (offset)
 * @param std2     variance (std^2)
 * @param weight   weight
 * @param dims     dims[dimId] = name (for toString)
 */
function CG (head, factors, mean, std2, weight)
{
  if (mean != undefined) {
    this.initCG(head, factors, mean, std2, weight);
  }
}

/* Inheritance */
// CG.prototype          = ;
// CG.prototype.parent   = ;
CG.prototype.constructor = CG; 

/**
 * Initialize this conditional gaussian 
 * (see constructor {@link CG}).
 */
CG.prototype.initCG = function (head, factors, mean, std2, weight)
{
  this.head = head;
  this.fact = factors;
  this.mean = mean;
  this.std2 = std2;
  this.wght = weight;
};

/**
 * Get the factor for the dimension with the given id
 * @returns  linear dependency factor (0 if there is none)
 */
CG.prototype.factor = function (id)
{
  return (this.fact[id] ? this.fact[id] : 0.0);
};

/**
 * Create an exact copy of this {@link CG}.
 */
CG.prototype.clone = function ()
{
  var factors = {};
  for (var t in this.fact) factors[t] = this.fact[t];
  return (new this.constructor(this.head, factors, this.mean, this.std2, this.wght));
};

/**
 * Get the weighted value of this conditional distribution.
 * @param x       input value
 * @param sample  a sample containing a value for at least all
 *                tail dimension ids of this CG (sample[t] = value)
 *                [optional for unconditional distributions]
 * @returns       weighted function value f(x|sample)
 */
CG.prototype.value = function (x, sample)
{
  var m = this.mean;
  for (var t in this.fact) m += this.fact[t] * sample[t];
  if (this.std2 == 0.0) {
    return ((x == this.mean ? Number.POSITIVE_INFINITY : 0.0));
  } else {
    return (this.wght * Math.exp(-Math.pow(x - m, 2) / (2.0 * this.std2)) / 
	    Math.sqrt(2.0 * Math.PI * this.std2));
  }
};

/**
 * Sample a value out of this conditional distribution.<br>
 * Notice: We are approximating a normal distributed value
 *         by taking the sum of three random numbers.
 * @param sample  a sample containing a value for at least all
 *                tail dimension ids of this CG (sample[t] = value)
 *                [optional for unconditional distributions]
 * @returns       sampled value (for the 'head' dimension)
 */
CG.prototype.sample = function (sample)
{
  var m = this.mean;
  for (var t in this.fact) m += this.fact[t] * sample[t];
  return (((Math.random() * 2 - 1) + 
	   (Math.random() * 2 - 1) +
	   (Math.random() * 2 - 1)) * Math.sqrt(this.std2) + m);
};

/**
 * Exchanges the heads in this and the given other 
 * distribution.<br>
 * <dl>
 * <dt>Pre:</dt><dd>
 *    cg   = Y|Wj...Wn    (does not depend on Z)<br>
 *    this = Z|Y,Wi...Wm  (depends on Y)</dd>
 * <dt>Post:</dt><dd>
 *    cg   = Y|Z,Wi...Wn  (depends on Z now)<br>
 *    this = Z|Wi...Wn    (does not depend on Y anymore)</dd>
 * <dt>Schematically:</dt><dd>
 *    W -&gt; Y -&gt; Z &lt;- W becomes<br>
 *    W -&gt; Y &lt;- Z &lt;- W</dd>
 * </dl>
 * The head of the given other CG (Y) has to be part 
 * of the tail of this CG!
 * @param cg         other distribution to exchange its 
 *                   head dimension with this CG
 * @param keepThis   do not change this CG?
 * @param keepOther  do not change the other given CG?
 */
CG.prototype.exchange = function (cg, keepThis, keepOther)
{        
  // get all tail dimension ids (which are not head dimensions)
  var tails = {};
  for (var t in cg.fact)   tails[t] = true;
  for (var t in this.fact) tails[t] = true;
  delete (tails[cg.head]);
  delete (tails[this.head]);

  // tail factor for Y (head of other CG(Y|...)) in this CG(Z|...)
  var b = this.factor(cg.head);
    
  // create new mean, variance and factors for this CG(Z|...)
  var std2 = this.std2 + (b * b * cg.std2), mean, fact;
  if (!keepThis) {
    mean = this.mean + (b   * cg.mean);
    fact = {};
    for (var t in tails) {
      fact[t] = this.factor(t) + (b * cg.factor(t)); // a + (b * c)
    }    
  } else {
    mean = fact = undefined;
  }

  // create and set new var, factors and dimensions for other CG(Y|...)
  if (!keepOther) {
    if ((cg.std2 > 0.0) && (this.std2 > 0.0)) {  
      var cgFact = {};                          // new factors for other CG(Y|...)
      for (var t in tails) {
        cgFact[t] = ((  cg.factor(t) * this.std2    ) - 
                     (this.factor(t) *   cg.std2 * b)) / std2;
      }
      cgFact[this.head] = (b * cg.std2) / std2; // head of this CG(Z|...) becomes a tail of other CG(Y|...)
      cg.fact = cgFact;
      cg.mean = ((cg.mean * this.std2) - (this.mean * b * cg.std2)) / std2;
      cg.std2 = (cg.std2 * this.std2) / std2;
    } else if (cg.std2 > 0) {
      var cgFact = {};                          // new factors for other CG(Y|...)
      for (var t in tails) {
        cgFact[t] = -this.factor(t) / b;
      }            
      cgFact[this.head] =  1.0 / b;             // head of this CG(Z|...) becomes a tail of other CG(Y|...)
      cg.fact = cgFact;
      cg.mean = this.mean / b;
      cg.std2 = 0;
    } else {
      cg.std2 = 0;
    }
  }
    
  // set var, factors and dimensions for this CG(Z|...)
  if (!keepThis) {
    this.mean = mean;
    this.std2 = std2;
    this.fact = fact;
  }    
};

/**
 * If the head dimesnion is observed this CG degenerates
 * to gaussian with mean = value and variance = 0.
 * @param v  value to be set for the head dimension (observation)
 */
CG.prototype.setHead = function (v) 
{
  this.mean = v;
  this.std2 = 0;
  this.fact = {};
};

/**
 * If a tail dimension is observed this tail dimension is
 * removed and the value times the conditional factor is
 * added to the mean. The variance remains the same.
 * @param t  tail dimension id to set the given value for
 * @param v  value to be set (observation)
 */
CG.prototype.setTail = function (t, v)
{
  this.mean += this.factor(t) * v;
  delete (this.fact[t]);
};

/**
 * Create a human readable string representation.
 */
CG.prototype.toString = function ()
{
  var str = 'gauss(mean = ';
  for (var t in this.fact) str += t + ' * ' + this.fact[t].toFixed(2) + ' + ';
  str += this.mean.toFixed(2) + ', var = ' + this.std2.toFixed(2) + ')';
  return (str);
};

/**
 * Create a human readable string representation.
 */
CG.prototype.toHTML = function ()
{
  var str = '&Nu;(&mu; = ';
  for (var t in this.fact) str += t + ' &sdot; ' + this.fact[t].toFixed(2) + ' + ';
  str += this.mean.toFixed(2) + ', &sigma; = ' + Math.sqrt(this.std2).toFixed(2) + ')';
  return (str);
};


// CONTINUOUS DIMENSIONS/NODES ======================================
// CONTINUOUS DIMENSIONS --------------------------------------------

/**
 * Constructor for a continuous dimension.
 * @param id    id
 * @param name  name
 * @param min   min. value of the interval
 * @param max   max. value of the interval
 */
function ContDim (id, name, min, max) 
{
  if (id != undefined) {
    this.initContDim(id, name, states);
  }
}

/* Inheritance */
// ContDim.prototype          = ;
// ContDim.prototype.parent   = ;
ContDim.prototype.constructor = ContDim; 

/**
 * Initialize this continuous dimension 
 * (see constructor {@link ContDim}).
 */
ContDim.prototype.initContDim = function (id, name, min, max)
{
  this.isContinuous = true;
  this.id           = id;     // id
  this.name         = name;   // name
  this.min          = min;    // min. value
  this.max          = max;    // max. value
};

// CONTINUOUS NODE IN A BAYESIAN NETWORK ----------------------------

/**
 * Constructor for a continuous conditional gaussian node in 
 * a hybrid bayesian network (which basically is a continuous 
 * dimension and therfor inherits from ContDim).
 * @param id:   id
 * @param name  name
 * @param min   min. value of the interval
 * @param max   max. value of the interval
 * @param x     x-coordinate
 * @param y     y-coordinate
 */
function CGNode (id, name, min, max, x, y)
{
  if (id != undefined) {
    this.initCGNode(id, name, min, max, x, y);
  }
}

/* Inheritance */
CGNode.prototype             = new ContDim();
CGNode.prototype.parent      = ContDim.prototype;
CGNode.prototype.constructor = CGNode;

/**
 * Initialize this (normal, diskrete) node of a 
 * bayesian network (see constructor {@link CGNode}).
 */
CGNode.prototype.initCGNode = function (id, name, min, max, x, y)
{
  this.initContDim(id, name, min, max);

  // coordinates
  this.x = x;
  this.y = y;

  // conditional parameter table for conditional gaussians (to be set later)
  this.cgt = null;   

  // init value (evidence; null mean not observed)
  this.v = null; 
};

/** 
 * Remove evidence from this node 
 */
CGNode.prototype.rmEvidence = function ()
{
  this.v = null;
};

/**
 * Create a table containing one 'CG' object for each
 * state-configuration of all the parents of this node.
 * @returns  a {@link PTable} of {@link CG} objects 
 *           (i.e. not just this.cgt!)
 */
CGNode.prototype.getCGT = function ()
{
  var cgt  = new PTable(this.cgt.dims.slice(1));  // all but the first dimension
  var step = this.cgt.sizes[this.cgt.dims[0].id];
  for (var i = 0, j = 0; i < this.cgt.n; i += step, j++) {
    var mean = this.cgt.values[i];
    var std2 = this.cgt.values[i+1];
    var fact = {};
    for (var s = 2; s < step; s++) {
      var dim = this.cgt.dims[0].states[s];       // (this state is a dimension)
      fact[dim.id] = this.cgt.values[i+s];
    }
    cgt.values[j] = new CG(this.id, fact, mean, std2, 1.0);
  }
  return (cgt);
};

/**
 * Recreate the CGT e.g. after the number of parents has
 * changed. Rescues as much information as possible.
 * @param meAndParents  array containg this node (first dimension) 
 *                      and all parent nodes (further dimensions)
 *                      (could be retreived by 'BayesNet.getParents()')
 */
CGNode.prototype.adaptCPT = function (meAndParents)
{
  var cgt = this.cgt;        // current parameter table

  // create dimensions and states for new parameter 
  // table for conditional continuous distributions
  var dims   = new Array();  // dimensions of the CGT
  var states = new Array();  // states of the CGT
  dims.push(new Dim(cgt.dims[0].id, this.name, states)); // re-use dimension Id
  states.push(cgt.dims[0].states[0]);                    // re-use states for...
  states.push(cgt.dims[0].states[1]);                    // ...mean and variance
  for (var d = 1; d < meAndParents.length; d++) {
    var dim = meAndParents[d];
    if (dim.states) {        // dim is discrete
      dims.push(dim);        // (using at as a normal dimension)
    } else {                 // dim is continuous
      states.push(dim);      // (using cont. dimension as a state)
    }
  }

  // sort states as they were in the current CGT
  // (this ensures that a new cont. dim. appears last)
  order = {};
  for (var s in cgt.dims[0].states) {
    order[cgt.dims[0].states[s].id] = s;
  }
  var comparator = function (a, b) { 
    return((order[a.id] ? order[a.id] : 1000) - 
	   (order[b.id] ? order[b.id] : 1000)); 
  };
  states.sort(comparator);

  // create new parameter table and rescue as
  // much information as possible from current CGT
  this.cgt = new PTable(dims, 0);
  cgt.cut(this.cgt);
  this.cgt.cut(cgt);
  if (this.cgt.dims.length <= cgt.dims.length) {
    this.cgt.marg(cgt);
  } else {
    this.cgt.infl(cgt);
  }
  this.cgt.uncut();
};
