/*
 * -------------------------------------------------------------------------
 *
 *  pn4webSampling.js Copyright (C) 2011, 
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *   Author: Patrick Rammelt 
 *   E-Mail: patrick.rammelt@go4more.de
 *   Site:   http://www.patricks-seite.de
 * -------------------------------------------------------------------------
 *
 * Common helpers for advanced (non-basic) propagation and learning
 * algorithms.
 */

// MATH =============================================================

/**
 * Lanczos approximation of the gamma function as described in 
 * "Numerical Recipes in C" (2nd edition Cambridge University 
 * Press, 1992).<br>
 * <dl><dt>Notice:</dl>
 * <dd>There are more precise implementations of gamma 
 *     and lgamma which are by far more complicated.</dd></dl>
 * @param v  value 
 * @returns  gamma(v), Infinity for v > 141.6
 */
function gamma (v) 
{
  var d1 = Math.sqrt(2 * Math.PI) / (v);
  var d2 = ((1.000000000190015           ) + 
            (76.18009172947146    / (v+1)) + 
            (-86.50532032941677   / (v+2)) + 
            (24.01409824083091    / (v+3)) + 
            (-1.231739572450155   / (v+4)) + 
            (1.208650973866179E-3 / (v+5)) + 
            (-5.395239384953E-6   / (v+6)));  
  var d3 = Math.pow((v + 5.5), (v + 0.5));
  var d4 = Math.exp(-(v + 5.5));
  return (d1 * d2 * d3 * d4);
}

/**
 * Log-gamma function using lanczos approximation of the gamma 
 * function as described in "Numerical Recipes in C" (2nd edition 
 * Cambridge University Press, 1992).<br>
 * <dl><dt>Notice:</dl>
 * <dd>There are more precise implementations of gamma 
 *     and lgamma which are by far more complicated.</dd></dl>
 * @param v  value
 * @returns  log(gamma(v)), works fine even for big v
 */
function lgamma (v) 
{ 
  var d1 = Math.log(2 * Math.PI) * 0.5 - Math.log(v);  // log(sqrt(x))/y = log(x^0.5)-log(y) = log(x)/2 - log(y)
  var d2 = Math.log((1.000000000190015           ) + 
                    (76.18009172947146    / (v+1)) + 
                    (-86.50532032941677   / (v+2)) + 
                    (24.01409824083091    / (v+3)) + 
                    (-1.231739572450155   / (v+4)) + 
                    (1.208650973866179E-3 / (v+5)) + 
                    (-5.395239384953E-6   / (v+6)));  
  var d3 = Math.log(v + 5.5) * (v + 0.5);              // log(x^y)    = log(x) * y
  var d4 = -(v + 5.5);                                 // log(exp(x)) = x
  return (d1 + d2 + d3 + d4);                          // log(x * y)  = log(x) + log(y)
}


// NATURAL ORDERING OF NODES ========================================

/**
 * Constructor for a natural ordering of a network, where a 
 * parent is guaranteed to have a lower 'level' than all its 
 * child-nodes. By a 'NaturalOrdering' an ordered array of 
 * nodes could be created, which is used by some sampling
 * algorithms ('getOrderedNodes()').<br> 
 * A 'NaturalOrdering' could also be used to determine whether 
 * or not a new link could be added without introducing cycles 
 * in a bayesian network: If the level of the new child-node is 
 * greater or equal to the level of its new parent node (i.e. 
 * 'level[child.id]' >= 'level[parent.id]', then it is always 
 * safe to add that link (parent->child).<br> 
 * Otherwise a more complicated check has to be performed. Use 
 * '!isAncestor(child, parent)' for that. 
 * After a new link has been added the 'NaturalOrdering' could 
 * be adapted by calling 'connect(parent,child)'.<br>
 * See also {@link Clustering} for another check that could be 
 * performed.<br>
 * <dl><dt>Notice:</dl>
 * <dd>The outcome for networks containing directed cycles 
 *     is undefined, although even in a cyclic network every 
 *     node gets a level associated.</dd></dl>
 * @param net  network to find the natural ordering for
 */
function NaturalOrdering (net)
{
  if (net != undefined) {
    this.initNaturalOrdering(net);
  }
}

/* Inheritance */
// NaturalOrdering.prototype          = ;
// NaturalOrdering.prototype.parent   = ;
NaturalOrdering.prototype.constructor = NaturalOrdering;

/**
 * Re-creates this natural ordering of a network 
 * (see constructor {@link NaturalOrdering}).
 */
NaturalOrdering.prototype.initNaturalOrdering = function (net)
{
  this.net   = net;  // network
  this.level = {};   // map: level[nodeId] = max. number of ancestors

  for (var nId in net.nodes) {
    this.setOrder(net.nodes[nId]);
  }
};

/**
 * Recursively find the natural ordering of the given
 * node and all ancestor nodes. If the given node is
 * already included in this ordering, nothing is done.
 * @param node  node to start with
 */
NaturalOrdering.prototype.setOrder = function (node)
{
  var nId = node.id;
  if (this.level[nId] == undefined) {
    this.level[nId] = 0;
    for (var paId in this.net.parents[nId]) {
      this.setOrder(this.net.nodes[paId]);
      this.level[nId] = Math.max(this.level[nId], 
				 this.level[paId] + 1);
    }
  } 
};

/**
 * Get an array of nodes ordered such that 
 * parents appear before their children.
 * @returns  array of nodes in natural ordering
 */
NaturalOrdering.prototype.getOrderedNodes = function ()
{
  var nodes = new Array();
  for (var nId in this.net.nodes) {
    nodes.push(this.net.nodes[nId]);
  }
  var me = this;
  var comparator = function (node1, node2) { 
    return (me.level[node1.id] - me.level[node2.id]); 
  };
  nodes.sort(comparator);
  return (nodes);
};

/**
 * Is the first given node an ancestor of the second one?
 * @param ancestor  suspected ancestor node of 'node'
 * @param node      node (suspected descendant node of 'ancestor')
 * @returns         true iff 'ancestor'=='node' or if 'ancestor' 
 *                  really is an ancestor of 'node'; false means
 *                  it is safe to add a new link 'node'->'ancestor'
 *                  (because 'ancestor' isn't really an ancestor)
 */
NaturalOrdering.prototype.isAncestor = function(ancestor, node)
{
  if (ancestor == node) {
    return (true);
  } else if (this.level[ancestor.id] >= this.level[node.id]) {
    return (false);
  } else {
    for (var paId in this.net.parents[node.id]) {
      var parent = this.net.nodes[paId];
      if (this.isAncestor(ancestor, parent)) return (true);
    }
    return (false);
  }
};

/**
 * Change levels as if the given first node has 
 * become a parent of the given second node.<br>
 * Notice: Levels for some nodes might become 
 *         higher (never smaller) using this.
 * @param parent  first node
 * @param child   child node
 */
NaturalOrdering.prototype.connect = function(parent, child)
{
  if (this.level[child.id] < this.level[parent.id] + 1) {
    this.level[child.id] = this.level[parent.id] + 1;
    for (var chId in this.net.children[child.id]) {
      this.connect(child, this.net.nodes[chId]);
    }
  }
};

// CLUSTERS OF NODES ================================================

/**
 * Constructor for a clustering of a network. Together with 
 * 'NaturalOrdering' a 'Clustering' could be used to determine
 * whether or not an additional link would violate the acyclic
 * structure of a bayesian network. Nodes in different clusters
 * (i.e. 'cluster[node1.id]' != 'cluster[node2.id]') could always
 * be linked by an additional link. Use 'isConnected(node1,node2)'
 * for that check. A 'Clustering' could be adapted by calling 
 * 'connect(node1,node2)' after a new link has been added.
 * @param net  network ({@link Net}) to find the clusters of
 */
function Clustering (net) 
{
  if (net != undefined) {
    this.initClustering(net);
  }
}

/* Inheritance */
// Clustering.prototype          = ;
// Clustering.prototype.parent   = ;
Clustering.prototype.constructor = Clustering;

/**
 * Re-initialize this clustering 
 * (see constructor {@link Clustering}). 
 */
Clustering.prototype.initClustering = function (net)
{
  this.net      = net;
  this.clusters = new Array(); // array of arrays of nodes per cluster
  this.cluster  = {};          // cluster[nodeId] = cluster index 

  for (var nId in net.nodes) {
    if (this.cluster[nId] == undefined) {
      this.clusters.push(new Array());
      this.setCluster(net.nodes[nId]);
    }
  }
};

/**
 * Recursively find the clustering of the given node and
 * all ancestor/descendant nodes. Nothing is done if the
 * given node is already included in this clustering.
 * @param node  node to start with
 */
Clustering.prototype.setCluster = function (node)
{
  var nId = node.id;
  if (this.cluster[nId] == undefined) {
    var c  = this.clusters.length - 1; // index of the current cluster
    this.clusters[c].push(node);
    this.cluster[nId] = c;
    for (var paId in this.net.parents[nId]) {
      this.setCluster(this.net.nodes[paId]);
    }
    for (var chId in this.net.children[nId]) {
      this.setCluster(this.net.nodes[chId]);
    }
  }
};

/**
 * Are the two given nodes in the same cluster?
 * @param node1  first node
 * @param node2  second node
 * @returns      true if both nodes are in same cluster;
 *               false if they are in disconnected parts
 *               of the network ('this.net')
 */
Clustering.prototype.isConnected = function (node1, node2)
{
  return (this.cluster[node1.id] == this.cluster[node2.id]);
};

/**
 * Unites the clusters of the two given nodes.<br>
 * Notice: 'this.clusters' might contain empty 
 *         arrays afterwards (which are not
 *         referenced by any node though).
 * @param node1  first node
 * @param node2  second node
 */
Clustering.prototype.connect = function (node1, node2)
{
  var c1 = this.cluster[node1.id];
  var c2 = this.cluster[node2.id];
  if (c1 != c2) {
    for (var i = this.clusters[c2].length - 1; i >= 0; i--) {
      var node = this.clusters[c2].pop();
      this.cluster[node.id] = c1;
      this.clusters[c1].push(node);
    }
  }
};


// DATASET OF SAMPLES ===============================================

/**
 * Constructor for a dataset of samples. Created by sampling 
 * algorithms, but also used by learning algorithms.
 * TODO: Should we use a two-dimensional 'PTable' to store samples, 
 *       where the states of the first dimension are the 'dims' and 
 *       the states of the second dimension are the sample numbers 
 *       (1...N). The main disadvantage would be that in a 'PTable' 
 *       rows couldn't be dynamically added or removed (which isn't 
 *       necessary so far).
 * @param dims  array of dimensions ("columns" in the dataset)
 * @param n     number of samples ("rows" in the dataset)
 */
function Samples (dims, n)
{
  this.initSamples(dims, n);
}

/* Inheritance */
// Samples.prototype          = ;
// Samples.prototype.parent   = ;
Samples.prototype.constructor = Samples;

/**
 * Re-initialize this dataset of samples 
 * (see constructor {@link Samples}).
 */
Samples.prototype.initSamples = function (dims, n)
{
  this.dims = dims;          // array of dimensions (order of colums)
  this.data = new Array(n);  // array of n arrays of size dims.length
  this.cols = {};            // data[r][cols[dimId]] = stateOfThatDim
  for (var r = 0; r < n; r++) {
    this.data[r] = new Array(dims.length);
    for (var c = 0; c < dims.length; c++) {
      this.data[r][c] = 0;
    }
    this.data[r].weight = 0;
  }
  for (var d in dims) {
    this.cols[dims[d].id] = d;
  }
};

/**
 * Forget each observation for single nodes in each 
 * row of this dataset with the given probability.
 * @param p  probability to forget the state of a node
 */
Samples.prototype.forget = function (p)
{
  for (var r = 0; r < this.data.length; r++) {
    var row = this.data[r];
    for (var c = 0; c < row.length; c++) {
      if (Math.random() < p) {
	row[c] = -1;
      }
    }
  }
};


// CLIQUES AND SEPARATORS ===========================================
// CLIQUES ----------------------------------------------------------

/**
 * Constructs a clique for a junction tree ({@link JTree}) or a loopy
 * belief propagator ({@link LoopyBelief}). For simplicity 'Cliques'
 * are also used by {@link GibbsSampler}.
 * @param id     id
 * @param name   name
 * @param nodes  mapping node id to node (for all contained nodes)
 * @param order  elimination order [optional]
 */
function Clique (id, name, nodes, order)
{
  if (id != undefined) {
    this.initClique(id, name, nodes, order);
  }
}

/* Inheritance */
// Clique.prototype          = ;
// Clique.prototype.parent   = ;
Clique.prototype.constructor = Clique;

/**
 * Initialize this clique 
 * (see constructor {@link Clique}).
 */
Clique.prototype.initClique = function (id, name, nodes, order)
{
  this.id     = id;    // every node in a net needs an unique id
  this.name   = name;  // name (just for visualization)
  this.nodes  = nodes;
  this.order  = order;
  this.addins = {};    // map: node-Id to node for nodes added afterwards (e.g. by y-propagation)
  this.x      = 0;     // x-coordinate (for 'JTrees' a layouter is needed)
  this.y      = 0;     // y-coordinate (for 'JTrees' a layouter is needed)
  this.createJPT();    // create joint probability table
};

/**
 * (Re-)Create the joint probability table for 
 * this clique ('this.jpt'), w.r.t. 'this.nodes'.
 */
Clique.prototype.createJPT = function () 
{
  var dims = new Array();
  for (var nId in this.nodes) {
    dims[dims.length] = this.nodes[nId];
  }
  this.jpt = new Float32Table(dims);  // joint probability table
};

/**
 * Get the number of nodes contained in this clique.
 */
Clique.prototype.size = function () 
{
  return (this.jpt.dims.length);
};

/**
 * Does this clique contain all nodes of the other given clique? 
 */
Clique.prototype.containsAll = function (other) 
{
  for (var id in other.nodes) {
    if (!this.nodes[id]) return (false);
  }
  return (true);
};

// SEPARATORS -------------------------------------------------------

/**
 * Constructs a separator to be added to the link between 
 * the two given 'Cliques'. The separator will contain all 
 * nodes which exist in both given cliques. Separators are
 * used by junction trees ({@link JTree}) and also by loopy 
 * belief propagators ({@link LoopyBelief}).
 * @param from  source clique
 * @param to    target clique
 */
function Separator (from, to)
{
  if (from != undefined) {
    this.initSeparator(from, to);
  }
}

/* Inheritance */
// Separator.prototype          = ;
// Separator.prototype.parent   = ;
Separator.prototype.constructor = Separator;

/**
 * Initialize this separator 
 * (see constructor {@link Separator}). 
 */
Separator.prototype.initSeparator = function (from, to)
{
  this.id    = from.id + '__' + to.id;       // (makes it usable as it was a node)
  this.name  = from.name + ' -> ' + to.name; // (makes it usable as it was a node)
  this.nodes = {};                           // map: node-id to node for every node in this separator
  var dims = new Array();
  for (var nId in from.nodes) {
    var node = to.nodes[nId];
    if (node) {
      this.nodes[nId]   = node;
      dims[dims.length] = node;
    }
  }
  this.jpt = new Float32Table(dims); // joint probability table (init)
};


// CLONING NETWORKS =================================================

/**
 * Creates an identical clone of this bayesian network. 
 * Nodes, states, tables, etc. will be cloned as well.
 * This is especially useful for learning algorithms,
 * where the original network should stay unchanged.
 * Currently only normal (discrete, static) bayesian
 * networks are supported.<br>
 * Notice: Extends {@link BayesNet}.
 */
BayesNet.prototype.clone = function ()
{
  // network
  var net = new this.constructor(this.id, this.name);

  // clone nodes (except CPTs / CGTs)
  for (var nId in this.nodes) {
    var node = this.nodes[nId];
    net.addNode(node.clone());
  }

  // clone directed links
  for (var paId in this.children) {          // from this network
    var parent = net.nodes[paId];            // from cloned network
    for (var chId in this.children[paId]) {  // from this network
      var child = net.nodes[chId];           // from cloned network
      net.addLink(parent, child);
    }
  }

  // clone undirected relations
  for (var n1Id in this.siblings) {          // from this network
    var node1 = net.nodes[n1Id];             // from cloned network
    for (var n2Id in this.siblings[n1Id]) {  // from this network
      var node2 = net.nodes[n2Id];           // from cloned network
      if (n1Id < n2Id) {
        net.addRelation(node1, node2);
      }
    }
  }

  // clone CPTs / CGTs
  for (var nId in this.nodes) {
    var node = this.nodes[nId];              // from this network
    var clone = net.nodes[nId];              // from cloned network
    clone.cloneConditionals(net, node);
  }  

  return (net);
};

/**
 * Creates an identical clone of this normal discrete 
 * {@link Node}, except the 'cpt' which is not cloned 
 * because it involves other nodes/dimensions, which 
 * might have to be cloned first.<br>
 * Notice: Extends {@link Node}
 */
Node.prototype.clone = function ()
{
  var states = new Array();
  for (var s in this.states) {
    var state = this.states[s];
    var clone = new state.constructor(state.id, state.name);
    states.push(clone);
  }      

  var clone = new this.constructor(this.id, this.name, states, this.x, this.y);
  clone.p.values = this.p.values.slice();  // it's ok to used 'Array.slice()' here
  clone.f.values = this.f.values.slice();  // because the value-order is the same
  return (clone);
};

/**
 * Clone conditional parameters (CPT) for this 
 * node from another node.
 * @param net   network of this node
 * @param node  other node (in another, yet identical network)
 */
Node.prototype.cloneConditionals = function (net, node)
{
  this.cpt = new Float32Table(net.getParents(this, true));
  this.cpt.marg(node.cpt);                 // parents might have a different order(!)
};

