/*
 * -------------------------------------------------------------------------
 *
 *  pn4webBasic.js Copyright (C) 2011, 
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *   Author: Patrick Rammelt 
 *   E-Mail: patrick.rammelt@go4more.de
 *   Site:   http://www.patricks-seite.de
 * -------------------------------------------------------------------------
 *
 * Basic types, containing all logic needed for bayesian networks - 
 * including classes for general and bayesian networks, states, discrete 
 * dimensions and nodes, multi-dimensional probability tables and a very 
 * simple propagator called "full joint propagator", which is usable for
 * very small networks only (a more complex propagator, called "junction 
 * tree propagator", could be found in "pn4webJTree.js").
 * (these types are fully independent of display functionalities, except
 * that nodes have x- and y-coordinates)
 */


// GRAPHS ===========================================================
// GENERAL GRAPHS ---------------------------------------------------

/** 
 * Constructor for a general graph
 * @param id   id
 * @param name name
 */
function Net (id, name)
{
  if (id != undefined) {
    this.initNet(id, name);
  }
}

/* Inheritance */
// Net.prototype          = ;
// Net.prototype.parent   = ;
Net.prototype.constructor = Net;

/**
 * Re-initialization of this 'Net' 
 * (see constructor {@link Net})
 */
Net.prototype.initNet = function (id, name)
{
  // net attributes
  this.id   = id;       // id
  this.name = name;     // name

  // nodes
  this.nodes = {};      // map: nodes.nodeId = node

  // directed links
  this.parents  = {};   // map: parents.toId.fromId  = true (or some link data)
  this.children = {};   // map: children.fromId.toId = true (or some link data)

  // undirected links
  this.siblings = {};   // map: siblings.fromId.toId = true (or some link data / and vice versa)
};

/** 
 * Get a node by its id
 */
Net.prototype.getNode = function (id) 
{ 
  return (this.nodes[id]);
};

/**
 * Get an array of all nodes
 */
Net.prototype.getNodes = function ()
{
  var nodes = new Array();
  for (var nId in this.nodes) {
    nodes.push(this.nodes[nId]);
  }
  return (nodes);
};

/** 
 * Adds a new node to this {@link Net} 
 */ 
Net.prototype.addNode = function (node) 
{
  this.nodes[node.id]    = node;
  this.parents[node.id]  = {};
  this.children[node.id] = {};
  this.siblings[node.id] = {};
};

/**
 * Removes an existing node from this {@link Net} 
 */
Net.prototype.rmNode = function (node) 
{
  // directed links
  for (var paId in this.parents[node.id]) {
    delete (this.children[paId][node.id]);
  }
  delete (this.parents[node.id]);
  for (var chId in this.children[node.id]) {
    delete (this.parents[chId][node.id]);
  }
  delete (this.children[node.id]);

  // undirected links
  for (var siId in this.siblings[node.id]) {
    delete (this.siblings[siId][node.id]);
  }
  delete (this.siblings[node.id]);
  
  // delete node
  delete (this.nodes[node.id]);
};

/**
 * Adds a directed (causal) link between two nodes.
 * @param from  starting node of the new link
 * @param to    target node of the new link
 * @param data  link data associated with the new link
 */
Net.prototype.addLink = function (from, to, data) 
{
  if (data == undefined) data = true;
  this.parents[to.id][from.id]  = data;
  this.children[from.id][to.id] = data;
};

/**
 * Removes a directed (causal) link between two nodes.
 * @param from  starting node of the link to be removed
 * @param to    target node of the link to be removed
 */
Net.prototype.rmLink = function (from, to) 
{
  delete (this.parents[to.id][from.id]);
  delete (this.children[from.id][to.id]);
};

/**
 * Remove all directed links.
 */
Net.prototype.rmLinks = function () 
{
  for (var id in this.nodes) {
    this.parents[id]  = {};
    this.children[id] = {};
  }
};

/**
 * Adds an undirected link between two nodes 
 * @param from  one node of the new link
 * @param to    other node of the new link
 * @param data  link data associated with the new link
 */
Net.prototype.addRelation = function (from, to, data) 
{
  if (data == undefined) data = true;
  this.siblings[to.id][from.id] = data;
  this.siblings[from.id][to.id] = data;
};

/**
 * Removes an undirected link between two nodes 
 * @param from  one node of the link to be removed
 * @param to    other node of the link to be removed
 */
Net.prototype.rmRelation = function (from, to) 
{
  delete (this.siblings[to.id][from.id]);
  delete (this.siblings[from.id][to.id]);
};

/**
 * Remove all undirected links. 
 */
Net.prototype.rmRelations = function () 
{
  for (var id in this.nodes) {
    this.siblings[id] = {};
  }
};

// BAYESIAN NETWORK -------------------------------------------------

/**
 * Constructor for a bayesian network.
 * @param id    id
 * @param name  name
 */
function BayesNet (id, name)
{
  if (id != undefined) {
    this.initBayesNet(id, name);
  }
}

/* Inheritance */
BayesNet.prototype             = new Net();
BayesNet.prototype.parent      = Net.prototype;
BayesNet.prototype.constructor = BayesNet;

/**
 * Re-Initialize this {@link BayesNet} 
 * (see constructor {@link BayesNet}).
 */
BayesNet.prototype.initBayesNet = function (id, name) 
{
  this.initNet(id, name);
  this.prop = null;       // propagator (to be set later)
};

/**
 * Get all parent nodes (dimensions) of a node
 * @param node      the node to get all parents nodes of
 * @param inclSelf  include the node itself as the first node
 * @returns         array of all parent nodes (if inclSelf is true
 *                  the given node is the first node in that array)
 */
BayesNet.prototype.getParents = function (node, inclSelf)
{
  var parents = new Array();
  if (inclSelf) parents.push(node);          // (first node)
  for (var paId in this.parents[node.id]) {
    parents.push(this.nodes[paId]);
  }
  return (parents);
};

/** 
 * Get all child nodes (dimensions) of a node.
 * @param node      the node to get all children nodes of
 * @param inclSelf  include the node itself as the first node
 * @returns         array of all child nodes (if inclSelf is true
 *                  the given node is the first node in that array)
 */
BayesNet.prototype.getChildren = function (node, inclSelf)
{
  var children = new Array();
  if (inclSelf) children.push(node);         // (first node)
  for (var chId in this.children[node.id]) {
    children.push(this.nodes[chId]);
  }
  return (children);
};

/**
 * Get all nodes (dimensions) of the  markov blanket 
 * set of a node (i.e. the node itself, all parents,
 * all children, and all other parents of child-nodes).
 * @param node   the node to get the markov blanket set of
 * @returns      array of nodes in the markov blanket set, 
 *               where the given node is always the first
 */
BayesNet.prototype.getMarkovBlanket = function (node)
{
  var nodes = {};
  for (var paId in this.parents[node.id]) {
    nodes[paId] = this.nodes[paId];
  }
  for (var chId in this.children[node.id]) {
    nodes[chId] = this.nodes[chId]; 
    for (var siId in this.parents[chId]) {
      if (siId != node.id) {      // do not include 'node'
	nodes[siId] = this.nodes[siId];
      }
    }
  }

  var markovBlanket = new Array();
  markovBlanket.push(node);       // first node
  for (var nId in nodes) {        // all other nodes
    markovBlanket.push(nodes[nId]);
  }
  return (markovBlanket);
};

/**
 * Remove all evidence from all nodes
 */
BayesNet.prototype.rmEvidence = function ()
{
  for (var nId in this.nodes) {
    this.nodes[nId].f.init(1);
  }
};


// DIMENSIONS / NODES ===============================================
// DISCRETE DIMENSIONS ----------------------------------------------

/**
 * Constructor for a (discrete) dimension.
 * @param id      id
 * @param name    name
 * @param states  array of states
 */
function Dim (id, name, states) 
{
  if (id != undefined) {
    this.initDim(id, name, states);
  }
}

/* Inheritance */
// Dim.prototype          = ;
// Dim.prototype.parent   = ;
Dim.prototype.constructor = Dim; 

/**
 * Initialize this (discrete) dimension
 * (see constructor {@link Dim}).
 */
Dim.prototype.initDim = function (id, name, states)
{
  this.id     = id;     // id
  this.name   = name;   // name
  this.states = states; // array of states
};

// DISCRETE NODE IN A BAYESIAN NETWORK ------------------------------

/**
 * Constructor for a (normal, diskrete) node in a bayesian network
 * (which basically is a dimension and therfor inherits from Dim).
 * @param id      id
 * @param name    name
 * @param states  array of states
 * @param x       x-coordinate
 * @param y       y-coordinate
 */
function Node (id, name, states, x, y)
{
  if (id != undefined) {
    this.initNode(id, name, states, x, y);
  }
}

/* Inheritance */
Node.prototype             = new Dim();
Node.prototype.parent      = Dim.prototype;
Node.prototype.constructor = Node;

/**
 * Initialize this (normal, diskrete) node of a 
 * bayesian network (see constructor {@link Node}).
 */
Node.prototype.initNode = function (id, name, states, x, y)
{
  this.initDim(id, name, states);

  // coordinates
  this.x = x;
  this.y = y;

  // conditional probability table (to be set later)
  this.cpt = null;   

  // init a-posteriori probabilities
  var pvals = new Float32Array(states.length);
  for (var i = 0; i < states.length; i++) pvals[i] = 1/states.length;
  this.p = new Float32Table(new Array(this), pvals); 

  // init finding vector (evidence)
  this.f = new Float32Table(new Array(this)); 
};

/** 
 * Get a state by its index 
 */
Node.prototype.getState = function (s) 
{ 
  return (this.states[s]); 
};

/**
 * Remove evidence from this node 
 */
Node.prototype.rmEvidence = function ()
{
  this.f.init(1);
};

/** 
 * Recreate the CPT of this node, e.g. after the number of parents
 * has changed. Rescues as much information as possible.
 * @param meAndParents  array containg this node (first dimension) 
 *                      and all parent nodes (further dimensions)
 *                      (could be retrieved by 'BayesNet.getParents()')
 */
Node.prototype.adaptCPT = function (meAndParents)
{
  // conditional probability table
  var cpt = this.cpt;
  this.cpt = new Float32Table(meAndParents);
  cpt.cut(this.cpt);
  this.cpt.cut(cpt);
  if (this.cpt.dims.length <= cpt.dims.length) {
    this.cpt.marg(cpt);
  } else {
    this.cpt.infl(cpt);
  }
  this.cpt.uncut();
  this.cpt.cnorm();

  // posteriors
  var p = this.p;
  this.p = new Float32Table(new Array(this)); 
  p.cut(this.p);
  this.p.marg(p);

  // finding
  this.f = new Float32Table(new Array(this)); 
};

// STATES ===========================================================

/**
 * Constrcutor for a state of a (normal, discrete) 
 * node in a bayesian network.
 * @param id    (local) id (normally just the index)
 * @param name  name
 */
function State (id, name)
{
  if (id != undefined) {
    this.initState(id, name);
  }
}

/* Inheritance */
// State.prototype          = ;
// State.prototype.parent   = ;
State.prototype.constructor = State;

/**
 * Initialize this state of a (normal, discrete) node 
 * in a bayesian network (see constructor {@link State}).
 */
State.prototype.initState = function (id, name)
{
  this.id   = id;
  this.name = name;
};


// PROBABILITY TABLE ================================================

/**
 * Constructor for a probability table 
 * (conditional or unconditional)
 * @param dims    array of dimensions 
 * @param values  either an array of values, or a single 
 *                value to be used for all values [optional]
 */
function PTable (dims, values)
{
  if (dims != undefined) {
    this.initPTable(dims, values);
  }
}

/* Inheritance */
// PTable.prototype          = ;
// PTable.prototype.parent   = ;
PTable.prototype.constructor = PTable;

/**
 * Initialize this probability table 
 * (see constructor {@link PTable}).
 */
PTable.prototype.initPTable = function (dims, values)
{
  // attributes:
  // - dimensions, 
  // - states per dimension (do not use dims[d].states.length 
  //   because there might be states added or removed before 
  //   the probability table is adapted), 
  // - offset between two succeeding values in a dimension
  this.dims  = dims;  // array of dimensions
  this.sizes = {};    // map: id -> number of states (do not use dims[d].states.length)
  this.steps = {};    // map: id -> step width
  this.n     = 1;     // number of elements (in all dimensions)

  // create this.steps and count this.n
  for (var d = 0; d < dims.length; d++) {
    var dim = dims[d];
    if (this.steps[dim.id] !== undefined) {
      throw('Dimension id "' + dims[d].id + '" is not unique!');
    }
    this.sizes[dim.id] = dim.states.length;
    this.steps[dim.id] = this.n; 
    this.n *= dim.states.length; 
  }

  // set probability table (array)
  this.setValues(values);
};

/**
 * Set or create the array of values
 * @param values  either the array values to be set or one 
 *                single value to be used for all elements 
 *                of the array or nothing at all (will set 
 *                all elements to 1 in that case) 
 */
PTable.prototype.setValues = function (values) 
{ 
  if (values instanceof Array) {
    this.values = values;    
  } else {
    var v = (values === undefined ? 1 : values);
    this.values = new Array(this.n);
    for (var i = 0; i < this.n; i++) this.values[i] = v;
  }  
};

/**
 * Get the i-th probability value 
 */
PTable.prototype.get = function (i) 
{ 
  return (this.values[i]); 
};

/**
 * Set all elements to the same value v. 
 */
PTable.prototype.init = function (v) 
{ 
  for (var i = 0; i < this.values.length; i++) {
    this.values[i] = v;
  }
};

/**
 * Randomize all values.
 */
PTable.prototype.randomize = function ()
{
  for (var i = 0; i < this.values.length; i++) {
    this.values[i] = Math.random();
  }
};

/**
 * Normalize this table and return the normalization 
 * value (i.e. the sum of all values before normalization).
 */
PTable.prototype.norm = function () 
{
  var f = this.sum();
  if (f != 0) {
    for (var i = 0; i < this.values.length; i++) {
      this.values[i] /= f;
    }
  }
  return (f);
};

/**
 * Max-normalize this table and return the normalization 
 * value (i.e. the maximum of all values before normalization).
 */
PTable.prototype.mnorm = function () 
{
  var f = this.max();
  if (f != 0) {
    for (var i = 0; i < this.values.length; i++) {
      this.values[i] /= f;
    }
  }
  return (f);
};

/**
 * Normalize this table such that the sum over all values 
 * in the first dimension is 1 for all configurations of 
 * the other dimensions.
 */
PTable.prototype.cnorm = function () 
{
  var s = this.sizes[this.dims[0].id];
  for (var j = 0; j < this.n; j += s) {
    var f = 0;
    for (var i = 0; i < s; i++) f += this.values[j+i];
    if (f != 0) {
      for (var i = 0; i < s; i++) this.values[j+i] /= f;
    }
  }
};

/**
 * Get the sum of all values 
 */
PTable.prototype.sum = function () 
{
  var sum = this.values[0];
  for (var i = 1; i < this.values.length; i++) {
    sum += this.values[i];
  }
  return (sum);
};

/**
 * Get the product of all values 
 */
PTable.prototype.prod = function () 
{
  var prod = this.values[0];
  for (var i = 1; i < this.values.length; i++) {
    prod *= this.values[i];
  }
  return (prod);
};

/**
 * Get the max. value of all values 
 */
PTable.prototype.max = function () 
{
  var max = this.values[0];
  for (var i = 1; i < this.values.length; i++) {
    max = Math.max(max, this.values[i]);
  }
  return (max);
};

/**
 * Get the min. value of all values 
 */
PTable.prototype.min = function () 
{
  var min = this.values[0];
  for (var i = 1; i < this.values.length; i++) {
    min = Math.min(min, this.values[i]);
  }
  return (min);
};

/**
 * Multiplicate with another p-table.<br> 
 * Notice: This table must contain every dimension
 *         also contained in the other table!
 */
PTable.prototype.mult = function (other) 
{
  var fnc = function (t1, t2, i1, i2) { t1.values[i1] *= t2.values[i2]; };
  this.merge(fnc, 0, 0, 0, other);
};

/**
 * Divide by another p-table.<br> 
 * Notice: This table must contain every dimension
 *         also contained in the other table!
 */
PTable.prototype.div = function (other) 
{
  var fnc = function (t1, t2, i1, i2) { 
    (t1.values[i1]==0 && t2.values[i2]==0 ? 0 : t1.values[i1] /= t2.values[i2]); 
  };
  this.merge(fnc, 0, 0, 0, other);
};
  
/**
 * Add another p-table to this table.<br> 
 * Notice: This table must contain every dimension
 *         also contained in the other table!
 */
PTable.prototype.add = function (other) 
{
  var fnc = function (t1, t2, i1, i2) { t1.values[i1] += t2.values[i2]; };
  this.merge(fnc, 0, 0, 0, other);
};

/**
 * Substract another p-table from this table.<br> 
 * Notice: This table must contain every dimension
 *         also contained in the other table!
 */
PTable.prototype.substr = function (other) 
{
  var fnc = function (t1, t2, i1, i2) { t1.values[i1] *= t2.values[i2]; };
  this.merge(fnc, 0, 0, 0, other);
};

/**
 * Add (weighted) values from another (bigger) 
 * table (marginalization).<br> 
 * Notice: The other table must contain every
 *         dimension also contained in this table!
 */
PTable.prototype.cumulate = function (other, weight) 
{
  var fnc = function (t1, t2, i1, i2) { t2.values[i2] += t1.values[i1] * weight; };
  other.merge(fnc, 0, 0, 0, this);
};

/**
 * Fill from another (bigger) table (marginalization).<br> 
 * Notice: The other table must contain every
 *         dimension also contained in this table!
 */
PTable.prototype.marg = function (other) 
{
  var fnc = function (t1, t2, i1, i2) { t2.values[i2] += t1.values[i1]; };
  this.init(0);
  other.merge(fnc, 0, 0, 0, this);
};

/**
 * Fill from another (bigger) table (max-marginalization).<br> 
 * Notice: The other table must contain every
 *         dimension also contained in this table!
 */
PTable.prototype.mmarg = function (other) 
{
  var fnc = function (t1, t2, i1, i2) { t2.values[i2] = Math.max(t2.values[i2], t1.values[i1]); };
  this.init(0);
  other.merge(fnc, 0, 0, 0, this);
};

/**
 * Fill from another (smaller) table (inflation).<br> 
 * Notice: This table must contain every dimension
 *         also contained in the other table!
 */
PTable.prototype.infl = function (other) 
{
  var fnc = function (t1, t2, i1, i2) { t1.values[i1] += t2.values[i2]; };
  this.init(0);
  this.merge(fnc, 0, 0, 0, other);
};

/**
 * Recursively apply an arbitrary function on 
 * each element of this table. 
 * @param fnc     function with two parameters:<br>
 *                - table (this table)<br>
 *                - current element index into the table
 * @param d       current dimension index in this table (this.dims[d])
 * @param i       current element index in this table (this.values[j])
 */
PTable.prototype.apply = function (fnc, d, i) 
{
  if (d >= this.dims.length) {
    fnc(this, i);
  } else {
    var dim  = this.dims[d];
    var size = this.size(dim);
    for (var s = 0; s < size; s++) { // over all states
      this.apply(fnc, d+1, i);
      i += this.step(dim);
    }
  }
};

/**
 * Recursively apply an arbitrary function on each pair
 * of corresponding elements of this and another table.<br> 
 * Notice: This table must contain every dimension
 *         also contained in the other table!
 * @param fnc    function with four parameters:<br>
 *               - first table (this table)<br>
 *               - second table (other table)<br>
 *               - current element index into the first table<br>
 *               - current element index into the second table
 * @param d      current dimension index in this table (this.dims[d])
 * @param i      current element index in this table (this.values[j])
 * @param j      current element index in the other table (other.values[k])
 * @param other  other table
 */
PTable.prototype.merge = function (fnc, d, i, j, other) 
{
  if (d >= this.dims.length) {
    fnc(this, other, i, j);
  } else {
    var dim  = this.dims[d];
    var size = this.size(dim); 
    for (var s = 0; s < size; s++) {                      // over all states
      this.merge(fnc, d+1, i, j, other);
      i += this.step(dim);
      j += other.step(dim);
    }
  }
};

/**
 * Cut sizes (number of states) in all dimensions of this table
 * such that they are not bigger than in the given other table
 * (only for dimensions which are present in both tables).
 * @param pt  other table
 */
PTable.prototype.cut = function (pt)
{
  for (var d = 0; d < this.dims.length; d++) {
    var dim = this.dims[d];
    if (pt.sizes[dim.id] != undefined) {
      this.sizes[dim.id] = Math.min(this.sizes[dim.id], pt.sizes[dim.id]);
    }
  }
};

/**
 * Undo cuting of sizes (see 'cut()') such that they are
 * reflecting the actutal number of states in each dimension.
 */
PTable.prototype.uncut = function ()
{
  for (var d = 0; d < this.dims.length; d++) {
    var dim = this.dims[d];
    this.sizes[dim.id] = dim.states.length;
  }
};

/**
 * Get the size of a dimension (number of states). If the given 
 * dimension is not represented in this table, then 1 is returned.
 */
PTable.prototype.size = function (dim) 
{
  return ((this.sizes[dim.id] ? this.sizes[dim.id] : 1));
};

/**
 * Get the step width for the given dimension. If the given 
 * dimension is not represented in this table, then 0 is returned.
 */
PTable.prototype.step = function (dim) 
{
  return ((this.steps[dim.id] ? this.steps[dim.id] : 0));
};

/**
 * Exchange values with another table (both tables 
 * should have the same dimensions in the same order,
 * at least they must have the same number of elements)
 * @param pt  other table (exchange 'this.values' with 
 *            'pt.values')
 */
PTable.prototype.exchange = function (pt) 
{
  var values  = this.values;
  this.values = pt.values;
  pt.values   = values;
};

/**
 * Copy all values from another table (both tables 
 * should have the same dimensions in the same order,
 * at least they must have the same number of elements)
 * @param pt  other table
 */
PTable.prototype.copy = function (pt)
{
  // TODO: Would this.values = pt.values.slice() be faster?
  for (var i = 0; i < this.values.length; i++) {
    this.values[i] = pt.values[i];
  }
};

/**
 * Find the index for an element (given by its state configuration).
 * Some dimensions could be skipped (e.g. if you want to find the
 * first index in the first dimension set start=1 instead of 0).
 * @param conf   'conf[dimId]' = state-index for (at least) all dimensions
 *               of this table in 'dims[start..end]'
 * @param start  index of the first dimension to use (use dims[start..end])
 * @param end    index of the last dimension to use  (use dims[start..end])
 * @returns      index into 'this.values' for the given state configuration
 */
PTable.prototype.getIndex = function (conf, start, end)
{
  var index = 0;
  for (var d = start; d <= end; d++) {
    var dim = this.dims[d];
    index += this.steps[dim.id] * conf[dim.id];
  }
  return (index);
};

// PROBABILITY TABLE (FLOAT32) --------------------------------------

/**
 * Constructor for a probability table of float32 values 
 * (conditional or unconditional)
 * @param dims    array of dimensions 
 * @param values  either an array of values, or a single 
 *                value to be used for all values [optional]
 */
function Float32Table (dims, values)
{
  if (dims != undefined) {
    this.initPTable(dims, values);
  }
}

/* Inheritance */
Float32Table.prototype             = new PTable();
Float32Table.prototype.parent      = PTable.prototype;
Float32Table.prototype.constructor = Float32Table;

/**
 * Set or create the array of values
 * @param values  either the array values to be set or one 
 *                single value to be used for all elements 
 *                of the array or nothing at all (will set 
 *                all elements to 1 in that case) 
 */
Float32Table.prototype.setValues = function (values) 
{ 
  if (values instanceof Float32Array) {
    this.values = values;    
  } else if (values instanceof Array) {
    this.values = new Float32Array(values);
  } else {
    var v = (values === undefined ? 1 : parseFloat(values));
    this.values = new Float32Array(this.n);
    for (var i = 0; i < this.n; i++) this.values[i] = v;
  }
};

/**
 * For compatability reasons we want to have a 
 * slice-method, just as for normal 'Arrays'.
 * @param a  start index
 * @param b  end index + 1
 * @return   new 'Float32Array' (it's a copy not, 
 *           i.e. just a new view on the same data)
 */
Float32Array.prototype.slice = function (a, b) 
{ 
  // notice: 'subarray()' just creates a new view on the same data! 
  return (new Float32Array(this.subarray((a === undefined ? 0           : a), 
                                         (b === undefined ? this.length : b)))); 
};

/**
 * For compatability reasons we want to have a 
 * join-method, just as for normal 'Arrays'.
 * @param sep  separator
 * @return     all elements joined into one string
 */
Float32Array.prototype.join = function (sep) 
{
  var arr = new Array();
  for (var i = 0; i < this.length; i++) arr.push(this[i]);
  return (arr.join(sep));
};


// DATA (STATIC) ====================================================

/**
 * Constructor for a data collection.
 */
function Data ()
{
  this.initData();
}

/* Inheritance */
// Data.prototype          = ;
// Data.prototype.parent   = ;
Data.prototype.constructor = Data;

/**
 * Re-initialize this data collection
 * (see constructor {@link Data}).
 */
Data.prototype.initData = function ()
{
  this.data = {};
};

/**
 * Store one piece of data.
 * @param id  id (normally a node id)
 * @param d   data to be stored (you might want
 *            to store a copy of the actual data!)
 */
Data.prototype.store = function (id, d)
{
  this.data[id] = d;
};

/**
 * Remove data.
 * @param id  id (normally a node id) [optional]
 */
Data.prototype.unstore = function (id)
{
  delete (this.data[id]);
};

/**
 * Is there data stored?
 * @param id  id (normally a node id)
 * @returns   true if there is data stored; 
 *            false otherwise
 */
Data.prototype.isStored = function (id)
{
  return (this.data[id] != undefined);
};

/**
 * Get one piece of stored data.
 * @param t   time slice (0,1,2,...)
 * @param id  id (normally a node id)
 * @returns   piece of data or 'undefined' if nothing is
 *            stored for the given id in time slice t
 */
Data.prototype.stored = function (id)
{
  return (this.data[id]);
};


// PROPAGATORS ======================================================
// FULL JOINT PROPAGATOR FOR STATIC BAYESIAN NETWORKS ---------------

/**
 * Constructor for a simple propagator which creates the 
 * full joint propability table for all nodes.
 * This approach is usable for the bayesian networks which 
 * contain a few nodes only, because the size of the full 
 * joint probability table grows exponentially with the 
 * number of nodes!
 * @param net  {@link BayesNet}
 * @param max  true for max-propagation
 */
function FullJoint (net, max)
{
  if (net != undefined) {
    this.initFullJoint(net, max);
  }
}

/* Inheritance */
// FullJoint.prototype          = ;
// FullJoint.prototype.parent   = ;
FullJoint.prototype.constructor = FullJoint;

/**
 * Compile / Recompile the given bayesian network 
 * (see constructor {@link FullJoint}).
 */
FullJoint.prototype.initFullJoint = function (net, max) 
{
  this.id    = net.id + '.fjp';
  this.name  = 'Full joint propagator for "' + net.name + '"';
  this.net   = net;          // bayesian network
  this.max   = max;          // max-propagation?
  this.pe    = 1.0;          // probability of evidence / of the most probable config (max-prop)
  this.nodes = new Array();  // array of nodes
  for (var id in net.nodes) {
    this.nodes[this.nodes.length] = net.nodes[id];
  }

  // create and init full joint probability table (fjpt)
  this.fjpt = new Float32Table(this.nodes);
  for (var i = 0; i < this.nodes.length; i++) {
    this.fjpt.mult(this.nodes[i].cpt); // multiply in node cpt's
  }

  // stored probability values (for reinitialization)
  this.initData = new Data();
  this.initData.store('fjpt', this.fjpt.values);
};

/**
 * Re-load probabilities from the bayesian network
 * (probabilities of some nodes might have changed
 * or the number of states of one dimensions might 
 * have changed)
 */
FullJoint.prototype.reload = function () 
{
  this.fjpt.initPTable(this.fjpt.dims);   // number of states might have changed!
  for (var i = 0; i < this.nodes.length; i++) {
    this.fjpt.mult(this.nodes[i].cpt);    // multiply in node cpt's
  }
  this.initData.store('fjpt', this.fjpt.values);
};

/**
 * Fully recreate this propagator.
 */
FullJoint.prototype.recreate = function () 
{
  this.initFullJoint(this.net);
};

/**
 * Return a joint probability table containing
 * at least the given node and all its parents.
 * node: node in the bayesian network ('this.net')
 * @returns  joint probability table P(node,pa(node),...)
 */
FullJoint.prototype.getJPT = function (node) 
{
  return (this.fjpt);
};

/**
 * Propagate (and update a-posteriori probability of all nodes).
 * @param observe  use current evidence?
 * @param update   update posterior probability tables of all nodes?
 * @returns        probability of evidence given the model:    P(e|M)<br>
 *                 or of the most probable config. (max-prop): P(C|e)
 */ 
FullJoint.prototype.propagate = function (observe, update) 
{
  this.init();                     // re-initialize full joint prop. table
  if (observe) this.observe();     // multiply in all evidence   
  this.pe = (this.max              // normalize and get probability
	     ? this.fjpt.mnorm()   // ...of the most probable config. (P(C|e))
	     : this.fjpt.norm());  // ...or of the current evidence   (P(e|M))
  if (update) this.update();       // update posteriors of all nodes
  return (this.pe);
};

/**
 * Reinitialize full joint probability table for next propagation.
 */ 
FullJoint.prototype.init = function () 
{
  this.fjpt.values = this.initData.stored('fjpt').slice();
};

/**
 * Multiply in all finding vectors of all nodes
 */ 
FullJoint.prototype.observe = function () 
{
  for (var i = 0; i < this.nodes.length; i++) {
    this.fjpt.mult(this.nodes[i].f);
  }
};

/**
 * Update a-posteriori probability of all nodes.
 */ 
FullJoint.prototype.update = function () 
{
  for (var i = 0; i < this.nodes.length; i++) {
    (this.max 
     ? this.nodes[i].p.mmarg(this.fjpt) 
     : this.nodes[i].p.marg(this.fjpt)); 
  }
};
