/*
 * -------------------------------------------------------------------------
 *
 *  dhtml.js Copyright (C) 2011, 
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *   Author: Patrick Rammelt 
 *   E-Mail: patrick.rammelt@go4more.de
 *   Site:   http://www.patricks-seite.de
 * -------------------------------------------------------------------------
 *
 * Helper functions for dynamic html. Some of these methods are used to 
 * avoid some problems with outdated browsers (e.g. IE6). Nevertheless
 * not every old browser is supported (it has to have basic DOM 
 * functionality).
 */

/* 
 * Main: check
 */
//if (!document.getElementById) {
//  alert("Your browser seems to be very old (it doesn't support DOM), so some script won't work (at least not correctly), sorry...");
//}

// GLOBALS =================================================================

/**
 * Globals
 */
var DHTML = 
{
  /** CSS3-animations: animation name -&gt; style-element containg key-frames */
  animationName2StyleSheet : {},
 
  /** CSS3-animations: element Id -&gt; animation callback function */ 
  elementId2animationStartCallback : {},

  /** CSS3-animations: element Id -&gt; animation callback function */ 
  elementId2animationIterationCallback : {},
  
  /** CSS3-animations: element Id -&gt; animation callback function */ 
  elementId2animationEndCallback : {},

  /** JavaScript-animations: animation Id -&gt; animation object */
  animations : {},
  
  /** Processes: process Id -&gt; process object */
  processes : {}
};

// BROWSER-CHECK ===========================================================

/** 
 * Browser recognition
 */  
function isBrowser (name)
{  
  var agent = navigator.userAgent.toLowerCase();  
  return (agent.indexOf(name.toLowerCase()) >- 1);
}  

// MATH ====================================================================

/**
 * Converts numeric (float) value to integer (cuts off decimals).
 */
function toInt (v) 
{
  if (Math.abs(v) < 1) return (0);  // notice: parseInt("3.6e-10") = 3
  else return (parseInt(v));
}

/**
 * Checks the given value to be number (notice: 'IsNumber()' 
 * @returns  true for strings containing white-space only).
 */
function isNum (v) 
{ 
  try {
    v.toFixed(0);
    return (true);
  } catch (exception) {
    return (false);
  }
}


// COLORS ==================================================================

/**
 * RGB-Color
 * @param r  red (0..255)
 * @param g  green (0..255)
 * @param b  blue (0..255)
 * @returns  color as a string ("rgb(...)")
 */
function rgb (r, g, b) 
{ 
  return ('rgb(' + r + ',' + g + ',' + b + ')'); 
}

/**
 * RGB-Color
 * @param r  red (0..255)
 * @param g  green (0..255)
 * @param b  blue (0..255)
 * @param a  alpha (0..1)
 * @returns  color as a string ("rgb(...)")
 */
function rgba (r, g, b, a) 
{ 
  return ('rgb(' + r + ',' + g + ',' + b + ',' + a + ')'); 
}

/**
 * HSL-Color
 * @param h  hue (0..360)
 * @param s  saturation (0..100)
 * @param l  luminance (0..100)
 * @returns  color as a string ("hsl(...)")
 */
function hsl (h, s, l) 
{ 
  return ('hsl(' + h + ',' + s + '%,' + l + '%)'); 
}

/**
 * HSLA-Color
 * @param h  hue (0..360)
 * @param s  saturation (0..100)
 * @param l  luminance (0..100)
 * @param a  alpha (0..1)
 * @returns  color as a string ("hsla(...)")
 */
function hsla (h, s, l, a) 
{ 
  return ('hsl(' + h + ',' + s + '%,' + l + '%,' + a + ')'); 
}

/**
 * Converts an RGB color value to HSL. Conversion formula
 * adapted from http://en.wikipedia.org/wiki/HSL_color_space.
 * Assumes r, g, and b are contained in the set [0, 255] and
 * returns h, s, and l in the set [0, 1].
 *
 * @param r  the red color value
 * @param g  the green color value
 * @param b  the blue color value
 * @return   the HSL representation
 */
function rgb2hsl (r, g, b)
{
  r /= 255, g /= 255, b /= 255;
  var max = Math.max(r, g, b);
  var min = Math.min(r, g, b);
  var h, s, l = (max + min) / 2;
  if (max == min){
    h = s = 0; // achromatic
  } else {
    var d = max - min;
    s = (l > 0.5 ? d / (2 - max - min) : d / (max + min));
    switch (max) {
    case r:  h = (g - b) / d + (g < b ? 6 : 0); break;
    case g:  h = (b - r) / d + 2;               break;
    case b:  h = (r - g) / d + 4;               break;
    default: h = 0; // (case never reached)
    }
    h /= 6;
  }
  return ([h, s, l]);
}

/**
 * Converts an HSL color value to RGB. Conversion formula
 * adapted from http://en.wikipedia.org/wiki/HSL_color_space.
 * Assumes h, s, and l are contained in the set [0, 1] and
 * returns r, g, and b in the set [0, 255].
 *
 * @param h  the hue
 * @param s  the saturation
 * @param l  the lightness
 * @return   the RGB representation
 */
function hsl2rgb (h, s, l)
{
  var r, g, b;
  if (s == 0) {
    r = g = b = l; // achromatic
  } else {
    function hue2rgb(p, q, t) 
    {
      if (t < 0) t += 1;
      if (t > 1) t -= 1;
      if (t < 1/6) return (p + (q - p) * 6 * t);
      if (t < 1/2) return (q);
      if (t < 2/3) return (p + (q - p) * (2/3 - t) * 6);
      return (p);
    }
    var q = l < 0.5 ? l * (1 + s) : l + s - l * s;
    var p = 2 * l - q;
    r = hue2rgb(p, q, h + 1/3);
    g = hue2rgb(p, q, h);
    b = hue2rgb(p, q, h - 1/3);
  }
  return ([r * 255, g * 255, b * 255]);
}

/**
 * Converts an RGB color value to HSV. Conversion formula
 * adapted from http://en.wikipedia.org/wiki/HSV_color_space.
 * Assumes r, g, and b are contained in the set [0, 255] and
 * returns h, s, and v in the set [0, 1].
 *
 * @param r  the red color value
 * @param g  the green color value
 * @param b  the blue color value
 * @return   the HSV representation
 */
function rgb2hsv (r, g, b)
{
  r = r/255, g = g/255, b = b/255;
  var max = Math.max(r, g, b);
  var min = Math.min(r, g, b);
  var h, s, v = max;
  
  var d = max - min;
  s = (max == 0 ? 0 : d / max);
  if (max == min) {
    h = 0; // achromatic
  } else {
    switch(max){
    case r:  h = (g - b) / d + (g < b ? 6 : 0); break;
    case g:  h = (b - r) / d + 2;               break;
    case b:  h = (r - g) / d + 4;               break;
    default: h = 0; // (case never reached)
    }
    h /= 6;
  }
  
  return ([h, s, v]);
}

/**
 * Converts an HSV color value to RGB. Conversion formula
 * adapted from http://en.wikipedia.org/wiki/HSV_color_space.
 * Assumes h, s, and v are contained in the set [0, 1] and
 * returns r, g, and b in the set [0, 255].
 *
 * @param h  the hue
 * @param s  the saturation
 * @param v  the value
 * @return   the RGB representation
 */
function hsv2rgb (h, s, v)
{
    var r, g, b;

    var i = Math.floor(h * 6);
    var f = h * 6 - i;
    var p = v * (1 - s);
    var q = v * (1 - f * s);
    var t = v * (1 - (1 - f) * s);

    switch (i % 6) {
        case 0:  r = v, g = t, b = p; break;
        case 1:  r = q, g = v, b = p; break;
        case 2:  r = p, g = v, b = t; break;
        case 3:  r = p, g = q, b = v; break;
        case 4:  r = t, g = p, b = v; break;
        case 5:  r = v, g = p, b = q; break;
        default: r = 0, g = 0, b = 0; break;
    }

    return ([r * 255, g * 255, b * 255]);
}

// STRING EXTENSIONS =======================================================

/**
 * String repeat. 
 */
String.prototype.repeat = function (times)
{
  return (new Array(times + 1).join(this));
};

/**
 * Converts text to html (e.g. ä -> &auml;) 
 */
String.prototype.toHTML = function ()
{
  var text = this;
  text = text.replace(/\&/g, "&#038;"); // (/g => replace all)
  text = text.replace(/</g, "&#060;");
  text = text.replace(/>/g, "&#062;");
  text = text.replace(/ä/g, "&auml;");
  text = text.replace(/ö/g, "&ouml;");
  text = text.replace(/ü/g, "&uuml;");
  text = text.replace(/Ä/g, "&Auml;");
  text = text.replace(/Ö/g, "&Ouml;");
  text = text.replace(/Ü/g, "&Uuml;");
  text = text.replace(/ß/g, "&szlig;");
  text = text.replace(/^---+\n/g,  "<hr width=\"100%\" size=\"2\">\n"); // horizontal line (start)
  text = text.replace(/\n---+\n/g, "<hr width=\"100%\" size=\"2\">\n"); // horizontal line
  text = text.replace(/\n/g, "<br>");
  return (text);
};


// CREATE AND SET ELEMENTS =================================================
  
/**
 * Create a new element.
 * @param tag     html element type
 * @param id      html element id
 * @param clazz   html element class
 * @param hidden  should the element being set to be invisible?
 * @returns       newly created element
 */
function createNewElement (tag, id, clazz, hidden)
{
  var elem = document.createElement(tag);
  if (id    != undefined) elem.id        = id;
  if (clazz != undefined) elem.className = clazz;
  if (hidden) elem.style.visibility = 'hidden';
  return (elem);
}

/**
 * Create a new element and append it to the given parent.
 * @param parent  parent element
 * @param tag     html element type
 * @param id      html element id
 * @param clazz   html element class
 * @param hidden  should the element being set to be invisible?
 * @returns       newly created element (already appended to parent)
 */
function appendNewElement (parent, tag, id, clazz, hidden)
{
  var elem = createNewElement(tag, id, clazz, hidden);
  parent.appendChild(elem);
  return (elem);
}


// REMOVE ELEMENTS =========================================================

/**
 * Remove the element with the given id. 
 */
function removeElement (id) 
{
  var elem = document.getElementById(id);
  if (elem) elem.parentNode.removeChild(elem);
}


// GET ELEMENTS ============================================================

/**
 * Get elements by tag name (only those elements having a valid ID).<br>
 *
 * Notice: 
 *   This function is useful only because there seems to be a bug  
 *   in SeaMonkey such that there could be multiple instances for 
 *   the same element if retrieved by: 'getElementsByTagName()'.
 *   Such duplicates are often incomplete representations of the 
 *   "real" elements (e.g. missing sub-nodes).
 *   Therfor we use elements retrived by 'getElementsByTagName()'
 *   only to get a unique set of IDs and retrieve the "real" set 
 *   of elements by 'document.getElementById()' for each such ID
 *   (i.e. the ID is used for two reasons: uniqueness and getting
 *   "real" elements).
 *
 * @param root     root element (search below this element)
 * @param tagName  tag name
 * @returns        an array of found elements
 */
function getIdElementsByTagName (root, tagName)
{
  // final array of elements (unique)
  var unique = new Array();

  // get elements (might contain incomplete duplicates of "real" elements)
  var elements = root.getElementsByTagName(tagName);

  // fill final array of unique elements
  var ids = {};              // "associative map" of IDs to recognize duplicates
  for (var i in elements) { 
    var id = elements[i].id;
    if (id && !ids[id]) {    // id exists and it is the first time we see it
      ids[id] = true; 

      // due to the bug in SeaMonkey, we do not use elements[i] 
      // directly, because this element might be wrong/incomplete
      // instead we get the "real" element by its ID
      var element = document.getElementById(id);
      unique[unique.length] = element;
    }
  }

  // return unique array of elements
  return (unique);
}


// GET COORDINATES =========================================================


/**
 * Get left x coordinate relative to the document. 
 */
function getLeft (element)
{
  var x = 0;
  while (element != null) {
    x += element.offsetLeft;
    element = element.offsetParent;
  }
  return (x);
}

/**
 * Get top y cordinate relative to the document. 
 */
function getTop (element)
{
  var y = 0;
  while (element != null) {
    y += element.offsetTop;
    element = element.offsetParent;
  }
  return (y);
}


// GET SIZE ================================================================

/**
 * Get width of an element 
 * (including scrolled, i.e. invisible parts). 
 */
function getWidth (element) 
{ 
  return (Math.max(element.offsetWidth,  element.scrollWidth)); 
}

/**
 * Get height of an element 
 * (including scrolled, i.e. invisible parts).
 */
function getHeight (element) 
{ 
  return (Math.max(element.offsetHeight,  element.scrollHeight)); 
}


// SCROLL OFFSETS ==========================================================

/**
 * Get top-most visible y coordinate. 
 */
function getVScrollOffset () 
{
  if (typeof window.pageYOffset == "number") {
    return (window.pageYOffset);
  } else if (typeof document.body.scrollTop == "number") {
    return (document.body.scrollTop); // IE6
  } else {
    return (0); // (we don't know)
  }
}

/**
 * Get left-most visible x coordinate. 
 */
function getHScrollOffset () 
{
  if (typeof window.pageXOffset == "number") {
    return (window.pageXOffset);
  } else if (typeof document.body.scrollLeft == "number") {
    return (document.body.scrollLeft); // IE6
  } else {
    return (0); // (we don't know)
  }
}


// EVENTS ==================================================================

/**
 * Add an event listener to an element.
 * @param element   html element
 * @param type      event type name (e.g. "mousemove")
 * @param listener  function with one parameter (event)
 * @param onCapture notify listener on capture phase,
 *                  i.e. a listener on a parent element will be 
 *                  notified before listeners on the same event 
 *                  applied to its children (otherwise the listener 
 *                  will be notified on the bubbling phase)
 */
function addEventListener (element, type, listener, onCapture)
{
  if (element.addEventListener) {
    element.addEventListener(type, listener, onCapture);
  } else if (element.attachEvent) {
    element.attachEvent('on' + type, listener);
  }
}

/**
 * Remove an event listener to an element.
 * @param element   html element
 * @param type      event type name (e.g. "mousemove")
 * @param listener  function with one parameter (event)
 * @param onCapture listener notified on capture phase?
 */
function removeEventListener (element, type, listener, onCapture)
{
  if (element.removeEventListener) {
    element.removeEventListener(type, listener, onCapture);
  } else if (element.detachEvent) {
    element.detachEvent('on' + type, listener);
  }
}

/**
 * Absolute mouse position of the given event. 
 */
function pageX (evnt) 
{
  return (evnt.pageX ? evnt.pageX : evnt.clientX + getHScrollOffset());
}

/**
 * Absolute mouse position of the given event. 
 */
function pageY (evnt) 
{
  return (evnt.pageY ? evnt.pageY : evnt.clientY + getVScrollOffset());
}

/**
 * Cancel default behavior for the given event. 
 * @param evnt  event (allowed to be undefined)
 */
function preventDefault (evnt)
{
  if (evnt) {
    if (evnt.preventDefault) {
      evnt.preventDefault(); 
    } else {
      evnt.returnValue = false;
    }
  }
}

/**
 * Get the element that fired the given event. 
 */
function target (evnt) 
{
  return (evnt.target ? evnt.target : evnt.srcElement);
}

/** 
 * Dispatch (fire) event
 * @param element  element
 * @param evnt     event
 */
function dispatchEvent (element, evnt)
{
  if (element.dispatchEvent) {
    element.dispatchEvent(evnt);
  } else {
    element.fireEvent("on" + evnt.eventType, evnt);
  }  
}

/**
 * Copy a mouse event (could be used e.g. to
 * fire/dispatch an event again on another element)
 * @param srcEvnt  mouse event to copy
 * @returns        copied mouse event
 */
function copyMouseEvent (srcEvnt)
{
  if (document.createEvent) {
    var destEvnt = document.createEvent('MouseEvents');
    destEvnt.initMouseEvent(srcEvnt.type, 
                            srcEvnt.bubbles, 
                            srcEvnt.cancelable, 
                            window, 
                            srcEvnt.detail, 
                            srcEvnt.screenX,  srcEvnt.screenY, 
                            srcEvnt.clientX,  srcEvnt.clientY, 
                            srcEvnt.ctrlKey,  srcEvnt.altKey, 
                            srcEvnt.shiftKey, srcEvnt.metaKey, 
                            srcEvnt.button,   srcEvnt.relatedTarget);  
    return (destEvnt);
  } else if (document.createEventObject) {
    var destEvnt = document.createEventObject(srcEvnt);
    return (destEvnt);
  } else {
    return (null);
  }
}

// WAIT FOR ================================================================

/**
 * Waits until a function has succeeded
 * @param func     function to be called (gets one parameter: number of 
 *                 trials left and has to return true in case of success
 *                 or false if waiting should continue for another timeout)
 * @param timeout  timeout before first and every succeeding trial
 * @param trials   max. number of trials
 */
function waitFor (func, timeout, trials)
{
  setTimeout(function() {
    if (!func(trials) && trials > 0) {
      waitFor(func, timeout, trials-1);  // still waiting
    } 
  }, timeout);
}


// DUMMY FUNCTION ==========================================================

/**
 * Function that doesn't do anything...
 */
function dummy () {}

/**
 * Function that doesn't do anything...
 */
function dummyEventHandler (evnt) {}


// I/O (XML-HTTP-REQUEST / AJAX) =========================================== 

/**
 * Browser compatability version for:<br>
 * new XMLHttpRequest()
 */
function newXMLHttpRequest ()
{
  var req = null;
  try {
    req = new XMLHttpRequest(); // Mozilla, Opera, Safari, IE (v7...)
  } catch(e) {
    try {
      req = new ActiveXObject("Microsoft.XMLHTTP"); // IE (v6)
    } catch(e) {
      try {
        req = new ActiveXObject("Msxml2.XMLHTTP");  // IE (v5)
      } catch(e) {
        req = null;
      }
    }
  }
  return (req);
}

/**
 * Parse XML from string and return the DOM element
 * @param str  string containing XML
 * @return     dom element
 */
function parseXML (str)
{
  var xml;
  if (window.DOMParser) {
    var parser = new DOMParser();
    xml = parser.parseFromString(str, "text/xml");
  } else if (window.ActiveXObject) {
    xml = new ActiveXObject("Microsoft.XMLDOM");
    xml.async = false;
    xml.loadXML(str);
  } else {
    throw("Cannot parse XML");
  }
  return (xml);
}

// PRE-LOAD IMAGES =========================================================

/**
 * Load image
 * @param path       base path
 * @param src        image source
 * @param unloaded   callback function called before the image gets loaded
 *                   with one parameter: the not yet loaded 'Image'
 * @param loaded     callback function called after the image has been 
 *                   loaded, with one parameter: the loaded 'Image'
 * @return           the image source (i.e. 'src') 
 *                   (<b>not the loaded 'Image'</b>)
 */
function loadImg (path, src, unloaded, loaded)
{
  var img = new Image();
  unloaded(img);
  img.onload = function () { loaded(img); };
  img.src = path + src;
  return (src);
}

// HTML5 / CSS3 =====================================================

/**
 * Set the given transform to the given element.
 * @param element    html element (div)
 * @param transform  transformation (CSS3)
 */
function setTransform (element, transform)
{
  element.style.transform       = transform;
  element.style.OTransform      = transform;
  element.style.MozTransform    = transform;
  element.style.KhtmlTransform  = transform;
  element.style.WebkitTransform = transform;
  element.style.msTransform     = transform;
  
  // makes smooth edges in Safari/Chrome but has also
  // serious side effects (elements don't move...) 
  //element.style.WebkitBackfaceVisibility = 'hidden'; 
}

/**
 * Is the given element transformable? (i.e. 
 * is CSS3 transform supported by the browser)
 * @param element  html element (div)
 * @returns        true iff CSS3 transform is supported
 */
function isTransformable (element)
{
  return (element.style.transform       !== undefined ||
          element.style.OTransform      !== undefined ||
          element.style.MozTransform    !== undefined ||
          element.style.KhtmlTransform  !== undefined ||
          element.style.WebkitTransform !== undefined ||
          element.style.msTransform     !== undefined);
}

/**
 * Set the given border image.
 * @param element  html element
 * @param n1       width of the top part of the image
 * @param e1       width of the right of the image
 * @param s1       width of the bottom part of the image
 * @param w1       width of the left part of the image
 * @param n2       width of the top border
 * @param e2       width of the right border
 * @param s2       width of the bottom border
 * @param w2       width of the left border
 * @param stretch  stretch the image to fill the element
 *                 (otherwise repeat it)
 */
function setBorderImage(element, src, 
                        n1, e1, s1, w1, 
                        n2, e2, s2, w2,
                        stretch)
{
  // border image (old version FF 14, Seamonkey 2.11) XXX
  var borderImgOld 
    = 'url(' + src + ') ' 
    + n1 + ' ' + e1 + ' ' + s1 + ' ' + w1 + ' ' 
    + (stretch ? 'stretch stretch' : 'repeat repeat');

  element.style.borderImage       = borderImgOld;
  element.style.OBorderImage      = borderImgOld;
  element.style.MozBorderImage    = borderImgOld;
  element.style.KhtmlBorderImage  = borderImgOld;
  element.style.WebkitBorderImage = borderImgOld;
  element.style.msBorderImage     = borderImgOld;

  // border image (new version: FF 15, Seamonkey 2.12)
  var borderImg 
    = 'url(' + src + ') fill ' 
    + n1 + ' ' + e1 + ' ' + s1 + ' ' + w1 + ' ' 
    + (stretch ? 'stretch stretch' : 'repeat repeat');
  
  element.style.borderImage       = borderImg;
  element.style.OBorderImage      = borderImg;
  element.style.MozBorderImage    = borderImg;
  element.style.KhtmlBorderImage  = borderImg;
  element.style.WebkitBorderImage = borderImg;
  element.style.msBorderImage     = borderImg;
  
  element.style.borderStyle = 'solid'; 

  // border widths
  element.style.borderTopWidth    = n2;
  element.style.borderRightWidth  = e2;
  element.style.borderBottomWidth = s2;
  element.style.borderLeftWidth   = w2;  
}

/**
 * Supports the given element border-images? (i.e. 
 * is CSS3 border-image supported by the browser)
 * @param element  html element (div)
 * @returns        true iff CSS3 border-image is supported
 */
function hasBorderImage (element)
{
  return (element.style.borderImage       !== undefined ||
          element.style.OBorderImage      !== undefined ||
          element.style.MozBorderImage    !== undefined ||
          element.style.KhtmlBorderImage  !== undefined ||
          element.style.WebkitborderImage !== undefined ||
          element.style.msBorderImage     !== undefined);
}

/**
 * Set the given opacity.
 * @param element  html element
 * @param opacity  opacity [0..1]
 */
function setOpacity (element, opacity)
{
  element.style.opacity       = opacity; 
  element.style.OOpacity      = opacity;
  element.style.MozOpacity    = opacity;
  element.style.KhtmlOpacity  = opacity; 
  element.style.WebkitOpacity = opacity;
  element.style.msOpacity     = opacity;
  element.style.filter        = 'alpha(opacity=' + toInt(opacity*100) + ')';  // IE7
  element.style.MsFilter      = 'progid:DXImageTransform.Microsoft.Alpha(Opacity=' + toInt(opacity*100) + ')'; // IE8
}

/**
 * Supports the given element opacity? (i.e. 
 * is CSS3 opacity supported by the browser)
 * @param element  html element (div)
 * @returns        true iff CSS3 opacity is supported
 */
function hasOpacity (element)
{
  return (element.style.opacity       !== undefined ||
          element.style.OOpacity      !== undefined ||
          element.style.MozOpacity    !== undefined ||
          element.style.KhtmlOpacity  !== undefined ||
          element.style.WebkitOpacity !== undefined ||
          element.style.msOpacity     !== undefined || 
          element.style.filter        !== undefined ||
          element.style.MsFilter      !== undefined);
}

/**
 * Set keyframes for animations.<br>
 * Example:
 * <pre>
 * setAnimationKeyFrames('myAnimations', 
 *    { 'myAnim1' : { '0%'   : 'PRE-transform: rotate(0deg);'  }
 *                  { '100%' : 'PRE-transform: rotate(90deg);' }
 *    { 'myAnim2' : { '0%'   : 'margin-top:   0px; PRE-animation-timing-function: ease-in;  }
 *    { 'myAnim2' : { '50%'  : 'margin-top:   0px; PRE-animation-timing-function: ease-out; }
 *                  { '100%' : 'margin-top: 100px; }});</pre>
 * <dl><dt>Notice:</dt><dd>
 * For multi-step animations ('myAnim2') a separate 
 * animation-timing-function should be set for each
 * step (overwriting a general setting when calling
 * an animation)</dd></dl>
 * 
 * @param id               just an unique Id for this 
 *                         set of keyframes
 * @param name2step2style  mapping: animation name -&gt;
 *                         percentage (e.g. '10%') -&gt;
 *                         css-style-settings (where all
 *                         browser-dependent attributes 
 *                         starts with prefix "PRE-" 
 */
function setAnimationKeyFrames (id, name2step2style)
{    
  // get style element or create a new one (notice: each
  // style element contains the keyframes for one Id only)
  var sheet = DHTML.animationName2StyleSheet[id];
  if (!sheet) {
    var head  = document.getElementsByTagName('head')[0];
    var sheet = appendNewElement(head, 'style');
    DHTML.animationName2StyleSheet[id] = sheet;
  }
  
  // create rules
  var rules    = new Array();
  var prefixes = ['', '-o-', '-moz-', '-khtml-', '-webkit-', '-ms-'];
  for (var name in name2step2style) {
    var steps = new Array;
    for (var step in name2step2style[name]) {
      steps.push(step + ' { ' + name2step2style[name][step] + ' }');
    }
    var frames = steps.join("\n");
    for (var i in prefixes) {
      rules.push('@' + prefixes[i] + 'keyframes ' + name + " {\n" +
                 '  ' + frames.replace(/PRE-/g, prefixes[i]) + 
                 "\n}");
    }
  }
  
  // set rules to the style element
  sheet.innerHTML = rules.join("\n");
}

/**
 * Set the given animation(s) (CSS3 animations).
 * Syntax:<br>
 * "name duration timing-function delay iteration-count direction fill-mode, ..."<br>
 * <dl><dt>Notice:</dt><dd> 
 * Multiple comma-separated animations doesn't seem to work 
 * (at least in mozilla browsers), so try using to add an
 * animation-end-listener (see 'onAnimationEnd()') which 
 * sets/triggers the next animation (calling this function).</dd></dl>  
 * where:
 * <ul>
 * <li>name is the animation name</li>
 * <li>duration is the duration time (e.g. '4000ms' or '4s')</li>
 * <li>timing-function is one of 'ease', 'ease-in', 'ease-out', 
 *     'ease-in-out', 'linear', 'cubic-bezier(x1, y1, x2, y2)' 
 *     (e.g. 'cubic-bezier(0.5, 0.2, 0.3, 1.0)');<br>
 *     <b>Notice:</b> For multi-step animations this should be overwritten 
 *                    by setting a separate timimg-function in the keyframe
 *                    definitions for each step(!)</li> 
 * <li>delay time delay before each interations (e.g. '2000ms' or '2s')</li>
 * <li>iteration-count is the number of full iterations 
 *     (e.g.: '1', '2', '3', ... 'infinite')</li>
 * <li>direction direction for &gt; 1 iterations 'normal', 'alternate'</li>
 * <li>fill-mode is one of 'none', 'forwards' (set 0%-properties immediately 
 *     even, if the animation is started), 'backwards' (keep properties set 
 *     by the animation after the animation has finished), 'both'
 * </ul>
 * For each animation name there has to be a corresponding definition 
 * of keyframes in the CSS-style-sheets for that site, e.g.:
 * <pre>
 * &#064;keyframes myAnimationName
 * {
 *    0%   { background: red;    }
 *    25%  { background: yellow; }
 *    50%  { background: blue;   }
 *    100% { background: green;  }
 * }</pre>
 * <dl><dt>Notice:</dt><dd> 
 * currently &#064;keyframes need to come in several variants:
 * &#064;-moz-keyframes, &#064;-webkit-keyframes, ...
 * (see 'setAnimationKeyFrames()' for creating them dynamically)</dd></dl>
 * 
 * @param element    html element (div), which must have an Id(!)
 * @param animation  animation (comma separated multiple animations)
 * @param onStart    function called (once) when animation starts
 * @param onIter     function called after each iteration (multi-iter.)
 * @param onEnd      function called (once) after animation has finished
 * @param run        initially run this animation? 
 */
function setAnimation (element, animation, onStart, onIter, onEnd, run)
{
  element.style.animation       = animation; 
  element.style.OAnimation      = animation;
  element.style.MozAnimation    = animation;
  element.style.KhtmlAnimation  = animation; 
  element.style.WebkitAnimation = animation;
  element.style.msAnimation     = animation;

  if (onStart) onAnimationStart(element, onStart);
  if (onIter)  onAnimationIter(element, onIter);
  if (onEnd)   onAnimationEnd(element, onEnd);
  
  runAnimation(element, run);
}

/**
 * Remove CSS3 animation from the given element.
 * If an animation should be re-played this function 
 * should be used to remove the animation before it
 * is re-set (which sould be called asynchronously). 
 * @param element   html element (div)
 */
function unsetAnimation (element)
{
  element.style.animation       = ''; 
  element.style.OAnimation      = '';
  element.style.MozAnimation    = '';
  element.style.KhtmlAnimation  = ''; 
  element.style.WebkitAnimation = '';
  element.style.msAnimation     = '';
}

/**
 * Set play-state of an animated element (CSS3 animations).
 * @param element  html element (div)
 * @param run      run? (otherwise pause) 
 */
function runAnimation (element, run)
{
  var state = (run ? 'running' : 'paused');
  element.style.animationPlayState       = state; 
  element.style.OAnimationPlayState      = state;
  element.style.MozAnimationPlayState    = state;
  element.style.KhtmlAnimationPlayState  = state; 
  element.style.WebkitAnimationPlayState = state;
  element.style.msAnimationPlayState     = state;
}

/** 
 * Add an event listener called when the animation 
 * set for the given element starts. (use this before 
 * actually setting the animation). This listener is 
 * unique, i.e. any previously listener set via this 
 * function will be removed in advance.
 * @param element  html element (div), which must have an Id(!)
 * @param callback function called when the animation starts
 *                 receiving an event parameter with the 
 *                 following special attributes:
 *                 <ul>
 *                 <li>type = animationstart</li>
 *                 <li>animationName</li>
 *                 <li>elapsedTime</li>
 *                 </ul>
 */
function onAnimationStart (element, callback)
{
  var oldCallback = DHTML.elementId2animationStartCallback[element.id];
  if (oldCallback) {
    element.removeEventListener('animationstart',       oldCallback, false); // (used by mozilla) 
    element.removeEventListener('oAnimationStart',      oldCallback, false);
    element.removeEventListener('mozAnimationStart',    oldCallback, false); // (not used by mozilla)
    element.removeEventListener('khtmlAnimationStart',  oldCallback, false);
    element.removeEventListener('webkitAnimationStart', oldCallback, false);
    element.removeEventListener('msAnimationStart',     oldCallback, false);
  }
  if (callback) {
    DHTML.elementId2animationStartCallback[element.id] = callback;
    element.addEventListener('animationstart',       callback, false); // used by mozilla
    element.addEventListener('oAnimationStart',      callback, false);
    element.addEventListener('mozAnimationStart',    callback, false); // (not used by mozilla)
    element.addEventListener('khtmlAnimationStart',  callback, false);
    element.addEventListener('webkitAnimationStart', callback, false);
    element.addEventListener('msAnimationStart',     callback, false);
  }
}

/** 
 * Add an event listener called when the animation 
 * set for the given element starts. (use this before 
 * actually setting the animation). This listener is 
 * unique, i.e. any previously listener set via this 
 * function will be removed in advance.
 * @param element  html element (div), which must have an Id(!)
 * @param callback function called when the animation starts
 *                 receiving an event parameter with the 
 *                 following special attributes:
 *                 <ul>
 *                 <li>type = animationiteration</li>
 *                 <li>animationName</li>
 *                 <li>elapsedTime</li>
 *                 </ul>
 */
function onAnimationIteration (element, callback)
{
  var oldCallback = DHTML.elementId2animationIterationCallback[element.id];
  if (oldCallback) {
    element.removeEventListener('animationiteration',       oldCallback, false); // (used by mozilla) 
    element.removeEventListener('oAnimationIteration',      oldCallback, false);
    element.removeEventListener('mozAnimationIteration',    oldCallback, false); // (not used by mozilla)
    element.removeEventListener('khtmlAnimationIteration',  oldCallback, false);
    element.removeEventListener('webkitAnimationIteration', oldCallback, false);
    element.removeEventListener('msAnimationIteration',     oldCallback, false);
  }
  if (callback) {
    DHTML.elementId2animationIterationCallback[element.id] = callback;
    element.addEventListener('animationiteration',       callback, false); // (used by mozilla)
    element.addEventListener('oAnimationIteration',      callback, false);
    element.addEventListener('mozAnimationIteration',    callback, false); // (not used by mozilla)
    element.addEventListener('khtmlAnimationIteration',  callback, false);
    element.addEventListener('webkitAnimationIteration', callback, false);
    element.addEventListener('msAnimationIteration',     callback, false);
  }
}

/** 
 * Add an event listener called when the animation 
 * set for the given element ends. (use this before 
 * actually setting the animation). This listener is 
 * unique, i.e. any previously listener set via this 
 * function will be removed in advance.
 * @param element  html element (div), which must have an Id(!)
 * @param callback function called after animation has finished
 *                 receiving an event parameter with the 
 *                 following special attributes:
 *                 <ul>
 *                 <li>type = animationend</li>
 *                 <li>animationName</li>
 *                 <li>elapsedTime</li>
 *                 </ul>
 */
function onAnimationEnd (element, callback)
{
  var oldCallback = DHTML.elementId2animationEndCallback[element.id];
  if (oldCallback) {
    element.removeEventListener('animationend',       oldCallback, false); // (used by mozilla) 
    element.removeEventListener('oAnimationEnd',      oldCallback, false);
    element.removeEventListener('mozAnimationEnd',    oldCallback, false); // (not used by mozilla)
    element.removeEventListener('khtmlAnimationEnd',  oldCallback, false);
    element.removeEventListener('webkitAnimationEnd', oldCallback, false);
    element.removeEventListener('msAnimationEnd',     oldCallback, false);
  }
  if (callback) {
    DHTML.elementId2animationEndCallback[element.id] = callback;
    element.addEventListener('animationend',       callback, false); // (used by mozilla)
    element.addEventListener('oAnimationEnd',      callback, false);
    element.addEventListener('mozAnimationEnd',    callback, false); // (not used by mozilla)
    element.addEventListener('khtmlAnimationEnd',  callback, false);
    element.addEventListener('webkitAnimationEnd', callback, false);
    element.addEventListener('msAnimationEnd',     callback, false);
  }
}

/**
 * Supports the given element animations? (i.e. 
 * is CSS3 animation supported by the browser)
 * @param element  html element (div)
 * @returns        true iff CSS3 animation is supported
 */
function isAnimatable (element)
{
  return (element.style.animation       !== undefined ||
          element.style.OAnimation      !== undefined ||
          element.style.MozAnimation    !== undefined ||
          element.style.KhtmlAnimation  !== undefined ||
          element.style.WebkitAnimation !== undefined ||
          element.style.msAnimation     !== undefined);
}

// ANIMATIONS =======================================================

/*
 * Helper classes for pure javascript animations. An animation here
 * is an object which has methods: 'start(fnc)' and 'stop()', where 
 * 'fnc' is a (wrapper) function which takes one parameter (the 
 * animated value). Furthermore each animation takes care that it 
 * stoppes whenever another animation on the same object (identified 
 * by some kind of Id) is started.
 */

/**
 * Request next animation frame. 
 * <dl><dt>Notice:</dt>
 * <dd>Framerate is fixed to 60 frames/sec.</dd></dl>
 * @param callback  callback function that has to call 
 *                  this function again as long as the 
 *                  animation should run
 */
function nextAnimationFrame (callback)
{
  if      (window.requestAnimationFrame)       window.requestAnimationFrame(callback); 
  else if (window.oRequestAnimationFrame)      window.oRequestAnimationFrame(callback);       
  else if (window.msRequestAnimationFrame)     window.msRequestAnimationFrame(callback); 
  else if (window.mozRequestAnimationFrame)    window.mozRequestAnimationFrame(callback); 
  else if (window.webkitRequestAnimationFrame) window.webkitRequestAnimationFrame(callback); 
  else                                         setTimeout(callback, 1000 / 60);
}

/**
 * Constructor for an animation object 
 * (pure javascript animation).
 * <dl><dt>Notice:</dt>
 * <dd>Frame-rate is fixed to 60 frames per second because we are
 *     using 'requestAnimationFrame' (see 'nextAnimationFrame()') 
 *     if available, where the frame-rate is fixed anyway to that
 *     value.</dd></dl>
 * @param id            id; this animation is stopped if there is
 *                      another animation started with the same id
 * @param duration      duration of this animation
 * @param delay         delay before actualls starting the animation
 * @param fnc           transform function object with the function
 *                      transformedValue = fnc.get(relativeTime)
 * @param data          data to be given to the callback (see 'start()') 
 */
function Animation (id, duration, delay, fnc, data)
{
  if (DHTML.animations[id]) DHTML.animations[id].stop();
  DHTML.animations[id] = this;

  this.id       = id;
  this.duration = duration;
  this.delay    = delay;
  this.fnc      = fnc;
  this.data     = data;
  this.t0       = -1;     // starting time (t0<0 means not started/stopped)
  this.t        = 0;      // time since start
}

/* Inheritance */
//Animation.prototype          = 
//Animation.prototype.parent   = 
Animation.prototype.constructor = Animation;

/**
 * Start this animation.
 * @param callback  function to be called at a frame-rate of 
 *                  approx. 60 frames/second with three parameters:
 *                  the transformed value, the relative time and a
 *                  data object (see 'Animation()')
 */
Animation.prototype.start = function (callback) 
{
  var me   = this;
  var anim = function () {
    if (me.t0 < 0) return;                   // stopped (see 'stop()')
    me.t = (new Date()).getTime() - me.t0;   // time since start
    if (me.t  < me.duration) {               // request next animation frame...
      nextAnimationFrame(anim);              // ...as soon as possible, for ...
    } else {                                 // ...getting close to 60 fr/sec
      me.t0 = -1;                            // finished (normally)
    }                                          
    var r = Math.min(1, me.t / me.duration); // relative time
    var v = me.fnc.get(r);                   // transformed value
    callback(v, r, me.data);
  };
  var init = function () {
    me.t0 = (new Date()).getTime();          // starting time
    anim();                                  // start animation
  };
  setTimeout(init, this.delay);
};

/**
 * Stop this animation
 */
Animation.prototype.stop = function ()
{
  this.t0 = -1; 
};

// TRANSFORM-FUNCTIONS FOR ANIMATIONS -------------------------------

/**
 * Linear function object
 */
function Linear () {}
Linear.prototype.get = function (r) 
{ 
  return (r); 
};

/**
 * Ease function object (slow - fast - slow)
 */
function Ease () {}
Ease.prototype.get = function (r) 
{ 
  return ((Math.cos(r * Math.PI) - 1) / -2); 
};

/**
 * Ease-in function object (slow - fast)
 */
function EaseIn () {}
EaseIn.prototype.get = function (r) 
{ 
  return (1 - Math.cos(r * Math.PI/2)); 
};

/**
 * Ease-out function object (fast - slow)
 */
function EaseOut () {}
EaseOut.prototype.get = function (r) 
{ 
  return (Math.sin(r * Math.PI/2)); 
};

/**
 * Cubic bezier curve from p0 = (0,0) to p3 = (1,1).
 * <dl><dt>Notice:</dt>
 * <dd>This is <b>not</b> the same used by CSS3-aniations!
 *     CSS3 bezier timing function calculates the y for 
 *     a given x, while this gives the y for a given t.
 * </dd></dl>
 * @param x1 x-coordinate of the first intermed. point
 * @param y1 y-coordinate of the first intermed. point
 * @param x2 x-coordinate of the second intermed. point
 * @param y2 y-coordinate of the second intermed. point
 */
function Bezier (x1, y1, x2, y2)
{
  this.x3 =  0; this.y3 =  0;
  this.x2 = x1; this.y2 = y1;
  this.x1 = x2; this.y1 = y2;
  this.x0 =  1; this.y0 =  1;
}
Bezier.prototype.get = function (t)  
{
  var b0 = t*t*t;
  var b1 = 3*t*t*(1-t);
  var b2 = 3*t*(1-t)*(1-t);
  var b3 = (1-t)*(1-t)*(1-t);
  //var x  = this.x0*b0 + this.x1*b1 + this.x2*b2 + this.x3*b3; // (not used here)
  var y  = this.y0*b0 + this.y1*b1 + this.y2*b2 + this.y3*b3;
  return (y);
};
  

// PROCESSES ========================================================

/**
 * Constructor for an process object. A process means a long-running
 * process which is done by succeeding calls to a callback function.
 * @param id     id; this process is stopped if there is
 *               another process started with the same id
 * @param delay  delay before actualls starting the animation
 * @param rate   delay between two succeeding calls ("frame-rate")
 * @param data   data to be given to the callback (see 'start()') 
 */
function Process (id, delay, rate, data)
{
  if (DHTML.processes[id]) DHTML.processes[id].stop();
  DHTML.processes[id] = this;

  this.id       = id;
  this.delay    = delay;
  this.rate     = rate;
  this.data     = data;
  this.running  = false; // is this process running? 
}

/* Inheritance */
//Process.prototype          = 
//Process.prototype.parent   = 
Process.prototype.constructor = Process;

/**
 * Start this process.
 * @param callback  function to be called at a rate 
 *                  of approx. 'rate' milliseconds, 
 *                  which returns 'false' if this 
 *                  process should stop
 */
Process.prototype.start = function (callback) 
{
  this.running = true;
  var me   = this;
  var proc = function () {
    if (!me.running) return;
    setTimeout(proc, this.rate);  
    if (!callback(me.data)) me.running = false; // stop
  };
  setTimeout(proc, this.delay);
};

/**
 * Stop this process
 */
Process.prototype.stop = function ()
{
  this.running = false; 
};
