/*
 * -------------------------------------------------------------------------
 *
 *  dhtml.js Copyright (C) 2011, 
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *   Author: Patrick Rammelt 
 *   E-Mail: patrick.rammelt@go4more.de
 *   Site:   http://www.patricks-seite.de
 * -------------------------------------------------------------------------
 *
 * Helper functions for dynamic html. Some of these methods are used to 
 * avoid some problems with outdated browsers (e.g. IE6). Nevertheless
 * not every old browser is supported (it has to have basic DOM 
 * functionality).
 */

/* 
 * Main: check
 */
//if (!document.getElementById) {
//  alert("Your browser seems to be very old (it doesn't support DOM), so some script won't work (at least not correctly), sorry...");
//}


// MATH ====================================================================

/*
 * Converts numeric (float) value to integer (cuts off decimals)
 */
function toInt (v) 
{
  if (Math.abs(v) < 1) return (0);  // notice: parseInt("3.6e-10") = 3
  else return (parseInt(v));
}


// STRING EXTENSIONS =======================================================

/* 
 * String repeat 
 */
String.prototype.repeat = function (times)
{
  return (new Array(times + 1).join(this));
};

/* 
 * Converts text to html (e.g.  -> &auml;) 
 */
String.prototype.toHTML = function ()
{
  var text = this;
  text = text.replace(/\&/g, "&#038;"); // (/g => replace all)
  text = text.replace(/</g, "&#060;");
  text = text.replace(/>/g, "&#062;");
  text = text.replace(//g, "&auml;");
  text = text.replace(//g, "&ouml;");
  text = text.replace(//g, "&uuml;");
  text = text.replace(//g, "&Auml;");
  text = text.replace(//g, "&Ouml;");
  text = text.replace(//g, "&Uuml;");
  text = text.replace(//g, "&szlig;");
  text = text.replace(/^---+\n/g,  "<hr width=\"100%\" size=\"2\">\n"); // horizontal line (start)
  text = text.replace(/\n---+\n/g, "<hr width=\"100%\" size=\"2\">\n"); // horizontal line
  text = text.replace(/\n/g, "<br>");
  return (text);
};


// CREATE AND SET ELEMENTS =================================================
  
/*
 * Create a new element.
 * tag:       html element type
 * id:        html element id
 * clazz:     html element class
 * invisible: should the element being set to be invisible?
 * return:    newly created element
 */
function createNewElement (tag, id, clazz, invisible)
{
  var elem = document.createElement(tag);
  if (id    != undefined) elem.id        = id;
  if (clazz != undefined) elem.className = clazz;
  if (invisible)   elem.style.visibility = 'hidden';
  return (elem);
}

/*
 * Create a new element and append it to the given parent.
 * parent:    parent element
 * tag:       html element type
 * id:        html element id
 * clazz:     html element class
 * invisible: should the element being set to be invisible?
 * return:    newly created element (already appended to parent)
 */
function appendNewElement (parent, tag, id, clazz, invisible)
{
  var elem = createNewElement(tag, id, clazz, invisible);
  parent.appendChild(elem);
  return (elem);
}


// REMOVE ELEMENTS =========================================================

/* 
 * Remove the element with the given id 
 */
function removeElement (id) 
{
  var elem = document.getElementById(id);
  if (elem) elem.parentNode.removeChild(elem);
}


// GET ELEMENTS ============================================================
  
/*
 * Get elements by tag name (only those elements having a valid ID).
 *
 * Notice: 
 *   This function is useful only because there seems to be a bug  
 *   in SeaMonkey such that there could be multiple instances for 
 *   the same element if retrieved by: 'getElementsByTagName()'.
 *   Such duplicates are often incomplete representations of the 
 *   "real" elements (e.g. missing sub-nodes).
 *   Therfor we use elements retrived by 'getElementsByTagName()'
 *   only to get a unique set of IDs and retrieve the "real" set 
 *   of elements by 'document.getElementById()' for each such ID
 *   (i.e. the ID is used for two reasons: uniqueness and getting
 *   "real" elements).
 *
 * root:     root element (search below this element)
 * tagName:  tag name
 * return:   an array of found elements
 */
function getIdElementsByTagName (root, tagName)
{
  // final array of elements (unique)
  var unique = new Array();

  // get elements (might contain incomplete duplicates of "real" elements)
  var elements = root.getElementsByTagName(tagName);

  // fill final array of unique elements
  var ids = {};              // "associative map" of IDs to recognize duplicates
  for (i in elements) { 
    var id = elements[i].id;
    if (id && !ids[id]) {    // id exists and it is the first time we see it
      ids[id] = true; 

      // due to the bug in SeaMonkey, we do not use elements[i] 
      // directly, because this element might be wrong/incomplete
      // instead we get the "real" element by its ID
      var element = document.getElementById(id);
      unique[unique.length] = element;
    }
  }

  // return unique array of elements
  return (unique);
};


// GET COORDINATES =========================================================


/* Get left x coordinate relative to the document */
function getLeft (element)
{
  var x = 0;
  while (element != null) {
    x += element.offsetLeft;
    element = element.offsetParent;
  }
  return (x);
}

/* Get top y cordinate relative to the document */
function getTop (element)
{
  var y = 0;
  while (element != null) {
    y += element.offsetTop;
    element = element.offsetParent;
  }
  return (y);
}


// GET SIZE ================================================================

/* 
 * Get width of an element 
 * (including scrolled, i.e. invisible parts) 
 */
function getWidth (element) 
{ 
  return (Math.max(element.offsetWidth,  element.scrollWidth)); 
}

/* 
 * Get height of an element 
 * (including scrolled, i.e. invisible parts) 
 */
function getHeight (element) 
{ 
  return (Math.max(element.offsetHeight,  element.scrollHeight)); 
}


// SCROLL OFFSETS ==========================================================

/* 
 * Get top-most visible y coordinate 
 */
function getVScrollOffset () 
{
  if (typeof window.pageYOffset == "number") {
    return (window.pageYOffset);
  } else if (typeof document.body.scrollTop == "number") {
    return (document.body.scrollTop); // IE6
  } else {
    return (0); // (we don't know)
  }
}

/* 
 * Get left-most visible x coordinate 
 */
function getHScrollOffset () 
{
  if (typeof window.pageXOffset == "number") {
    return (window.pageXOffset);
  } else if (typeof document.body.scrollLeft == "number") {
    return (document.body.scrollLeft); // IE6
  } else {
    return (0); // (we don't know)
  }
}


// EVENTS ==================================================================

/*
 * Add an event listener to an element.
 * element:  html element
 * type:     event type name (e.g. "mousemove")
 * listener: function with one parameter (event)
 */
function addEventListener (element, type, listener)
{
  if (element.addEventListener) {
    element.addEventListener(type, listener, true);
  } else if (element.attachEvent) {
    element.attachEvent('on' + type, listener);
  }
}

/*
 * Remove an event listener to an element.
 * element:  html element
 * type:     event type name (e.g. "mousemove")
 * listener: function with one parameter (event)
 */
function removeEventListener (element, type, listener)
{
  if (element.removeEventListener) {
    element.removeEventListener(type, listener, true);
  } else if (element.detachEvent) {
    element.attachEvent('on' + type, listener);
  }
}

/* 
 * Absolute mouse position of the given event 
 */
function pageX (evnt) 
{
  return (evnt.pageX ? evnt.pageX : evnt.clientX + getHScrollOffset());
}

/* 
 * Absolute mouse position of the given event 
 */
function pageY (evnt) 
{
  return (evnt.pageY ? evnt.pageY : evnt.clientY + getVScrollOffset());
}

/* 
 * Cancel default behavior for the given event 
 */
function preventDefault (evnt)
{
  if      (evnt.preventDefault) evnt.preventDefault(); 
  else if (evnt.returnValue)    evnt.returnValue = false;
}

/* 
 * Get the element that fired the given event 
 */
function target (evnt) 
{
  return (evnt.target ? evnt.target : evnt.srcElement);
}


// WAIT FOR ================================================================

/*
 * Waits until a function has succeeded
 * func:     function to be called (gets one parameter: number of 
 *           trials left and has to return true in case of success
 *           or false if waiting should continue for another timeout)
 * timeout:  timeout before first and every succeeding trial
 * trials:   max. number of trials
 */
function waitFor (func, timeout, trials)
{
  setTimeout(function() {
	       if (!func(trials) && trials > 0) {
		 waitFor(func, timeout, trials-1);  // still waiting
	       } 
	     }, timeout);
}


// DUMMY FUNCTION ==========================================================

/* 
 * Function that doesn't do anything...
 */
function dummy () {}

/* 
 * Function that doesn't do anything...
 */
function dummyEventHandler (evnt) {}
