/*
 * -------------------------------------------------------------------------
 *
 *  galery.js Copyright (C) 2011, 
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *   Author: Patrick Rammelt 
 *   E-Mail: patrick.rammelt@go4more.de
 *   Site:   http://www.patricks-seite.de
 * -------------------------------------------------------------------------
 *
 * For each link to an image (*.jpg, *.png, *.gif, *.tif) this module 
 * modifies those links such that a popup-window is opened instead, which
 * shows these images (if javascript is disabled the link stays unaffacted, 
 * i.e. images will be opened directly by the browser)
 *
 * To make this work a site has to start like this:
 * <html>
 * <head>
 *   [...meta tags, etc...]
 *   <script type="text/javascript" src="../js/dhtml.js"></script>
 *   <script type="text/javascript" src="../js/galery.js"></script>
 *   <style type="text/css">
 *     body                         { background-color:rgb(0, 0, 102); 
 *                                    background-image:url(bg.jpg); 
 *                                    color:rgb(0, 51, 0); 
 *                                    margin:0; padding:25px 10px; }
 *     div.galeryPopup              { position:absolute; top:20px; left:0;
 *                                    background-image:url(../js/bg.png);
 *                                    color:#888888; 
 *                                    text-decoration:none;
 *                                    font-weight:bold; font-size:12px; 
 *                                    font-family:arial,helvetica; }
 *     table.galeryFrame            { border-width:10px; border-style:solid; 
 *                                    border-top-color:rgb(150,150,150);
 *                                    border-left-color:rgb(128,128,128);	
 *                                    border-bottom-color:rgb(64,64,64);
 *                                    border-right-color:rgb(96,96,96);
 *                                    border-spacing: 0px; } 
 *     table.galeryFrame td         { padding-top:0px;  padding-bottom:0px; 
 *                                    padding-left:0px; padding-right:0px; } 
 *     div.galeryButtons            { position:absolute; top:12px; right:12; }
 *     div.galeryZoom               { position:absolute; top:12px; left:12;  }
 *     table.galeryPosition         { position:absolute; bottom:2px; right:4;
 *                                    padding:0px; border-spacing:1px; 
 *                                    table-layout:fixed; height:6px; }
 *     table.galeryPosition td      { padding:0px; padding-bottom:0px; 
 *                                    padding-left:0px; padding-right:0px; } 
 *     table.galeryPosition td.pre  { width:2px; background-color:rgb(96,96,96); }
 *     table.galeryPosition td.curr { width:2px; background-color:rgb(150,150,150); }
 *     table.galeryPosition td.post { width:2px; background-color:rgb(96,96,96); }
 *     div.galeryNavigate           { position:absolute; top:12px; left:12; }
 *     div.galeryDetail             { color:#555555; text-decoration:none; font-weight:bold;
 *                                    font-size:12px; font-family:arial,helvetica; padding:5px; }
 *     div.galeryPreview            { position:absolute; visibility:hidden; 
 *                                    background-image:url(preview.png); }
 *     div.detail                   { position:absolute; top:0px; left:0; visibility:hidden; }
 *   </style>
 * </head>
 * <body onload="initGalery('Galery1', 1, 20)" vlink="#003300" alink="#ff0000" link="#003300">
 * <div id="GaleryPopup1" class="galeryPopup">Notice: Enable JavaScript to use Popup-view.</div>
 * <div id="GaleryPopup2" class="galeryPopup">Notice: Enable JavaScript to use Popup-view.</div>
 * <div id="GaleryPreview" class="galeryPreview">Preview-Image</div>
 * <div id="Galery1">
 * [...]
 * <td>                                  // just a sourrounding elemenet which is allowed to contain divs
 *   <a id="image1" href="image1.jpg">   // <a> needs to have an ID (just something unique)
 *     <img src="image1_small.jpg">
 *   </a>
 *   <div class="detail">                // first (even invisible) <div> in sourrounding element (e.g. <td>)
 *       This text will be displayed in conjunction with image1 by the popup window
 *   </div>
 * </td>
 * [...]
 *
 * At least the call to initGalery(subLevls) has to be adapted for each 
 * site (subLevels is the number of directories you have to go down from 
 * this site to get to the base directory). Galery*1 and Galery*2 should 
 * define the same style. They are used alternately for succeeding images: 
 * While the new popup is build (not yet visible) the previous image is 
 * still shown.
 * You also need progress.js - see there for additional requirements.
 */

/* Container for all galery parameters (constants and variables) */
var GAL = {};

/* Index (toggling between 0 and 1 for two popup windows) */
GAL.popupIndex = 0;

/* Zoom */
GAL.zoom = 1;          // Zoom factor (negative means full size)
GAL.ZOOM_OUTBY = 0.75; // zoom out factor (zoom in is 1/GAL.ZOOM_OUTBY)

/* IDs of html elements (div) for two popups (previously shown, next shown) */
GAL.galeryID       = null;                                            // current galery
GAL.POPUP_IDS      = new Array("GaleryPopup1",    "GaleryPopup2");    // popup window
GAL.FRAME_IDS      = new Array("GaleryFrame1",    "GaleryFrame2");    // frame inside the popup window
GAL.IMAGE_IDS      = new Array("GaleryImage1",    "GaleryImage2");    // image inside the popup window
GAL.BUTTONS_IDS    = new Array("GaleryButtons1",  "GaleryButtons2");  // close/resize button inside the popup window
GAL.NAVIGATE_IDS   = new Array("GaleryNavigate1", "GaleryNavigate2"); // navigation area inside the popup window
GAL.ZOOM_IDS       = new Array("GaleryZoom1",     "GaleryZoom2");     // zoom area inside the popup window
GAL.DETAIL_IDS     = new Array("GaleryDetail1",   "GaleryDetail2");   // detail area inside the popup window
GAL.POSITION_IDS   = new Array("GaleryPosition1", "GaleryPosition2"); // shows which image of the galery is currently shown
GAL.PREVIEW_ID     = "GaleryPreview";                                 // preview div if mouse in over the position bar
GAL.PREVIEW_IMG_ID = "GaleryPreviewImg";                              // preview thumbnail image inside preview div

/* Classes */
GAL.POPUP_CLASS    = "galeryPopup";    // extends div
GAL.FRAME_CLASS    = "galeryFrame";    // extends table
GAL.IMAGE_CLASS    = "galeryImage";    // extends div
GAL.BUTTONS_CLASS  = "galeryButtons";  // extends div
GAL.NAVIGATE_CLASS = "galeryNavigate"; // extends div
GAL.ZOOM_CLASS     = "galeryZoom";     // extends div
GAL.DETAIL_CLASS   = "galeryDetail";   // extends div
GAL.POSITION_CLASS = "galeryPosition"; // extends table
GAL.POS_PRE_CLASS  = "pre";            // extends td in position table
GAL.POS_CURR_CLASS = "curr";           // extends td in position table
GAL.POS_POST_CLASS = "post";           // extends td in position table

/* Images / Icons */
GAL.FIRST_IMG    = "js/first.png";         // first-image button inside the popup window
GAL.LAST_IMG     = "js/last.png";          // last-image button inside the popup window
GAL.ISFIRST_IMG  = "js/isfirst.png";       // disabled first button inside the popup window
GAL.ISLAST_IMG   = "js/islast.png";        // disbaled last button inside the popup window
GAL.PREV_IMG     = "js/prev.png";          // previous-image button inside the popup window
GAL.NEXT_IMG     = "js/next.png";          // next-image button inside the popup window
GAL.NOPREV_IMG   = "js/noprev.png";        // disabled previous button inside the popup window
GAL.NONEXT_IMG   = "js/nonext.png";        // disabled next button inside the popup window
GAL.CLOSE_IMG    = "js/closepopup.png";    // close button inside the popup window
GAL.MAX_IMG      = "js/maximizepopup.png"; // full size button inside the popup window
GAL.MIN_IMG      = "js/minimizepopup.png"; // fit to page button inside the popup window
GAL.RECENTER_IMG = "js/recenter.png";      // re-center image button inside the popup window
GAL.ZOOMOUT_IMG  = "js/zoomout.png";       // zoom out button inside the popup window
GAL.ZOOMIN_IMG   = "js/zoomin.png";        // zoom in button inside the popup window
GAL.ZOOM100_IMG  = "js/zoom100.png";       // zoom back to 100% button inside the popup window

/* Tool-tips */
GAL.FIRST_TOOLTIP    = "Erstes Bild";
GAL.PREV_TOOLTIP     = "Voriges Bild";
GAL.NEXT_TOOLTIP     = "N&auml;chstes Bild";
GAL.LAST_TOOLTIP     = "Letztes Bild";
GAL.ZOOMIN_TOOLTIP   = "Gr&ouml;&szlig;er";
GAL.ZOOMOUT_TOOLTIP  = "Kleiner";
GAL.ZOOM100_TOOLTIP  = "100%";
GAL.CLOSE_TOOLTIP    = "Fenster schlie&szlig;en";
GAL.FIT_TOOLTIP      = "Angepasste Gr&ouml;&szlig;e (Fenster / Zoom)";
GAL.FULL_TOOLTIP     = "Volle (echte) Bildgr&ouml;&szlig;e";
GAL.RECENTER_TOOLTIP = "Wieder zentral ausrichten (nach Verschiebung\ndurch 'Drag-and-Drop' auf dem Bild)";
GAL.POSITION_TOOLTIP = "Click f&uuml;r direkten Sprung";

/* Dragging */
GAL.dragX0 = 0;       // start x-coordinate of dragging
GAL.dragY0 = 0;       // start y-coordinate of dragging
GAL.dragXOffset = -1; // offset grab position to left border
GAL.dragYOffset = -1; // offset grab position to top border
GAL.popupXOffset = 0; // offset created by dragging
GAL.popupYOffset = 0; // offset created by dragging

/* 
 * Images, thumbnails, details and other galery dependant parameters for all 
 * managed galeries - e.g. if initGalery was called for id=Galery1, then 
 * - GAL.galeries.Galery1.images           is a list of image source names, 
 * - GAL.galeries.Galery1.thumbs           is a list of thumbnail images to be
 *                                         shown in a little preview window, 
 * - GAL.galeries.Galery1.details          is a list of html detail descriptions
 *                                         to be shown below the image) 
 * - GAL.galeries.Galery1.verticalBorders  is the cumulated height of top and
 *                                         bottom borders
 * - GAL.galeries.Galery1.fixDetailSize    says whether the size of the detail
 *                                         area should be fixed before the image
 *                                         is downsized such that the detail fits
 *                                         into the window (this could lead to
 *                                         additional horizontal borders, but 
 *                                         ensures that the popup  fits on the 
 *                                         window even if the detail area is large
 */                                       
GAL.galeries = {};

/* 
 * Path to the root directory 
 * (set by 'initGalery()' - has to be the same for all galeries on one site) 
 */
GAL.BASE_PATH = "";

/* Minimum image height */
GAL.MIN_IMG_HEIGHT = 100;

/* 
 * Offsets to adjust the preview image to a given point 
 * (we cannot find out this automatically because the background image might 
 * not be symmetrical, e.g. due to a shadow an the right and bottom side...) 
 */
GAL.PREVIEW_XOFFSET    = -49;
GAL.PREVIEW_YOFFSET    = -111;
GAL.PREVIEW_MAXIMGSIZE = 90;   // width or height

/*
 * Used by running "processes" (functions) started via 'setTimeout()' 
 * or 'dhtml:waitFor()' (as created by 'showImg()'). Before a new 
 * function is started any for any existing object in 'GAL.processCtrl'
 * 'Gal.processCtrl.cancel' has to be set to 'true'.
 */
GAL.processCtrl = null;


/* Set zoom (1 means fit to page, negative means full size) */
function setZoom (zoom) { GAL.zoom = zoom; }


/* Set galery ID */
function setGalery (galeryID) { GAL.galeryID = galeryID; }


/* 
 * Show image with index in popup window 
 * index:    index of the image to be shown
 * zoom:     zoom factor (see setZoom())        [optional]
 * galeryID: ID of the galery (see setGalery()) [optional]
 * xoffset:  x-offset of the popup-window       [optional]
 * yoffset:  y-offset of the popup-window       [optional]
 */
function showImg (index, zoom, galeryID, xoffset, yoffset) 
{ 
  if (typeof zoom     == "number") GAL.zoom         = zoom;     else zoom     = GAL.zoom;
  if (typeof galeryID == "string") GAL.galeryID     = galeryID; else galeryID = GAL.galeryID;
  if (typeof xoffset  == "number") GAL.popupXOffset = xoffset;  else xoffset  = GAL.popupXOffset;
  if (typeof yoffset  == "number") GAL.popupYOffset = yoffset;  else yoffset  = GAL.popupYOffset;

  var popupIndex = (GAL.popupIndex + 1) % 2; // next index
  var popup = document.getElementById(GAL.POPUP_IDS[popupIndex]);

  var top    = getVScrollOffset();         // top-most visible y-coordinate
  var height = document.body.clientHeight; // window height (without scroolbars)

  popup.style.top = Math.max(0, top + 25 + yoffset);

  var galery    = GAL.galeries[galeryID];
  var images    = galery.images;
  var details   = galery.details;
  var image     = images[index];
  var detail    = details[index];
  var imgHeight = parseInt((height - 50 - galery.verticalBorders) * zoom);
  if (imgHeight < GAL.MIN_IMG_HEIGHT) imgHeight = GAL.MIN_IMG_HEIGHT;
  div 
    // frame, image and detail area
    = "<table id=\""    + GAL.FRAME_IDS[popupIndex] + "\" "
    +        "class=\"" + GAL.FRAME_CLASS           + "\" "
    +        "style=\"text-align: center; width: 1px; height: 1px;\">"
    +   "<tbody>"
    +     "<tr>"
    +       "<td style=\"vertical-align: middle;\">"
    + (zoom < 0 
       ?       "<img id=\""    + GAL.IMAGE_IDS[popupIndex] + "\" " 
       +            "class=\"" + GAL.IMAGE_CLASS           + "\" "
       +            "alt=\"\" src=\"" + image + "\" " +
       +            "style=\"border: 0px solid;\">" 
       :       "<img id=\""    + GAL.IMAGE_IDS[popupIndex] + "\" "
       +            "class=\"" + GAL.IMAGE_CLASS           + "\" "
       +            "alt=\"\" src=\"" + image + "\" "
       +            "style=\"border: 0px solid;\" height=" + imgHeight + "px>")
    + (!detail ? "" 
       : "<br><div id=\""     + GAL.DETAIL_IDS[popupIndex] + "\" " 
       +           "class=\"" + GAL.DETAIL_CLASS           + "\">"
       +    detail 
       + "</div>")
    +       "</td>"
    +     "</tr>"
    +   "</tbody>"
    + "</table>"

    // zoom area
    + "<div id=\""     + GAL.ZOOM_IDS[popupIndex] + "\" " 
    +       "class=\"" + GAL.ZOOM_CLASS           + "\">" 
    + (zoom < 0 ? ""
       :  "<a href=\"JavaScript:showImg(" + index + ", " + (zoom*GAL.ZOOM_OUTBY) + ")\">"
       +     "<img alt=\"(-)\" title=\"" + GAL.ZOOMOUT_TOOLTIP + "\" "
       +          "src=\"" + GAL.BASE_PATH + GAL.ZOOMOUT_IMG + "\" "
       +          "style=\"border: 0px solid; width:20px; height:20px\">"
       +   "</a>"
       +   "&nbsp;"
       +   "<a href=\"JavaScript:showImg(" + index + ", " + (zoom*(1.0/GAL.ZOOM_OUTBY)) + ")\">"
       +     "<img alt=\"(+)\" title=\"" + GAL.ZOOMIN_TOOLTIP + "\" "
       +          "src=\"" + GAL.BASE_PATH + GAL.ZOOMIN_IMG + "\" "
       +          "style=\"border: 0px solid; width:20px; height:20px\">"
       +   "</a>"
       +   "&nbsp;"
       +   "<a href=\"JavaScript:showImg(" + index + ", " + (zoom=1) + ")\">"
       +     "<img alt=\"(100%)\" title=\"" + GAL.ZOOM100_TOOLTIP + "\" "
       +          "src=\"" + GAL.BASE_PATH + GAL.ZOOM100_IMG + "\" "
       +          "style=\"border: 0px solid; width:20px; height:20px\">"
       +   "</a>"
       +   "&nbsp;")
    + "</div>"

    // buttons (fit-to-page/full-size, close)
    + "<div id=\""     + GAL.BUTTONS_IDS[popupIndex] + "\" " 
    +       "class=\"" + GAL.BUTTONS_CLASS           + "\">" 
    +   "<a href=\"JavaScript:showImg(" + index + ",null,null,0,0)\">"
    +     "<img alt=\"RECENTER\" title=\"" + GAL.RECENTER_TOOLTIP + "\" src=\"" + GAL.BASE_PATH + GAL.RECENTER_IMG + "\" "
    +       "style=\"border: 0px solid; width:20px; height:20px\">"
    +   "</a>"
    +   "&nbsp;"
    +   "<a href=\"JavaScript:showImg(" + index + ", " + (-zoom) + ")\">"
    +     "<img alt=\""   + (zoom < 0 ? "FIT"           : "FULL"          ) + "\" "
    +          "title=\"" + (zoom < 0 ? GAL.FIT_TOOLTIP : GAL.FULL_TOOLTIP) + "\" "
    +          "src=\"" + GAL.BASE_PATH + (zoom < 0 ? GAL.MIN_IMG : GAL.MAX_IMG) + "\" "
    +          "style=\"border: 0px solid; width:20px; height:20px\">"
    +   "</a>"
    +   "&nbsp;"
    +   "<a href=\"JavaScript:hideImg()\">"
    +     "<img alt=\"CLOSE\" title=\"" + GAL.CLOSE_TOOLTIP + "\" src=\"" + GAL.BASE_PATH + GAL.CLOSE_IMG + "\" "
    +       "style=\"border: 0px solid; width:20px; height:20px\">"
    +   "</a>"

    // position "slider"
    + "</div>"    
    +   "<table id=\""    + GAL.POSITION_IDS[popupIndex] + "\" "
    +          "class=\"" + GAL.POSITION_CLASS           + "\"" 
    +          "title=\"" + GAL.POSITION_TOOLTIP         + "\">"
    +     "<tbody>"
    +       "<tr>"
    +         ("<td class=\"" + GAL.POS_PRE_CLASS  + "\"></td>").repeat(index)
    +         ("<td class=\"" + GAL.POS_CURR_CLASS + "\"></td>")
    +         ("<td class=\"" + GAL.POS_POST_CLASS + "\"></td>").repeat(images.length-index-1)
    +       "</tr>"
    +     "</tbody>"
    +   "</table>"

    // navigation area
    + "<div id=\""    + GAL.NAVIGATE_IDS[popupIndex] + "\" " 
    +      "class=\"" + GAL.NAVIGATE_CLASS           + "\">" 
    + (index <= 0 
       ? "<img alt=\"&lt;&lt;-\" title=\"\" "
       +       "src=\"" + GAL.BASE_PATH + GAL.ISFIRST_IMG + "\" "
       +       "style=\"border: 0px solid; width:20px; height:20px\">" 
       + "&nbsp;"
       + "<img alt=\"&lt;-\" title=\"\" "
       +       "src=\"" + GAL.BASE_PATH + GAL.NOPREV_IMG + "\" "
       +       "style=\"border: 0px solid; width:20px; height:20px\">" 
       : "<a href=\"JavaScript:showImg(" + 0 + ")\">"
       +   "<img alt=\"&lt;&lt;-\" title=\"" + GAL.FIRST_TOOLTIP + "\" "
       +        "src=\"" + GAL.BASE_PATH + GAL.FIRST_IMG + "\" "
       +        "style=\"border: 0px solid; width:20px; height:20px\">"
       + "</a>"
       + "&nbsp;"
       + "<a href=\"JavaScript:showImg(" + (index-1) + ")\">"
       +   "<img alt=\"&lt;-\" title=\"" + GAL.PREV_TOOLTIP + "\" "
       +        "src=\"" + GAL.BASE_PATH + GAL.PREV_IMG + "\" "
       +        "style=\"border: 0px solid; width:20px; height:20px\">"
       + "</a>")
    + "&nbsp;"
    + (index >= images.length-1 
       ? "<img alt=\"&lt;\" title=\"\" "
       +      "src=\"" + GAL.BASE_PATH + GAL.NONEXT_IMG + "\" "
       +      "style=\"border: 0px solid; width:20px; height:20px\">" 
       + "&nbsp;"
       + "<img alt=\"&lt;&gt;\" title=\"\" "
       +      "src=\"" + GAL.BASE_PATH + GAL.ISLAST_IMG + "\" "
       +      "style=\"border: 0px solid; width:20px; height:20px\">" 
       : "<a href=\"JavaScript:showImg(" + (index+1) + ")\">"
       +   "<img alt=\"-&gt;\" title=\"" + GAL.NEXT_TOOLTIP + "\" "
       +        "src=\"" + GAL.BASE_PATH + GAL.NEXT_IMG + "\" "
       +        "style=\"border: 0px solid; width:20px; height:20px\">"
       + "</a>"
       + "&nbsp;"
       + "<a href=\"JavaScript:showImg(" + (images.length-1) + ")\">"
       +   "<img alt=\"-&gt;&gt;\" title=\"" + GAL.LAST_TOOLTIP + "\" "
       +        "src=\"" + GAL.BASE_PATH + GAL.LAST_IMG + "\" "
       +        "style=\"border: 0px solid; width:20px; height:20px\">"
       + "</a>") 
    + "</div>";  
  
  popup.innerHTML = div;
  var nav = document.getElementById(GAL.NAVIGATE_IDS[popupIndex]);
  var pos = document.getElementById(GAL.POSITION_IDS[popupIndex]);
  var det = document.getElementById(  GAL.DETAIL_IDS[popupIndex]);
  var img = document.getElementById(   GAL.IMAGE_IDS[popupIndex]);    
  var frm = document.getElementById(   GAL.FRAME_IDS[popupIndex]);

  // max. number of trials
  var maxTrials = 200;

  // stop any running "process" and create a new process control object
  if (GAL.processCtrl != null) GAL.processCtrl = true;
  var processCtrl = {};
  processCtrl.cancel = false;
  GAL.proessCtrl = processCtrl;
  
  // function to adjust horizontal alignment and set visible
  var alignPopup = 
    function(trials) 
    { 
      showProgressBar();

      // check (Safari: sets the hight before the width is known, 
      // therfor height is already correct while width is still 0)
      if (processCtrl.cancel) {
	return (true);
      } else if (trials > 0 && (img.offsetWidth <= 0 || (zoom < 0 ? img.offsetHeight <= 0 : img.offsetHeight != imgHeight))) {
	return (false);
      } else {

	// readjust popup position and navigation buttons (centered)
	var left  = getHScrollOffset();
	var width = document.body.clientWidth;  // window width (without scrollbars);
	var x1 = (width - popup.offsetWidth) / 2 + left + xoffset;
	var x2 = (popup.offsetWidth - nav.offsetWidth) / 2;
	var x3 = (popup.offsetWidth - pos.offsetWidth) / 2;
	popup.style.left = Math.max(0, x1);
	nav.style.left   = x2;
	pos.style.left   = x3;
	hideImg();
	GAL.popupIndex = popupIndex; // (after hideImg()!)
	popup.style.visibility = "visible";
	hideProgressBar();
	return (true);

      }
    };
  
  // function to adjust image height if necessary and call adjustHorizontal
  var adjustPopup =
    function(trials) 
    { 
      showProgressBar();

      // check (Safari: sets the hight before the width is known, 
      // therfor height is already correct while width is still 0)
      if (processCtrl.cancel) {
	return (true);
      } else if (trials > 0 && (img.offsetWidth <= 0 || (zoom < 0 ? img.offsetHeight <= 0 : img.offsetHeight != imgHeight))) {
	return (false);
      } else {

	if (det && zoom > 0) {
	  if (galery.fixDetailSize) {                        // fix detail size (introduces left/right image-borders)
	    det.style.width  = det.offsetWidth;              // fix width
	    det.style.height = det.offsetHeight;             // fix height
	    imgHeight = img.offsetHeight - det.offsetHeight; // used in check
	  } else {                                           // approximate adaption of image height
	    var h  = imgHeight;                              // total height
	    var d  = det.offsetWidth * det.offsetHeight;     // detail area size
	    var f  = img.offsetWidth / img.offsetHeight;     // relation image height<->width
	    var sq = (h*h/4.0) - (d/f);
	    imgHeight = (sq > 0 ? parseInt(h/2 + Math.sqrt(sq)) : img.offsetHeight - det.offsetHeight);
	  }	  
	  if (imgHeight < GAL.MIN_IMG_HEIGHT) imgHeight = GAL.MIN_IMG_HEIGHT;
	  img.height = imgHeight;
	  waitFor(alignPopup, 50, trials);
	} else {
	  alignPopup(0);
	}
	return (true);

      }
    }; 

  // make adjustments
  waitFor(adjustPopup, 50, maxTrials);

  // show a preview image (thumbnail) if the mouse is over the position 
  // bar / directly move to that image if the position bar is clicked
  addEventListener(pos, "mouseover", showPreview);
  addEventListener(pos, "mouseout",  hidePreview);
  addEventListener(pos, "click",     moveTo);

  // drag
  addEventListener(img,      "mousedown", grab);
  addEventListener(document, "mousemove", drag);
  addEventListener(document, "mouseup",   release);
}


/* Hide popup-wiondow */
function hideImg () 
{
  var popup = document.getElementById(GAL.POPUP_IDS[GAL.popupIndex]);
  popup.style.visibility = "hidden";
}


/* Grab for dragging (mouse down) */
function grab (evnt)
{
  if (!evnt) evnt = window.event;
  preventDefault(evnt);
  var popup = document.getElementById(GAL.POPUP_IDS[GAL.popupIndex]);
  GAL.dragX0 = pageX(evnt);
  GAL.dragY0 = pageY(evnt);
  GAL.dragXOffset = GAL.dragX0 - getLeft(popup);
  GAL.dragYOffset = GAL.dragY0 - getTop(popup);
}


/* Drag (mouse move) */
function drag (evnt)
{
  if (!evnt) evnt = window.event;
  if (GAL.dragXOffset > 0 && GAL.dragYOffset > 0) {
    preventDefault(evnt);
    var popup = document.getElementById(GAL.POPUP_IDS[GAL.popupIndex]);
    popup.style.left = Math.max(0, pageX(evnt) - GAL.dragXOffset);
    popup.style.top  = Math.max(0, pageY(evnt) - GAL.dragYOffset);
  }
}


/* Release (mouse up) */
function release (evnt)
{
   if (!evnt) evnt = window.event;
   if (GAL.dragXOffset > 0 && GAL.dragYOffset > 0) {
     preventDefault(evnt);
     var dx = Math.min(0, pageX(evnt) - GAL.dragXOffset);
     var dy = Math.min(0, pageY(evnt) - GAL.dragYOffset);
     GAL.popupXOffset += pageX(evnt) - GAL.dragX0 - dx;
     GAL.popupYOffset += pageY(evnt) - GAL.dragY0 - dy;
     GAL.dragXOffset = -1;
     GAL.dragYOffset = -1;
   }
}


/* 
 * Event handler which reacts on mouseover events on the position bar 
 * and shows the corresponding preview image (thumbnail image).
 */
function showPreview (evnt)
{
  if (!evnt) evnt = window.event;

  document.body.style.cursor='pointer';
  var thumbs = GAL.galeries[GAL.galeryID].thumbs;
  var pos    = document.getElementById(GAL.POSITION_IDS[GAL.popupIndex]);
  var absX   = getLeft(pos);
  var absY   = getTop(pos);
  var relX   = pageX(evnt) - absX;
  var index  = parseInt(relX / pos.offsetWidth * thumbs.length);
  var thumb  = thumbs[index]; 

  var preview = document.getElementById(GAL.PREVIEW_ID);
  var img     = document.getElementById(GAL.PREVIEW_IMG_ID);
  preview.style.left = parseInt(absX + (pos.offsetWidth / thumbs.length * (index + 0.5)) + GAL.PREVIEW_XOFFSET);
  preview.style.top  = Math.max(0, absY + GAL.PREVIEW_YOFFSET);
  var w, h;
  if (thumb.width > thumb.height) { 
    w = GAL.PREVIEW_MAXIMGSIZE; h = parseInt(thumb.height*w/thumb.width); 
  } else { 
    h = GAL.PREVIEW_MAXIMGSIZE; w = parseInt(thumb.width*h/thumb.height); 
  }
  img.src        = thumb.src;
  img.width      = w;
  img.height     = h;
  img.style.left = Math.max(0, parseInt(-GAL.PREVIEW_XOFFSET - w/2));
  preview.style.visibility = "visible";
}


function hidePreview (evnt)
{
  if (!evnt) evnt = window.event;

  document.body.style.cursor='default';

  var preview = document.getElementById(GAL.PREVIEW_ID);
  preview.style.visibility = "hidden";
}


/* 
 * Event handler which reacts on clicks on the position bar 
 * (move directly to the corresponding image).
 */
function moveTo (evnt)
{
  if (!evnt) evnt = window.event;
  var images = GAL.galeries[GAL.galeryID].images;
  var pos    = document.getElementById(GAL.POSITION_IDS[GAL.popupIndex]);
  var x      = pageX(evnt) - getLeft(pos);
  var index  = parseInt(x / pos.offsetWidth * images.length);
  showImg(index);
}


/* 
 * Init a galery
 * galeryID         id of a div element which contains all images of a galery
 * subLevel         sub directory level (0 in main directory, 1 for the upper example...)
 * verticalBorders  top and bottom borders (sum)
 * fixDetailSize    says whether the size of the detail area should be fixed 
 *                  before the image is downsized such that the detail fits
 *                  into the window (this could lead to additional horizontal
 *                  borders, but ensures that the popup  fits on the window 
 *                  even if the detail area is large
 * zoomFactor:      initial zoom factor (negative means full size) [optional]
 */
function initGalery (galeryID, subLevel, verticalBorders, fixDetailSize, zoomFactor) 
{
  // get base path
  GAL.BASE_PATH = "";
  for (i = 0; i < subLevel; i++) GAL.BASE_PATH += "../";

  // set zoom factor
  if (zoomFactor) GAL.zoom = zoomFactor;

  // init div-element used for the popup-window
  GAL.popupIndex = 0; hideImg();
  GAL.popupIndex = 1; hideImg();
  document.getElementById(GAL.POPUP_IDS[0]).innerHTML = "";
  document.getElementById(GAL.POPUP_IDS[1]).innerHTML = "";
  
  // change all links to images to a call to showImg()
  var images  = new Array();
  var details = new Array();
  var thumbs  = new Array();
  var galery  = document.getElementById(galeryID);
  var links   = getIdElementsByTagName(galery, "a"); // don't use 'getElementsByTagName()' (->bug in SeaMonkey)
  for (var i in links) {
    var link = links[i];
    var href = link.getAttribute("href");
    var imgs = link.getElementsByTagName("img");
    var divs = link.parentNode.getElementsByTagName("div");
    if (href &&
	(href.indexOf(".jpg") >= 0 || 
	 href.indexOf(".png") >= 0 || 
	 href.indexOf(".gif") >= 0 || 
	 href.indexOf(".tif") >= 0)) {

      // modify link (call showImg() instead)
      link.setAttribute("href", "javascript:showImg(" + images.length + ", null, '" + galeryID + "')"); 

      // store image reference, div-content and preview image (thumbnail)
      images[images.length]   = href;
      details[details.length] = (divs.length > 0 ? divs[0].innerHTML : null); 
      thumbs[thumbs.length]   = (imgs.length > 0 ? imgs[0] : null);
    }
  } 

  // create and store new entry 
  var entry = {};
  entry.images           = images;
  entry.details          = details;
  entry.thumbs           = thumbs;
  entry.verticalBorders  = verticalBorders;
  entry.fixDetailSize    = fixDetailSize;
  GAL.galeries[galeryID] = entry;
}
